/***************************************************************************
	File	: main.c
	Synop	: This is the source file for the MC68360 boot prom
	Modi	:	15/05/96 Vidy added the S/W watchdog timer reset
				19/07/96 Vidy added delay after response for next command to be ok
				Chetan added modifications for memory relocation
****************************************************************************/
#include	"..\include\mctypes.h"
#include	"..\include\boot.h"
#include	"..\include\downldr.h"
#include	"..\include\quote.h"
#include	"..\include\utils.h"

byte	ReceiveBuffer[RCV_BUF_SIZE];	/* quoted buffer */
dword SMCRcvBufMask ; /* Used so that asm files also look into this */
word	ReceiveIndex;
word	ReceiveHead;
word	ReceiveTail;
byte	ReceiveState;
byte	DownloadType;

/* all variables used in smc.68k */
byte	WaitForXON;
byte	NumOfXONs;
byte	NumOfXOFFs;
dword	NextBytePtr;
word	NumBytesLeft;
byte	ModemPresentAndOffline;		/* Modem responds OK in this case */
int	WriteToFlash (char *Buffer, char *FlashAddress, dword Length);

/* Essential hardware already inited */
extern	void	InitSMC(void);
extern	void	InitTimer(void);
extern	void	TransmitCnt(byte *, word);
extern	void	Reset(void);

extern void Execute (byte *loc) ;
extern int TestSCC1 (void) ;
extern int TestSCC2 (byte index) ;
extern int TestSCC3 (byte index) ;
extern int TestSCC4 (byte index) ;
extern void StartTimer (word count) ;
extern void StopTimer (void) ;
extern void Break (void) ;

void Delay (void)
{
	word i, j ;
	for (i = 0 ; i < 4 ; i ++)
		for (j = 0 ; j < 0xffff ; j ++) ;
}

#pragma separate	BootHeader segment BootSeg
BootConfigType	BootHeader = {
	1,					/* Major version */
	0,					/* minor version */
	0xFFFF,			/* first word of ethernet address */
	0xFFFFFFFF,		/* Second dword of ethernet address */
	"",

	4,					/* Number of Hardware Ports (1LAN + 3WAN) */
	CLOCK_258048MHZ,
	PROM_WIDTH_16BITS,
	MODEL_MTSR3_100,

	MAGIC_NUM,		/* This magic number is just to check sanity */
	0,
	0x80000l,			/* 512K  Flash Prom size */
	0x40000l,			/* SRAM size - 256K */
	0x80000l,			/* SRAM start at 512K */
} ;

#pragma separate	CfgHeader segment CfgSeg
HeaderType	CfgHeader;

#pragma separate	CodeHeader segment CodeSeg
HeaderType	CodeHeader;

#pragma separate	BootBinIOPB segment BootBinIOPBSeg
BootBinIOPBType	BootBinIOPB;

volatile int x;
int	CopyToSector(word *SectorAddress, word *Source, dword length);

main()
{
	int	count;
	x = 1;				/* this for debugging stops using "while(x);" */

	SMCRcvBufMask = (RCV_BUF_SIZE - 1);		/* for receive ISR */
	InitData() ;
	InitSMC() ;				/* initialize the BOOT port */
	InitTimer() ;			/* Initialise PIT */

#if KPRINT
	kprintinit();			/* for debug print in boot */
#endif


	/* Check if there is any modem */
	/* AT&FS0=1 ie. factory defaults, auto answer */
	/* Since SMC has no DTR line, the modem will not respond to any command.
		So either the modem has to be set for DTR always high or DTR is to be
		made high externally from a Breakout Box. So no factory defaults. */

	TransmitStr ("AT") ;
	Delay() ;

	if (SendModemStr ("S0=1&D0\r\n"))
	{	/* ignore DTR (no DTR on SMC) */
		/* modem initiated XON/XOFF and echo */
		SendModemStr ("AT&E5E0\r\n") ;
	}

/*	TransmitStr ("ProxyServer 3_100 Boot - Ver 1.00\r\n") ; */
	/* If both magic numbers are OK and Boot mode is not download mode
			if config checksum is OK
				if config code checksum is OK
					if user doesn't want do download anything
						execute code
		goto download mode */

	if ( (CfgHeader.MagicNum == MAGIC_NUM) &&
			(CodeHeader.MagicNum == MAGIC_NUM) &&
			(BootBinIOPB.BootMode != DOWNLOAD_MODE))
	{
		if (CheckCRC((byte *)CodeHeader.DnLdAddr,
							CodeHeader.CodeLength, (word *) &CodeHeader.CRC))
		{
			int (* CodeStartAddress)();

			if (!CheckCRC((byte *)CfgHeader.DnLdAddr,
								CfgHeader.CodeLength, (word *) &CfgHeader.CRC))
			{
				TransmitStr ("Setup Checksum error\r\n") ;
				goto WaitDownLoad ;
			}
			ReceiveHead = ReceiveTail = 0 ;
			*ReceiveBuffer = 0 ;
			TransmitStr ("Type 'd' to download Setup or Firmware.\r\n") ;
			for (count = 0 ; count < 7 ; count ++)
			{
				Delay() ;
				if ((*ReceiveBuffer == 'd') || (*ReceiveBuffer == 'D'))
					goto WaitDownLoad ;
			}
			TransmitStr ("Booting...\r\n") ;
			BootBinIOPB.BootMode = 0;	/* Reset this flag */

			CodeStartAddress = CodeHeader.StartAddr;
			CodeStartAddress();
		}
		else
		{
			TransmitStr ("Firmware Checksum error\r\n") ;
		}
	}
	else
	{
		TransmitStr ("Download mode or Wrong Magic number\r\n") ;
	}

WaitDownLoad:
	TransmitStr ("Waiting download.\n\r");
	BootBinIOPB.BootMode = 0;	/* Reset this flag */

	/* Invoke Command interpreter */
	while (1)
	{
		if (GetPacket())
			ProcessCommand() ;
	}
}

BOOL GetPacket (void)
{
	char ch ;

	char *SWSR = (char *) 0xFFFF102F ;		/* (ModeBase+RegBase+$2f) */

	*SWSR = 0x55 ;		/* reset software watchdog timer */
	*SWSR = 0xAA ;

	if (ReceiveTail != ReceiveHead)
	{
		ch = ReceiveBuffer[ReceiveHead] ;
		INC_CIR_IDX (ReceiveHead, RCV_BUF_SIZE) ;
		ReceiverEnabled = 1 ;
		switch (ReceiveState)
		{
			case RECV_7E :
				if (ch == FRAME_DELIM)
				{
					ReceiveState = RECV_DATA ;
					ReceiveIndex = 0 ;
				}
				break ;

			case RECV_DATA :
				if (ch == FRAME_DELIM)
				{
					if (ReceiveIndex >= MIN_PKT_SIZE)
					{
						ReceiveState = RECV_7E ;
						return TRUE ;
					}
					else
					{
						/* stay in the RECV_DATA state . But start a new packet */
						ReceiveIndex = 0 ;
						break ;
					}
				}
				else if (ch == 0x7d)
					ReceiveState = RECV_QUOTE ;

				else if (ReceiveIndex < RCV_BUF_SIZE)
					ProcessBuffer[ReceiveIndex++] = ch ; 

				else 
					ReceiveState = RECV_7E ;

				break ;

			case RECV_QUOTE :
				if (ReceiveIndex < RCV_BUF_SIZE)
				{
					ProcessBuffer[ReceiveIndex++] = (ch ^ 0x20) ;
					ReceiveState = RECV_DATA ; 
				}
				else 
					ReceiveState = RECV_7E ;
				break ;
		}			
	}
	return FALSE ;
}

void ProcessCommand (void)
{
 	TargetCmdType *TrgtCmd ;
	ResultType *ResultPkt ;
	byte *PktPtr ;
	word PktLen ;
	byte ResetSystem = FALSE ;

	DownloadType = NO_DNLD ;
	memcpy(TmpBuffer, "RPLY", 4);
	ResultPkt = (ResultType *) (TmpBuffer + sizeof (SMCPktHeadType)) ;
	ResultPkt->Ptype = RES_TYPE ;
	ResultPkt->Errcode = 0 ;

	/* Process the Command */
	if (!CheckCRC(ProcessBuffer, (dword) ReceiveIndex, 0))
	{
kprint("CRC Err\n\r");
		ReceiveState = RECV_7E ;
		ResultPkt->Errcode = ERR_CRC ;
		goto sendpkt ;
	} 
	
	/* Actual Pkt starts at an offset of BridgeId(4) */
	PktPtr = ProcessBuffer + sizeof (SMCPktHeadType) ;
	switch (PktPtr[0])
	{
		case PASSWORD_TYPE :
			if (!CheckPassword (((PasswordPktType *) PktPtr)->Password))
				/* send NOK pkt */
				ResultPkt->Errcode = ERR_PASSWORD ;
			break ;

		case TRGT_CMND_TYPE :
			TrgtCmd = (TargetCmdType *) PktPtr ;
			switch (TrgtCmd->TargetAction)
			{
				case DNLOAD_CODE_TYPE :
					/* send back a result type pkt and wait for download pkts */ 
					DownloadType = DNLD_CODE ;
					break ;

				case HW_DIAG_TYPE :
					switch (TrgtCmd->Port)
					{
						case SCC1 :
							if (!TestSCC1())
								ResultPkt->Errcode = ERR_SCC1 ;
							break ;

						case SCC2 :
							if (!TestSCC2 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC2 ;
							break ;

						case SCC3 :
							if (!TestSCC3 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC3 ;
							break ;
							
						case SCC4 :
							if (!TestSCC4 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC4 ;
							break ;
					}
					break ;

				case DNLOAD_CFG_TYPE:
					DownloadType = DNLD_CFG ;
/* kprint("Download conf\r\n"); */
					break ;
			}
			break ;

		case READ_REQ_TYPE :
		{
			ReadLocReqType *RRqT ;
			ReadLocRespType *RRspT ;

			RRqT = (ReadLocReqType *) PktPtr ;
			if (RRqT->Length > 128)
				RRqT->Length = 128 ;

			/* send back the response pkt */
			RRspT = (ReadLocRespType *) (TmpBuffer + sizeof (SMCPktHeadType)) ;
			RRspT->Ptype = READ_RESP_TYPE ;
			if ((RRqT->Address + RRqT->Length) < (512 * 1024))
			{
				RRspT->ErrCode = 0 ;
				RRspT->Address = RRqT->Address ;
				RRspT->Length = RRqT->Length ;
				memcpy ((byte *) RRspT->buffer,
							(byte *) RRqT->Address, RRqT->Length) ;
			}
			else
				RRspT->ErrCode = ERR_INV_MEM_RANGE ;
			PktLen = sizeof (ReadLocRespType) + sizeof (SMCPktHeadType) ;		
			goto sendresp ;
		}

		case  WRITE_REQ_TYPE :
		{
			WriteLocReqType *WRqT ;

			WRqT = (WriteLocReqType *) PktPtr ;
			if (WRqT->Address < BOOT_CODE_SIZE)
			{
				kprint("Error: WriteMem to Boot area!\r\n");
				break;
			}
			if ( ((dword)WRqT->Address + WRqT->Length) > FL_SIZE)
			{
				kprint("Error: WriteMem overflow!\r\n");
				break;
			}
			/* we may check if the write address is in boot segment */
			WriteToFlash ((char*)WRqT->Buffer,
										(char*)WRqT->Address, WRqT->Length);
			break ;
		}
			
		case REBOOT_TYPE :
			/* move reset value into the reset location and the system 
			will reboot */
			BootBinIOPB.BootMode = PktPtr[1];	/* assert boot type before reset */
			ResetSystem = TRUE ;
			break ;

		default :
			/* send back a reply with ErrCode = BridgeNotup and request
						password if present */
			if (BootHeader.PassWord[0]) 
				ResultPkt->Errcode = ERR_NO_BRD_GET_PASSWORD ;
			else
				ResultPkt->Errcode = ERR_NO_BRD_NO_PASSWORD ;
			break ;
	}

sendpkt :
	PktLen =	sizeof (ResultType) + sizeof (SMCPktHeadType) ;
sendresp :
	AddCRC(TmpBuffer, (dword) PktLen) ;
	/* do not send PktLen but (Pktlen + 2) b'cos
	2 extra zeroes are appended while adding CRC*/
	PktLen = Quote ((byte *) TmpBuffer,
								(byte *) ResponseBuffer + 1, PktLen + 2) ;
	ResponseBuffer[0] = ResponseBuffer[PktLen + 1] = FRAME_DELIM ;
	PktLen += 2 ;
	TransmitCnt (ResponseBuffer, PktLen) ;
	if (DownloadType)
		DownLoadNewInfo() ;
	else
		if (ResetSystem)
			Reset() ;
}

BOOL CheckPassword (char *PW)
{
	if (!strcmp (BootHeader.PassWord, PW))
		return TRUE ;
	else
		return FALSE ;
}

void ClearRxBuffer (void)
{
	ReceiveTail = ReceiveHead = 0 ;
}

word SearchRxBuffer (char *Str)
{
	word i, j, Len ;

	if (!ReceiveTail)
		return FALSE ;
	Len = strlen (Str) ;
	if (ReceiveTail < Len)
		return FALSE ;
	for (i = ReceiveHead ; i <= (ReceiveTail - Len) ; i ++)
	{
		for (j = 0 ; j < Len ; j ++)
		{
			if (ReceiveBuffer[i+j] != Str[j])
				break ;
		}
		if (j == Len)
			return TRUE ;
	}
	return FALSE ;
}

word SendModemStr (byte *Str)
{
	ClearRxBuffer() ;
	TransmitCnt (Str, strlen (Str)) ;		/* TransmitStr may block */
	Delay() ;
	StartTimer (MDM_RESP_TIMEOUT) ;
	while (!TimeoutOccured)
	{
		if (SearchRxBuffer ("OK"))
		{
			Delay() ;
			Delay() ;				/* Vidy 19/07/96 - for next command */
			ModemPresentAndOffline = 1 ;	/* Modem responded OK in this case */
			StopTimer() ;
			return TRUE ;
		}
	}
	StopTimer() ;
	return FALSE ;
}

int	WriteToFlash (char *Buffer, char *FlashAddress, dword Length)
{
	dword SegmentBaseAddress;
	dword SegmentSize, OffsetInSegment;

	SegmentBaseAddress = GetSectorSizeOrBaseAddress((dword)FlashAddress,
					GET_BASE_ADDR);
	SegmentSize = GetSectorSizeOrBaseAddress((dword)FlashAddress, GET_SIZE);

	if( ((dword) SegmentBaseAddress > (512 * 1024L)) || (SegmentSize > (64 * 1024L)) )
		 	return 0;

	memcpy(FlashPromBuf, (void *)SegmentBaseAddress, SegmentSize);

	OffsetInSegment = FlashAddress - SegmentBaseAddress;
	memcpy(FlashPromBuf+OffsetInSegment, Buffer, Length);

	CopyToSector((word *)SegmentBaseAddress, (word *)FlashPromBuf, SegmentSize);
	return 1;
}

