/* AGINBOUN.C -- Handling of inbound calls and the inbound state machine.
**				 Some of the state functions are in AGINMENU.C and support
**				 functions are in AGINBSUP.C.
** By: Sanjay S
** Start: 12, August, 1996
** Done: 30, August, 1996
*/

/* General Notes:
**	1. A few states do not make sense here so they do nothing and skip to
**		the next state. These are check_for_ring, send_modem_string and
**		wait_for_string_to_be_sent.
**	2. check_for_cd is actually informed by a low-level device driver.
**	3. The functions start_aterm_timer, look_for_aterm, send_escape_sequence,
**		wait_for_sequence_to_be_sent, wait_for_ack_1, wait_for_ack_2, 
**		wait_for_act_3, reset_aterm, and wait_reset_aterm currently do 
**		nothing and simply skip over to the next state. It "seems" these 
**		states were required to take care of problems when working with old 
**		versions of PC-Anywhere and CarbonCopy. Since RN-AG has removed 
**		these, we also skip these.
**	4. During the course of the inbound handling (which can take around 2
**	  	minutes), it is possible that the session on which the inbound
**		occurs is aborted. So the session aborting is delayed until the
**		inbound state machine is complete.
*/

#include "rtrstd.h"

#include "kag.h"
#include "vagstr.h"

#include "agtx.h"
#include "aginbsup.h"
#include "aginmenu.h"
#include "aginboun.h"
#include "agpkttyp.h"
#include "agutil.h"
#include "agnetif.h"
#include "agline.h"
#include "agintern.h"

#include <msm.h>
#include <serial.h>

#include <udb.h>
#include <wanmgr.h>

/* External Prototypes */
extern BYTE old_modem_control_byte[MAX_WAN_PORTS] ;

/* Local Prototypes */
STATIC enum BOOLEAN check_for_ring(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN send_modem_string(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_for_string_to_be_sent(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN check_for_cd(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN start_aterm_timer(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN look_for_aterm(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN send_escape_sequence(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_for_sequence_to_be_sent(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_for_ack_1(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_for_ack_2(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_for_ack_3(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN display_remote_menu(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN prompt_user_name(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN get_user_name(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN display_bad_name(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN prompt_password(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN get_password(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN display_bad_password(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN reset_aterm(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_aterm_reset(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN make_connection(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN display_login_aborted(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN end_state(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN check_callback(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN callback_user(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_call_cd(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN call_hangup(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN get_phone_number(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum BOOLEAN wait_login_aborted(LINE_INFO_ENTRY *sptr_line_info);

STATIC void callback_modem_hangup_complete(USHORT port_number);
STATIC enum BOOLEAN connect_inbound(LINE_INFO_ENTRY *sptr_line_info);
STATIC void inbound_modem_hangup_complete(USHORT port_number);

/* Statics */
STATIC enum BOOLEAN (*inbound_state_functions[])(LINE_INFO_ENTRY *sptr_line_info) =
{
	check_for_ring,
	send_modem_string,
	wait_for_string_to_be_sent,
	check_for_cd,
	start_aterm_timer,
	look_for_aterm,
	send_escape_sequence,
	wait_for_sequence_to_be_sent,
	wait_for_ack_1,
	wait_for_ack_2,
	wait_for_ack_3,
	display_remote_menu,
	prompt_user_name,
	get_user_name,
	display_bad_name,
	prompt_password,
	get_password,
	display_bad_password,
	reset_aterm,
	wait_aterm_reset,
	make_connection,
	display_login_aborted,
	end_state,
	check_callback,
	callback_user,
	wait_call_cd,
	handle_host_list,
	display_host_list,
	validate_host,
	call_hangup,
	get_phone_number,
	wait_login_aborted		/* New to this version */
};

STATIC char *prompt_name_message = "\n\rEnter your user name : ";
STATIC char *bad_name_message = "\n\rInvalid user name";
STATIC char *prompt_password_message = "\n\rEnter your password : ";
STATIC char *bad_password_message = "\n\rInvalid password";
STATIC char *login_aborted_message = "\n\rLogin aborted";
STATIC char *callback_message = "\n\rPlease wait, the server will call you back \n\r";
STATIC char *enter_number_message = "\b\rEnter you phone number for the server to call you back \n\r";

STATIC char *modem_disable_answering_string = "ATS0=0\r";

char remote_menu[23][83] = 
{
"\n\r\n\r",
"        Welcome to Multi-Tech Asynchronous Communication Gateway (IPX)",
"\n\r\n\r\n\r",
" ####    ####   #    #  #    #  ######   ####    #####  ###    ####   #    #\n\r",
"#    #  #    #  ##   #  ##   #  #       #    #     #     #    #    #  ##   #\n\r",
"#       #    #  # #  #  # #  #  #####   #          #     #    #    #  # #  #\n\r",
"#       #    #  #  # #  #  # #  #       #          #     #    #    #  #  # #\n\r",
"#    #  #    #  #   ##  #   ##  #       #    #     #     #    #    #  #   ##\n\r",
" ####    ####   #    #  #    #  ######   ####      #    ###    ####   #    #\n\r",
"\n\r\n\r\n\r",
"          #    #    ##    #    #    ##     ####   ######  #####\n\r",
"          ##  ##   #  #   ##   #   #  #   #    #  #       #    #\n\r",
"          # ## #  #    #  # #  #  #    #  #       #####   #    #\n\r",
"          #    #  ######  #  # #  ######  #  ###  #       #####\n\r",
"          #    #  #    #  #   ##  #    #  #    #  #       #   #\n\r",
"          #    #  #    #  #    #  #    #   ####   ######  #    #\n\r",
"\n\r\n\r",
"\0"    
};

/* Some globals that are not really used in this version of the AG */

char set_ascii_sequence[] = { 0x1B, 0x56, 0xFF, 0x07, 0xF8, 00 };
char set_aterm_sequence[] = { 0x1B, 0x56, 0xFF, 0x08, 0xF7, 00 };
char ascii_mode_ack[] = { 0x66, 0x07, 0xF8 };
char aterm_mode_ack[] = { 0x06, 0x08, 0xF7 };


/* -- CODE ------------------------------------------------------------- */

void move_inbound_state_machine()
{
	LINE_INFO_ENTRY *sptr_line_info;


	sptr_line_info = (LINE_INFO_ENTRY *)get_pointer_to_first_entry_in_list((LINK *)&ag.lines_awaiting_inbound);
	while (sptr_line_info != NULL)
	{
		if (sptr_line_info->inbound_state_timer)
			sptr_line_info->inbound_state_timer--;

		while (inbound_state_functions[sptr_line_info->next_inbound_state](sptr_line_info));

		sptr_line_info = (LINE_INFO_ENTRY *)get_pointer_to_next_entry_in_list((LINK *)sptr_line_info);
	}
}

STATIC enum BOOLEAN check_for_ring(LINE_INFO_ENTRY *sptr_line_info)
{
	/* This function is for compatibility with the DOS version. We actually
	** go directly to the checking for CD state.
	*/
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state CHECK_FOR_RING\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_CHECK_CD;
	return TRUE;
}

STATIC enum BOOLEAN send_modem_string(LINE_INFO_ENTRY *sptr_line_info)
{
	/* This function is for compatibility with the DOS version. We actually
	** go directly to the checking for CD state.
	*/

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state SEND MODEM STRING\n\r");
#endif 

	sptr_line_info->next_inbound_state = IS_CHECK_CD;
	return TRUE;
}

STATIC enum BOOLEAN wait_for_string_to_be_sent(LINE_INFO_ENTRY *sptr_line_info)
{
	/* This function is for compatibility with the DOS version. We actually
	** go directly to the checking for CD state.
	*/

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT FOR STRING TO BE SENT\n\r");
#endif 

	sptr_line_info->next_inbound_state = IS_CHECK_CD;
	return TRUE;
}

STATIC enum BOOLEAN check_for_cd(LINE_INFO_ENTRY *sptr_line_info)
{
	/* The inbound state machine will normally (when no inbound connection
	** has been established) be stuck at this point. The original WAN
	** device driver will keep looking at incoming packets and when it 
	** determines that the incoming packet is not a PPP packet, will
	** check if there is anyone waiting on inbound requests on this port.
	** If so, the port is alloted to the inbound module and the inbound
	** state machine will move from then on to different states until
	** either a satisfactory connection is established or some error occurs.
	*/
	if (sptr_line_info->inbound_init_on == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state CHECK FOR CD\n\r");
#endif 

	set_line_status(sptr_line_info, AGLS_CALL_COMING_IN);

	if (ag.show_connection_manager_menu != FALSE)
		sptr_line_info->next_inbound_state = IS_DISPLAY_MENU;
	else
		sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
		
	return TRUE;
}

STATIC enum BOOLEAN start_aterm_timer(LINE_INFO_ENTRY *sptr_line_info)
{
	if (sptr_line_info->inbound_state_timer != 0)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state START ATERM TIMER\n\r");
#endif 
	if (ag.show_connection_manager_menu != FALSE)
		sptr_line_info->next_inbound_state = IS_DISPLAY_MENU;
	else
		sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
		
	return TRUE;
}

STATIC enum BOOLEAN look_for_aterm(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state LOOK FOR ATERM\n\r");
#endif 

	if (ag.show_connection_manager_menu != FALSE)
		sptr_line_info->next_inbound_state = IS_DISPLAY_MENU;
	else
		sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
		
	return TRUE;
}

STATIC enum BOOLEAN send_escape_sequence(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state SEND ESCAPE SEQUENCE\n\r");
#endif 
	write_serial_port((USHORT)sptr_line_info->line_id, set_ascii_sequence, sizeof(set_ascii_sequence));

	sptr_line_info->next_inbound_state = IS_WAIT_ASCII;
	return TRUE;
}

STATIC enum BOOLEAN wait_for_sequence_to_be_sent(LINE_INFO_ENTRY *sptr_line_info)
{
	if (is_serial_tx_buffer_empty((USHORT)sptr_line_info->line_id) == FALSE)
	{
		return FALSE;
	}

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT FOR SEQUENCE TO BE SENT\n\r");
#endif 

	sptr_line_info->next_inbound_state = IS_WAIT_ACK1;
	return TRUE;
}

STATIC enum BOOLEAN wait_for_ack_1(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT FOR ACK1\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_WAIT_ACK2;
	return TRUE;
}

STATIC enum BOOLEAN wait_for_ack_2(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT FOR ACK2\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_WAIT_ACK3;
	return TRUE;
}

STATIC enum BOOLEAN wait_for_ack_3(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT FOR ACK3\n\r");
#endif 
	if (ag.show_connection_manager_menu != FALSE)
		sptr_line_info->next_inbound_state = IS_DISPLAY_MENU;
	else
		sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
		
	return TRUE;
}

STATIC enum BOOLEAN display_remote_menu(LINE_INFO_ENTRY *sptr_line_info)
{
	char *sptr_string;


	if (is_serial_tx_buffer_empty((USHORT)sptr_line_info->line_id) == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state DISPLAY REMOTE MENU\n\r");
#endif 
	sptr_string = &remote_menu[sptr_line_info->menu_line_number][0];
	if (*sptr_string != '\0')
	{
		if (write_serial_port((USHORT)sptr_line_info->line_id, sptr_string, strlen(sptr_string)) != FAIL)
			sptr_line_info->menu_line_number += 1;
		return FALSE;
	}

	sptr_line_info->menu_line_number = 0;
	sptr_line_info->next_inbound_state = IS_PROMPT_NAME;
	return TRUE;
}

STATIC enum BOOLEAN prompt_user_name(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	if (is_serial_tx_buffer_empty(port_number) == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state PROMPT USER NAME\n\r");
#endif 
	flush_serial_rx_buffers(port_number);
	reset_inbound_input_state(sptr_line_info, TRUE);
	write_serial_port(port_number, prompt_name_message, strlen(prompt_name_message));

	sptr_line_info->next_inbound_state = IS_GET_NAME;
	sptr_line_info->inbound_state_timer = GET_NAME_DELAY;
	return TRUE;
}

STATIC enum BOOLEAN get_user_name(LINE_INFO_ENTRY *sptr_line_info)
{
	BYTE *user_entry;
	SESSION_TABLE_ENTRY *sptr_inbound_session_entry;
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	if (is_serial_tx_buffer_empty(port_number) == FALSE)
		return FALSE;

	if (sptr_line_info->inbound_state_timer == 0)
	{
		sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		return TRUE;
	}

	/* Wait for a user name to be filled in */
	if (sptr_line_info->inbound_input_valid == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state GET USER NAME %s\n\r", sptr_line_info->inbound_user_name);
#endif 
	user_entry = is_valid_ras_user(sptr_line_info->inbound_user_name);
	sptr_line_info->sptr_session_entry = NULL;
	sptr_inbound_session_entry = find_matching_user_entry_in_line_inbound_list(sptr_line_info, user_entry);
	if (sptr_inbound_session_entry != NULL && 
			sptr_inbound_session_entry->session_status == AG_SESS_ACTIVE)
	{
		sptr_inbound_session_entry->session_status = AG_SESS_CONNECTED;
		sptr_line_info->sptr_session_entry = sptr_inbound_session_entry;
		if (has_ras_user_password(user_entry))
			sptr_line_info->next_inbound_state = IS_PROMPT_PASSWORD;
		else
			sptr_line_info->next_inbound_state = IS_CHECK_CALLBACK;
	}
	else
	{
		sptr_line_info->inbound_login_attempts++;
		if (sptr_line_info->inbound_login_attempts == MAX_INBOUND_LOGIN_ATTEMPTS)
		{
			sptr_line_info->inbound_login_attempts = 0;
			sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		}
		else
		{
			sptr_line_info->next_inbound_state = IS_DISPLAY_NAME_ERROR;
			sptr_line_info->inbound_state_timer = GET_NAME_DELAY;
		}
	}
	return TRUE;
}

STATIC enum BOOLEAN display_bad_name(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state DISPLAY BAD NAME\n\r");
#endif 
	write_serial_port((USHORT)sptr_line_info->line_id, bad_name_message, strlen(bad_name_message));

	if (ag.show_connection_manager_menu == FALSE)
		sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
	else
		sptr_line_info->next_inbound_state = IS_PROMPT_NAME;
	return TRUE;
}

STATIC enum BOOLEAN prompt_password(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	if (is_serial_tx_buffer_empty(port_number) == FALSE)
		return FALSE;
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state PROMPT PASSWORD\n\r");
#endif 

	flush_serial_rx_buffers(port_number);
	reset_inbound_input_state(sptr_line_info, FALSE);
	write_serial_port(port_number, prompt_password_message, strlen(prompt_password_message));

	sptr_line_info->next_inbound_state = IS_GET_PASSWORD;
	sptr_line_info->inbound_state_timer = GET_PASSWORD_DELAY;
	return TRUE;
}

STATIC enum BOOLEAN get_password(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	if (is_serial_tx_buffer_empty(port_number) == FALSE)
		return FALSE;

	if (sptr_line_info->inbound_state_timer == 0)
	{
		sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		return TRUE;
	}

	/* Wait for a password to be filled in */
	if (sptr_line_info->inbound_input_valid == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state GET PASSWORD\n\r");
#endif 
	if (is_valid_ras_user_password(sptr_line_info->sptr_session_entry->user_entry, sptr_line_info->inbound_user_password))
		sptr_line_info->next_inbound_state = IS_CHECK_CALLBACK;
	else
	{
		sptr_line_info->inbound_login_attempts++;
		if (sptr_line_info->inbound_login_attempts == MAX_INBOUND_LOGIN_ATTEMPTS)
		{
			sptr_line_info->inbound_login_attempts = 0;
			sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		}
		else
		{
			sptr_line_info->next_inbound_state = IS_DISPLAY_PASSWORD_ERROR;
			sptr_line_info->inbound_state_timer = GET_PASSWORD_DELAY;
		}
	}
	return TRUE;
}

STATIC enum BOOLEAN display_bad_password(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state DISPLAY BAD PASSWORD\n\r");
#endif 
	write_serial_port((USHORT)sptr_line_info->line_id, bad_password_message, strlen(bad_password_message));

	sptr_line_info->next_inbound_state = IS_PROMPT_NAME;
	return TRUE;
}

STATIC enum BOOLEAN reset_aterm(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state RESET ATERM\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_CONNECT_INBOUND;
	return TRUE;
}

STATIC enum BOOLEAN wait_aterm_reset(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT ATERM RESET\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_CONNECT_INBOUND;
	return TRUE;
}

STATIC enum BOOLEAN make_connection(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state MAKE CONNECTION\n\r");
#endif 
	connect_inbound(sptr_line_info);
	sptr_line_info->next_inbound_state = IS_STOP_STATE;
	return TRUE;
}

STATIC enum BOOLEAN display_login_aborted(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state DISPLAY LOGIN ABORTED\n\r");
#endif 
	if (write_serial_port((USHORT)sptr_line_info->line_id, login_aborted_message, strlen(login_aborted_message)) == FAIL)
		return FALSE;

	sptr_line_info->next_inbound_state = IS_WAIT_ABORTED_LOGIN;
	sptr_line_info->inbound_state_timer = DELAY_FOR_SERIAL_TRANSFER;
	return TRUE;
}																						 

STATIC enum BOOLEAN wait_login_aborted(LINE_INFO_ENTRY *sptr_line_info)
{
/*
	if (is_serial_tx_buffer_empty((USHORT)sptr_line_info->line_id) == FALSE)
		return FALSE;
*/

	if (sptr_line_info->inbound_state_timer != 0)
	{
		return FALSE;
	}
	abort_inbound_call(sptr_line_info);
	return TRUE;
}

STATIC enum BOOLEAN end_state(LINE_INFO_ENTRY *sptr_line_info)
{
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state END STATE\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_CHECK_RING;
	return TRUE;
}

STATIC enum BOOLEAN check_callback(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number = (USHORT)sptr_line_info->line_id;


#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state CHECK CALLBACK\n\r");
#endif 

	if (sptr_line_info->callback_check_done == FALSE &&
		is_wan_direct_connect((USHORT)sptr_line_info->line_id) == FALSE &&
		is_ras_callback_enabled(sptr_line_info->sptr_session_entry->user_entry, port_number) == TRUE)
	{
		/* Need to do callback */
		if (is_ras_callback_callback_security(sptr_line_info->sptr_session_entry->user_entry, port_number))
		{
			/* Callback security feature is enabled -- the server will dial-up
			** a pre-configured number assuming the remote is at that end.
			*/
			write_serial_port(port_number, callback_message, strlen(callback_message));
			ras_get_callback_number(sptr_line_info->sptr_session_entry->user_entry, (USHORT)sptr_line_info->line_id, sptr_line_info->inbound_user_callback_number);
			sptr_line_info->next_inbound_state = IS_CALL_HANGUP;
		}
		else
		{
			/* Else, mobile computing feature is enabled -- remote provides
			** a number that we call up.
			*/
			write_serial_port(port_number, enter_number_message, strlen(enter_number_message));
			flush_serial_rx_buffers(port_number);
			reset_inbound_input_state(sptr_line_info, TRUE);
			sptr_line_info->next_inbound_state = IS_GET_PHONE_NUMBER;
			sptr_line_info->inbound_state_timer = GET_PHONE_NUMBER_DELAY;
		}
	}
	else
	{
		sptr_line_info->callback_check_done = FALSE;
		sptr_line_info->next_inbound_state = IS_RESET_ATERM;
	}

	return TRUE;
}

STATIC enum BOOLEAN callback_user(LINE_INFO_ENTRY *sptr_line_info)
{
	char converted_buffer[21 + 60 + 21];
	char unconverted_buffer[21 + 60 + 21];
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	/* Wait for the hangup to go thro' */
	if (sptr_line_info->callback_hangup_done == FALSE)
		return FALSE;

	/* Wait for some time after hangup is done */
	if (sptr_line_info->inbound_state_timer != 0)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state CALLBACK USER\n\r");
#endif 
	/* Dial the callback number */
	strcpy(unconverted_buffer, get_wan_modem_dial_prefix(port_number));
	strcat(unconverted_buffer, sptr_line_info->inbound_user_callback_number);
	strcat(unconverted_buffer, get_wan_modem_dial_suffix(port_number));

	memset(converted_buffer, 0, sizeof(converted_buffer));
	convert_controls(converted_buffer, unconverted_buffer);
	write_serial_port(port_number, converted_buffer, strlen(converted_buffer));

	sptr_line_info->next_inbound_state = IS_WAIT_CALL_CD;
	sptr_line_info->inbound_state_timer = ag.clock_ticks_per_second * 60;
	return TRUE;
}

STATIC enum BOOLEAN wait_call_cd(LINE_INFO_ENTRY *sptr_line_info)
{
	if (sptr_line_info->inbound_state_timer == 0)
	{
		abort_inbound_call(sptr_line_info); 
		return FALSE;
	}

	if (is_dcd_present((USHORT)sptr_line_info->line_id) == FALSE)
		return FALSE;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state WAIT CALL CD\n\r");
#endif 
	/* Made it! So, continue with the connection. */

								/* To avoid repeating callback */
	sptr_line_info->callback_check_done = TRUE; 
	sptr_line_info->next_inbound_state = IS_ATERM_TIMER;
								/* Give 2 seconds for the "CONNECT" string */
	sptr_line_info->inbound_state_timer = 2 * ag.clock_ticks_per_second;	
	return TRUE;
}

STATIC enum BOOLEAN call_hangup(LINE_INFO_ENTRY *sptr_line_info)
{
	char *modem_hangup_string;

#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state CALL HANGUP\n\r");
#endif 
	/* Hangup the connection and then call user back either with provided
	** or known number.
	*/

	set_line_status(sptr_line_info, AGLS_CALLING_BACK);
	sptr_line_info->next_inbound_state = IS_CALLBACK;
	sptr_line_info->inbound_state_timer = (ULONG)ras_get_callback_delay(sptr_line_info->sptr_session_entry->user_entry, (USHORT)sptr_line_info->line_id);

	if (sptr_line_info->sptr_session_entry != NULL)
		if (sptr_line_info->sptr_session_entry->session_status == AG_SESS_CONNECTED)
			sptr_line_info->sptr_session_entry->session_status = AG_SESS_ACTIVE;

	sptr_line_info->inbound_login_attempts = 0;

	/* Need to call the modem state machine to hangup */
	sptr_line_info->callback_hangup_done = FALSE;
	modem_hangup_string = get_wan_modem_hangup_string((USHORT)sptr_line_info->line_id);
	start_modem_hangup_state_machine((USHORT)sptr_line_info->line_id, modem_hangup_string, (void (*)(USHORT, BYTE *, USHORT))write_serial_port, callback_modem_hangup_complete);

	return TRUE;
}

STATIC enum BOOLEAN get_phone_number(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	if (is_serial_tx_buffer_empty(port_number) == FALSE)
		return FALSE;

	if (sptr_line_info->inbound_state_timer == 0)
	{
		sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		return TRUE;
	}

	/* Wait for the number to be entered */
	if (sptr_line_info->inbound_input_valid == FALSE)
		return FALSE;
		
#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: In state GET PHONE NUMBER\n\r");
#endif 
	sptr_line_info->next_inbound_state = IS_CALL_HANGUP;
	return TRUE;
}

/* Some support code */
STATIC void callback_modem_hangup_complete(USHORT port_number)
{
	LINE_INFO_ENTRY *sptr_line_info;


	sptr_line_info = &ag.line_info_array[port_number];
	sptr_line_info->callback_hangup_done = TRUE;

	/* To prevent callers from calling in before we are done with this
	** callback mechanism, put the modem in "no answering" mode.
	*/
	write_serial_port(port_number, modem_disable_answering_string, strlen(modem_disable_answering_string));

	/* Arrange to wait for an OK after this string is sent */
	sptr_line_info->inbound_state_timer = NO_ANSWERING_DELAY;

	return;
}

STATIC enum BOOLEAN connect_inbound(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT data_size;
	UNION_AG_PACKET *sptr_ag_packet;
	SESSION_TABLE_ENTRY *sptr_session_entry;
	CONNECT_RESP_TYPE *sptr_connect_response;
	USHORT requested_channel = 0;


	ag_printf(AG_INBOUND_PRINTF, "AG: Setting up an inbound connection\n\r");

	/* Basically inform the client that someone has connected inbound
	** successfully. i.e send him a connect response packet.
	*/
	sptr_ag_packet = (UNION_AG_PACKET *)get_a_network_send_packet(sizeof(CONNECT_RESP_TYPE) + sizeof(BYTE));
	if (sptr_ag_packet == NULL)
	{
		ag_printf(AG_ALARM_PRINTF, "AG: Insufficient memory to complete inbound connection on WAN %u\n\r", (USHORT)sptr_line_info->line_id);

		return FALSE;
	}

	sptr_ag_packet->packet_class = AG_INBOUND_CONNECT_RESP;
	sptr_connect_response = &sptr_ag_packet->packet_type.connect_response;

	/* Deleteing from inbound */
	delete_line_from_inbound_list(sptr_line_info);
	reset_inbound_state_machine_globals(sptr_line_info);

	/* Statistics */
	ag.stats.number_of_connections++;
	ag.stats.inbound_connections++;
	ag.stats.total_dialin_calls++;
	sptr_line_info->stats.total_dialin_calls++;
	/* Record connect time */
	get_current_date_time(&sptr_line_info->stats.connect_time);


	/* Prepare the response packet */
	memcpy(sptr_connect_response->general_name, sptr_line_info->line_vars.general_name, GENERAL_NAME_LENGTH);
	memcpy(sptr_connect_response->specific_name, sptr_line_info->line_vars.specific_name, SPECIFIC_NAME_LENGTH);
	sptr_connect_response->channel_number = sptr_line_info->line_id;
	sptr_connect_response->connect_status = AGCR_CONNECTED;
	sptr_connect_response->EIA_status = get_EIA_status((USHORT)sptr_line_info->line_id);
	set_connect_vars_info(&sptr_connect_response->connect_vars_info, &sptr_line_info->line_vars);

	requested_channel = sptr_line_info->line_id;


	/* Update some line information */
	sptr_line_info->connect_mode = INBOUND_MODEM;
	set_line_status(sptr_line_info, AGLS_LINE_BUSY);
	sptr_line_info->inactivity_timer = sptr_line_info->line_vars.idle_time;
	modify_serial_rx_buffer_size((USHORT)sptr_line_info->line_id, sptr_line_info->line_vars.current_max_packet_size);

	/* Update session related stuff . Notice that the session status
	** is not (and should not be) set here.
	*/
	sptr_session_entry = sptr_line_info->sptr_session_entry;
	associate_user_parameter_with_serial_driver((USHORT)sptr_line_info->line_id, (BYTE *)sptr_session_entry);
	sptr_session_entry->line_in_use = sptr_line_info->line_id;


	change_endian(sptr_ag_packet);
	data_size = sizeof(CONNECT_RESP_TYPE) + sizeof(BYTE);
#ifdef DEBUG
	ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending INBOUND CONNECT RESPONSE packet \n\r");
#endif /* DEBUG */
	if (send_network_internal_packet(sptr_session_entry, sptr_ag_packet, data_size) == FAIL)
	{
		ag_printf(AG_ALARM_PRINTF, "AG: Insufficient memory to inform client of inbound connection on WAN %u\n\r", (USHORT)sptr_line_info->line_id);
		free_a_network_send_packet((BYTE *)sptr_ag_packet);

		/* We cannot possibly proceed. Easiest way out is to terminate 
		** connection 
		*/
		send_disconnect_packet(sptr_session_entry, sptr_line_info, AGAR_ABNORMAL_TERMINATION);
	}

	/* From now all serial rx's should go to the session partner */
	sptr_line_info->inbound_init_on = FALSE;	

	old_modem_control_byte[requested_channel] = 0;
	return TRUE;
}

void abort_inbound_call(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number;
	char *modem_hangup_string;


#ifdef DEBUG
	ag_printf(AG_INBOUND_PRINTF, "AG: Aborting inbound call\n\r");
#endif 

	reset_inbound_state_machine_globals(sptr_line_info);
	if (sptr_line_info->sptr_session_entry != NULL)
	{
		if (sptr_line_info->sptr_session_entry->session_status == AG_SESS_CONNECTED)
			sptr_line_info->sptr_session_entry->session_status = AG_SESS_ACTIVE;
		sptr_line_info->sptr_session_entry = NULL;
	}

	set_line_status(sptr_line_info, AGLS_CLEANING_UP);

	port_number = (USHORT)sptr_line_info->line_id;
	if (is_wan_direct_connect(port_number) == FALSE)
	{
		modem_hangup_string = get_wan_modem_hangup_string(port_number);
		start_modem_hangup_state_machine(port_number, modem_hangup_string, (void (*)(USHORT, BYTE *, USHORT))write_serial_port, inbound_modem_hangup_complete);
	}
	else
	{
		inbound_modem_hangup_complete(port_number);
	}
}

STATIC void inbound_modem_hangup_complete(USHORT port_number)
{
	LINE_INFO_ENTRY *sptr_line_info;

	sptr_line_info = &ag.line_info_array[port_number];
	sptr_line_info->inbound_init_on = FALSE;

	free_serial_port_from_use(port_number);
	set_wan_port_owner(port_number, OWNED_BY_NONE);
	ag.owned_by_module[port_number] = UNKNOWN_MODULE;

	set_line_status(sptr_line_info, AGLS_LINE_FREE);
}

/* -- END CODE -------------------------------------------------------- */

