/* AGINMENU.C -- Handling of part of inbound state machine related to
**				 displaying inbound hosts list and accepting user input for
**				 choice of host to use.
** By: Sanjay
** Start: 13, August, 1996
** Done: 30, August, 1996
*/

#include "rtrstd.h"

#include "kag.h"
#include "vagstr.h"

#include "agpkttyp.h"
#include "agnetif.h"
#include "agline.h"
#include "agtx.h"
#include "aginbsup.h"
#include "aginmenu.h"

#include <udb.h>
#include <serial.h>

/* Global Prototypes */

/* Local Prototypes */
STATIC enum TEST blank_inbound_screen(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum TEST display_inbound_title(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum TEST display_inbound_footer(LINE_INFO_ENTRY *sptr_line_info);
STATIC enum TEST display_host_info(LINE_INFO_ENTRY *sptr_line_info, BYTE inbound_entry_index);

/* Statics */
STATIC char newline[] = { '\n', '\r' };
STATIC char *inbound_hosts_list_title = 
"        Welcome to Multi-Tech Asynchronous Communication Gateway (IPX)\n\r\
\n\r\
                       List of hosts waiting for calls\n\r\
\n\r\
    HOST ID   HOST NAME           NETWORK NUMBER   NODE NUMBER     STATUS \
\n\r\
    -------   ---------           --------------   -----------     ------ \
\n\r";

STATIC char inbound_hosts_list_footer[] = {
0x0D, 0x0A, \
0x0D, 0x0A, \
0x20, 0x20, 0x20, 0x20, 0x20, \
'U', ':', ' ', ' ', 'S', 'C', 'R', 'O', 'L', 'L', ' ', 'U', 'P', \
0x20, 0x20, 0x20, 0x20, 0x20, 0x20, \
'D', ':', ' ', ' ', 'S', 'C', 'R', 'O', 'L', 'L', ' ', 'D', 'O', 'W', 'N', \
0x20, 0x20, 0x20, 0x20, 0x20, 0x20, \
'P', ':', ' ', ' ', 'P', 'A', 'G', 'E', ' ', 'U', 'P', \
0x20, 0x20, 0x20, 0x20, 0x20, 0x20, \
'N', ':', ' ', ' ', 'P', 'A', 'G', 'E', ' ', 'D', 'O', 'W', 'N', \
0x0D, 0x0A, \
0x0D, 0x0A, \
0x0D, 0x0A, \
0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, \
0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, \
'E', 'N', 'T', 'E', 'R', ' ', 'H', 'O', 'S', 'T', ' ', 'I', 'D', ' ', \
'Y', 'O', 'U', ' ', 'W', 'A', 'N', 'T', ' ', 'T', 'O', ' ', \
'C', 'O', 'N', 'N', 'E', 'C', 'T', ' ', 'T', 'O', ':', ' ', ' ',  0 };

/* -- CODE ------------------------------------------------------------- */

enum BOOLEAN display_host_list(LINE_INFO_ENTRY *sptr_line_info)
{
	BYTE i;


	if (sptr_line_info->blank_screen_done == FALSE)
	{
		if (blank_inbound_screen(sptr_line_info) == FAIL)
			return FALSE;
		else
			sptr_line_info->blank_screen_done = TRUE;
	}
	if (sptr_line_info->title_display_done == FALSE)
	{
		if (display_inbound_title(sptr_line_info) == FAIL)
			return FALSE;
		else
			sptr_line_info->title_display_done = TRUE;
	}
	if (sptr_line_info->hosts_display_done == FALSE)
	{
		for (i = sptr_line_info->next_host_index; i < sptr_line_info->top_host_index + INBOUND_HOSTS_LIST_SIZE; i++)
		{
			if (display_host_info(sptr_line_info, i) == FAIL)
			{
				sptr_line_info->next_host_index = i;
				return FALSE;
			}
		}
		sptr_line_info->hosts_display_done = TRUE;
	}

	if (sptr_line_info->footer_display_done == FALSE)
	{
		if (display_inbound_footer(sptr_line_info) == FAIL)
			return FALSE;
		else
			sptr_line_info->footer_display_done = TRUE;
	}

	sptr_line_info->blank_screen_done = FALSE;
	sptr_line_info->title_display_done = FALSE;
	sptr_line_info->hosts_display_done = FALSE;
	sptr_line_info->footer_display_done = FALSE;
	sptr_line_info->next_host_index = sptr_line_info->top_host_index;

	sptr_line_info->next_inbound_state = IS_GET_MENU_INPUT;
	sptr_line_info->inbound_state_timer = GET_MENU_INPUT_DELAY;

	flush_serial_rx_buffers((USHORT)sptr_line_info->line_id);
	reset_inbound_input_state(sptr_line_info, TRUE);

	return TRUE;
}

STATIC enum TEST blank_inbound_screen(LINE_INFO_ENTRY *sptr_line_info)
{
	for (; sptr_line_info->next_line_to_blank < REMOTE_TERMINAL_HEIGHT; sptr_line_info->next_line_to_blank++)
		if (write_serial_port((USHORT)sptr_line_info->line_id, newline, sizeof(newline)) == FAIL)
			return FAIL;

	sptr_line_info->next_line_to_blank = 0;
	return PASS;
}

STATIC enum TEST display_inbound_title(LINE_INFO_ENTRY *sptr_line_info)
{
	return write_serial_port((USHORT)sptr_line_info->line_id, inbound_hosts_list_title, strlen(inbound_hosts_list_title));
}

STATIC enum TEST display_inbound_footer(LINE_INFO_ENTRY *sptr_line_info)
{
	return write_serial_port((USHORT)sptr_line_info->line_id, inbound_hosts_list_footer, strlen(inbound_hosts_list_footer));
}

STATIC enum TEST display_host_info(LINE_INFO_ENTRY *sptr_line_info, BYTE inbound_entry_index)
{
#define NAME_DISPLAY_SIZE	20

	int i;
	char user_name_buf[41];			/* Why 41, huh? */
	char blank_buffer[NAME_DISPLAY_SIZE + 1];		/* Same as space to print user name */
	SESSION_TABLE_ENTRY *sptr_inbound_session_entry;


	sptr_inbound_session_entry = get_ith_element_from_inbound_list(sptr_line_info, inbound_entry_index);

	/* If there is no such element, display a dummy line */
	if (sptr_inbound_session_entry == NULL)
	{
		if (write_serial_port((USHORT)sptr_line_info->line_id, newline, sizeof(newline)) == FAIL)
			return FAIL;
		else
			return PASS;
	}

	get_ras_user_name(sptr_inbound_session_entry->user_entry, user_name_buf);
	user_name_buf[USER_NAME_LENGTH - 1] = '\0';		/* We support 16 char name only */
	strcpy(blank_buffer, user_name_buf);
	for (i = strlen(blank_buffer); i <= NAME_DISPLAY_SIZE; i++)
		blank_buffer[i] = ' ';
	blank_buffer[NAME_DISPLAY_SIZE + 1] = '\0';

	/* If there is such an element, format and display the line. */
	sprintf(sptr_line_info->line_display_buffer, 
			"% 8d      %s  %08X      %08X%04X    %s\n\r",
			((int)inbound_entry_index + 1), 
			blank_buffer,
			get_client_network_number(sptr_inbound_session_entry),
			get_client_node_number_ulong(sptr_inbound_session_entry),
			get_client_node_number_ushort(sptr_inbound_session_entry),
			(sptr_inbound_session_entry->session_status == AG_SESS_ACTIVE) ? "FREE" : "BUSY");

	if (write_serial_port((USHORT)sptr_line_info->line_id, sptr_line_info->line_display_buffer, strlen(sptr_line_info->line_display_buffer)) == FAIL)
		return FAIL;

	return PASS;

#undef NAME_DISPLAY_SIZE
}

SESSION_TABLE_ENTRY *get_ith_element_from_inbound_list(LINE_INFO_ENTRY *sptr_line_info, BYTE inbound_entry_index)
{
	BYTE *sptr_session_link;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	sptr_session_link = (BYTE *)get_pointer_to_first_entry_in_list((LINK *)&sptr_line_info->sessions_awaiting_inbound);
	while (inbound_entry_index != 0 && sptr_session_link != NULL)
	{
		sptr_session_link = get_pointer_to_next_entry_in_list((LINK *)sptr_session_link);
		inbound_entry_index--;
	}

	if (sptr_session_link == NULL)
		return NULL;

	sptr_session_entry = (SESSION_TABLE_ENTRY *)(sptr_session_link - offsetof(SESSION_TABLE_ENTRY, inbound_links));
	return sptr_session_entry;
}

enum BOOLEAN handle_host_list(LINE_INFO_ENTRY *sptr_line_info)
{
	if (sptr_line_info->inbound_state_timer == 0)
	{
		sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		return TRUE;
	}

	/* Wait for input */
	if (sptr_line_info->inbound_input_valid == FALSE)
		return FALSE;

	sptr_line_info->next_inbound_state = IS_VALIDATE_HOST;
	return TRUE;
}

enum BOOLEAN validate_host(LINE_INFO_ENTRY *sptr_line_info)
{
	int host_entry_index;
	SESSION_TABLE_ENTRY *sptr_inbound_session_entry;


	host_entry_index = atoi(sptr_line_info->inbound_input_buffer);
	host_entry_index--;					/* Make 0 based */
	if (host_entry_index >= sptr_line_info->number_of_inbound_users)
	{
		sptr_line_info->inbound_login_attempts++;
		if (sptr_line_info->inbound_login_attempts == MAX_INBOUND_LOGIN_ATTEMPTS)
		{
			sptr_line_info->inbound_login_attempts = 0;
			sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		}
		else
			sptr_line_info->next_inbound_state = IS_DISPLAY_NAME_ERROR;

		return TRUE;
	}

	sptr_inbound_session_entry = get_ith_element_from_inbound_list(sptr_line_info, host_entry_index);
	sptr_line_info->sptr_session_entry = NULL;
	if (sptr_inbound_session_entry == NULL ||
		!(sptr_inbound_session_entry->session_status == AG_SESS_ACTIVE ||
		sptr_inbound_session_entry->session_status == AG_SESS_CONNECTED))
	{
		sptr_line_info->inbound_login_attempts++;
		if (sptr_line_info->inbound_login_attempts == MAX_INBOUND_LOGIN_ATTEMPTS)
		{
			sptr_line_info->inbound_login_attempts = 0;
			sptr_line_info->next_inbound_state = IS_DISPLAY_ABORTED_LOGIN;
		}
		else
			sptr_line_info->next_inbound_state = IS_DISPLAY_NAME_ERROR;

		return TRUE;
	}

	sptr_line_info->sptr_session_entry = sptr_inbound_session_entry;
	sptr_inbound_session_entry->session_status = AG_SESS_CONNECTED;

	if (has_ras_user_password(sptr_line_info->sptr_session_entry->user_entry))
		sptr_line_info->next_inbound_state = IS_PROMPT_PASSWORD;
	else
		sptr_line_info->next_inbound_state = IS_CHECK_CALLBACK;

	return TRUE;
}

/* -- END CODE -------------------------------------------------------- */

