/* AGNETIF.C -- Generic network interface functions
** By: Sanjay
** Start: 23, July, 1996
** Done: 30, August, 1996
*/

#include "rtrstd.h"

#include "kag.h"
#include "vagstr.h"
#include "agpkttyp.h"
#include "agutil.h"

#include "agipx.h"
#include "agspx.h"
#include "agnetif.h"

/* Local Prototypes */

/* -- CODE ------------------------------------------------------------- */

enum BOOLEAN is_ag_net_interface_up(void)
{
	BYTE_ENUM(BOOLEAN) protocol_enabled;


	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		if (lsl_control(PROTOCOL_STACK_CONTROL, "IPX Routing", IS_PROTOCOL_STACK_ENABLED, 0, (ULONG)&protocol_enabled) == FAIL)
			return FALSE;
		if (protocol_enabled == FALSE)
			return FALSE;

		if (lsl_control(RESOLVE_SOCKET_API, TRANSPORT_LAYER_TYPE, IS_TRANSPORT_ENABLED, SPX_TRANSPORT, (ULONG)&protocol_enabled) == FAIL)
			return FALSE;
		return (enum BOOLEAN) protocol_enabled;


		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Currently IP and NETBIOS are not supported */
	default:
		return FALSE;
	}

	return TRUE;
}

enum TEST initialize_network_interface()
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		if (initialize_for_ipx_spx() == FAIL)
		{
			ag_printf(AG_INIT_PRINTF, "AG: IPX/SPX interface initialization failed\n\r");
			return FAIL;
		}
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Currently IP and NETBIOS are not supported */
	default:
		return FAIL;
	}

	return PASS;
}

enum TEST allocate_protocol_structure(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_allocate_protocol_structure(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Currently not supported */
	default:
		break;
	}
	return FAIL;
}

void free_protocol_structure(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_free_protocol_structure(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Currently not supported */
	default:
		break;
	}
}

void do_network_interface_periodic_actions()
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		do_ipx_interface_periodic_actions();
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
	default:
		/* Currently not supported */
		break;
	}
}

void send_network_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_send_packet(sptr_session_entry, sptr_ag_packet, data_size, cb_ptr);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemeted */
	default:
		break;
	}
}

enum TEST send_network_internal_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_send_internal_packet(sptr_session_entry, sptr_ag_packet, data_size);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemeted */
	default:
		break;
	}

	return FAIL;
}

enum TEST send_serial_rx_packet_on_network(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_send_serial_rx_packet_on_network(sptr_session_entry, sptr_ag_packet, data_size);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemeted */
	default:
		break;
	}
	return PASS;
}

USHORT get_max_network_rx_buffer_size()
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag.ipx_interface.max_net_rx_buffer_size;
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}

	return 0;
}

BYTE *get_a_network_send_packet(USHORT data_size)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return get_a_spx_send_buffer(data_size);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}

	return NULL;
}

void free_a_network_send_packet(BYTE *bptr)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		free_a_spx_send_buffer(bptr);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void reduce_network_receives(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_reduce_network_receives(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void increase_network_receives(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_increase_network_receives(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void free_network_stuff(UNION_AG_PACKET *sptr_ag_packet)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_free_network_stuff(sptr_ag_packet);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

enum BOOLEAN wait_for_network_packets(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		if (((IPX_SPECIFIC_INFO *)sptr_session_entry->protocol_specific_info)->receives_issued == 0)
		{
			ag_spx_issue_receives_for_session(sptr_session_entry);
			return TRUE;
		}
		return FALSE;
		break;

	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void reduce_receives_and_wait_for_network_packets(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		((IPX_SPECIFIC_INFO *)sptr_session_entry->protocol_specific_info)->receives_issued--;
		ag_spx_issue_max_receives_for_session(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void terminate_network_connection(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		if (sptr_session_entry->session_status != AG_SESS_ABORTED)
			ag_spx_terminate_connection(sptr_session_entry);
		break;

	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */

	default:
		break;
	}
}

enum BOOLEAN network_says_ok_to_cleanup(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_says_ok_to_cleanup(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

enum BOOLEAN network_says_cancels_done(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_says_cancels_done(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

void cancel_pending_network_calls(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_cancel_pending_calls(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

ULONG get_client_network_number(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_get_client_network_number(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
	return 0L;
}

ULONG get_client_node_number_ulong(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_get_client_node_number_ulong(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
	return 0L;
}

USHORT get_client_node_number_ushort(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		return ag_spx_get_client_node_number_ushort(sptr_session_entry);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
	return 0;
}

void fill_user_address(SESSION_TABLE_ENTRY *sptr_session_entry, BYTE *sptr_address)
{
	switch (ag.network_interface_to_use)
	{
	case NET_INTERFACE_IPXSPX:
		ag_spx_fill_user_address(sptr_session_entry, sptr_address);
		break;
	case NET_INTERFACE_IP:
	case NET_INTERFACE_NETBIOS:
		/* Not implemented */
	default:
		break;
	}
}

/* -- END CODE -------------------------------------------------------- */

