/* AGRESP.C -- Handlers for the various requests from MCSI clients 
**			   (RESP for RESPONSE)
** By: Sanjay
** Start: 12, July, 1996
** Done: 30, August, 1996
*/

/* General Notes:
**	1. All the functions here return TRUE if they result in a network send
**	   or if for some reason the receive buffer is retained as in-use.
**	   Else they return FALSE. TRUE or FALSE here does not imply function 
**	   success or failure.
*/
	
#include "rtrstd.h"
#include <serial.h>
#include <msm.h>
#include <wanmgr.h>

#include "kag.h"
#include "vagstr.h"
#include "agpkttyp.h"
#include "agutil.h"
#include "agline.h"
#include "agnetif.h"
#include "agsess.h"
#include "agresp.h"
#include "agtx.h"
#include "aginbsup.h"
#include "agrcon.h"

#include <udb.h>

/* External Prototypes */
extern BYTE old_modem_control_byte[MAX_WAN_PORTS] ;

/* Local Prototypes */
STATIC enum BOOLEAN validate_name(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN validate_password(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_connect_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_status_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_disconnect_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_change_parameters(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_inbound_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN transfer_data_to_line(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);
STATIC enum BOOLEAN process_control_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr);

/* Global Data */
enum BOOLEAN (*packet_function_table[])(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr) = 
{
	validate_name,
	validate_password,
	process_connect_request,
	process_status_request,
	process_disconnect_request,
	process_change_parameters,
	process_remote_console_request,
	process_inbound_request,
	transfer_data_to_line,
	process_control_packet
};
/* Added by Naveen ... */
RAS_USER_DATABASE_RECORD common_temp_udb_record = 
			{
				"Guest",
				"",
				0,
				0xFFFF,
				0xFF,
				0,
				"",
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0
			};
/* ... Naveen 20/1/1997 */
/* -- CODE ------------------------------------------------------------- */

STATIC enum BOOLEAN validate_name(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE *user_entry;
	NAME_VALIDATE_REQ_TYPE name_validate_request;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a VALIDATE NAME request\n\r");

	change_endian(sptr_ag_packet);

	name_validate_request = sptr_ag_packet->packet_type.name_validate_request;
	name_validate_request.user_name[USER_NAME_LENGTH - 1] = '\0';	/* Make sure of termination */

	/* From now the IN packet will be treated as an OUT packet */

	sptr_ag_packet->packet_class = AG_NAME_VALIDATE_RESP;
	memset(&sptr_ag_packet->packet_type.name_validate_response, 0, sizeof(sptr_ag_packet->packet_type.name_validate_response));

/* Added by Naveen ...*/
	if (!get_ras_number_of_users())
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending successful NAME VALIDATE RESPONSE packet because UserDataBase is Empty\n\r");
		sptr_ag_packet->packet_type.name_validate_response.name_valid = TRUE;
		sptr_ag_packet->packet_type.name_validate_response.password_present = FALSE;
		user_entry = (BYTE*) &common_temp_udb_record;
	}
/* ... Naveen  20/1/1997*/
	else
	{
		user_entry = is_valid_ras_user(name_validate_request.user_name);
		if (user_entry != NULL)
		{
			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending successful NAME VALIDATE RESPONSE packet \n\r");
			sptr_ag_packet->packet_type.name_validate_response.name_valid = TRUE;
			sptr_ag_packet->packet_type.name_validate_response.password_present = has_ras_user_password(user_entry);
		}
		else
		{
			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending unsuccessful NAME VALIDATE RESPONSE packet \n\r");
			sptr_ag_packet->packet_type.name_validate_response.name_valid = FALSE;
			sptr_ag_packet->packet_type.name_validate_response.password_present = FALSE;
		}
	}
	/* Packet is ready to send */
	change_endian(sptr_ag_packet);

	send_network_packet(sptr_session_entry, sptr_ag_packet, (USHORT)(sizeof(NAME_VALIDATE_RESP_TYPE) + sizeof(BYTE)), cb_ptr);
	return TRUE;
}

STATIC enum BOOLEAN validate_password(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE *user_entry;
	PASSWORD_VALIDATE_REQ_TYPE password_validate_request;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a VALIDATE PASSWORD request\n\r");

	change_endian(sptr_ag_packet);
	
	password_validate_request = sptr_ag_packet->packet_type.password_validate_request;
	password_validate_request.user_name[USER_NAME_LENGTH - 1] = '\0';	/* Make sure of termination */
	password_validate_request.encrypted_password[PASSWORD_LENGTH - 1] = '\0';	/* Make sure of termination */

	/* From now the IN packet will be treated as an OUT packet */

	sptr_ag_packet->packet_class = AG_PASSWORD_VALIDATE_RESP;
	memset(&sptr_ag_packet->packet_type.password_validate_response, 0, sizeof(sptr_ag_packet->packet_type.password_validate_response));

	user_entry = is_valid_ras_user(password_validate_request.user_name);
	if (user_entry != NULL)
	{
		if (has_ras_user_password(user_entry))
		{
			if (is_valid_ras_user_password(user_entry, password_validate_request.encrypted_password))
			{
				ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending successful PASSWORD VALIDATE RESPONSE packet \n\r");
				sptr_ag_packet->packet_type.password_validate_response.password_valid_state = AGPV_VALID_PASSWORD;
			}
			else
			{
				ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending unsuccessful PASSWORD VALIDATE RESPONSE packet\n\r");
				sptr_ag_packet->packet_type.password_validate_response.password_valid_state = AGPV_INVALID_PASSWORD;
			}
		}
		else
		{
			if (strlen(password_validate_request.encrypted_password) == 0)
			{
				ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending successful PASSWORD VALIDATE RESPONSE packet \n\r");
				sptr_ag_packet->packet_type.password_validate_response.password_valid_state = AGPV_VALID_PASSWORD;
			}
			else
			{
				ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending unsuccessful PASSWORD VALIDATE RESPONSE packet \n\r");
				sptr_ag_packet->packet_type.password_validate_response.password_valid_state = AGPV_INVALID_PASSWORD;
			}
		}
	}
	else
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: User %s does not exist\n\r", password_validate_request.user_name);
		sptr_ag_packet->packet_type.password_validate_response.password_valid_state = AGPV_INVALID_PASSWORD;
	}

	change_endian(sptr_ag_packet);

	send_network_packet(sptr_session_entry, sptr_ag_packet, (USHORT)(sizeof(PASSWORD_VALIDATE_RESP_TYPE) + sizeof(BYTE)), cb_ptr);
	return TRUE;
}

STATIC enum BOOLEAN process_connect_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	USHORT i;
	BYTE *user_entry;
	enum BOOLEAN atleast_one_line;
	LINE_VARS_TYPE *sptr_line_vars;
	LINE_INFO_ENTRY *sptr_line_info;
	CONNECT_REQ_TYPE connect_request;
	CONNECT_RESP_TYPE *sptr_connect_resp;
	enum AG_PORT_AVAILABILITY port_status;
	USHORT requested_channel = 0;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a CONNECT request\n\r");

	change_endian(sptr_ag_packet);
	
	connect_request = sptr_ag_packet->packet_type.connect_request;
	connect_request.user_name[USER_NAME_LENGTH - 1] = '\0';	/* Make sure of termination */
	connect_request.general_name[GENERAL_NAME_LENGTH - 1] = '\0';	/* Make sure of termination */
	connect_request.specific_name[SPECIFIC_NAME_LENGTH - 1] = '\0';	/* Make sure of termination */

	/* From now IN packet will be used for OUT purposes */

	sptr_ag_packet->packet_class = AG_CONNECT_RESP;
	sptr_connect_resp = &sptr_ag_packet->packet_type.connect_response;
	memset(sptr_connect_resp, 0, sizeof(CONNECT_RESP_TYPE));

	sptr_line_info = &ag.line_info_array[0];
	for (i = 0; i < ag.number_of_lines; i++, sptr_line_info++)
	{
		if (!strcmp(connect_request.general_name, sptr_line_info->line_vars.general_name) &&
				!strcmp(connect_request.specific_name, sptr_line_info->line_vars.specific_name))
			break;
	}
	requested_channel = i;
	user_entry = is_valid_ras_user(connect_request.user_name);
/* Added by Naveen ...*/
	if (user_entry == NULL && !(get_ras_number_of_users()))
	{
		user_entry = (BYTE*) &common_temp_udb_record;
	}
/* ... Naveen 20/1/1997*/

	if (user_entry == NULL)
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: user %s does not exist\n\r", connect_request.user_name);
		sptr_connect_resp->connect_status = AGCR_INVALID_USER_NAME;
	}
	else if (i == ag.number_of_lines)
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: Requested line does not exist\n\r");
		sptr_connect_resp->connect_status = AGCR_INVALID_LINE_NAME;
	}
	else if (has_dialout_access(user_entry, i) == FALSE)
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: User %s does not have dial-out access\n\r", connect_request.user_name);
		sptr_connect_resp->connect_status = AGCR_ACCESS_DENIED;
	}
/*Added By  Naveen.P.N. ... */
	else if(connect_request.connect_type == INBOUND_MODEM && !get_ras_number_of_users())
	{
		ag_printf(AG_MCSI_OUT_PRINTF, "AG: user %s does not exist\n\r", connect_request.user_name);
		sptr_connect_resp->connect_status = AGCR_INVALID_USER_NAME;
	}
/* ... Naveen.P.N 20/1/1997 */
	else
	{
		port_status = ag_get_port_availability(i);
		if (connect_request.connect_type == INBOUND_MODEM)
		{
			/* This is like a INBOUND CONNECT request except that instead of
			** the line specified, all known lines are used for inbound.
			*/

			sptr_connect_resp->connect_status = AGCR_NOT_CONNECTED;
			atleast_one_line = FALSE;

			for (i = 0; i < ag.number_of_lines; i++)
			{
				port_status = ag_get_port_availability(i);
				if (port_status == AG_PORT_DISABLED || port_status == AG_PORT_NOT_ASYNC)
					continue;

				if (has_dialout_inbound_access(user_entry, i))
				{
					atleast_one_line = TRUE;

					sptr_line_info = &ag.line_info_array[i];
					add_inbound_session_to_line(sptr_line_info, sptr_session_entry);

					if (sptr_line_info->line_status == AGLS_LINE_FREE)
						add_line_to_inbound_list(sptr_line_info);
				}
			}

			if (atleast_one_line == TRUE)
			{
				sptr_session_entry->user_entry = user_entry;

				ag.stats.total_dialin_users++;

				sptr_connect_resp->connect_status = AGCR_CONNECTED;
				sptr_connect_resp->channel_number = UNKNOWN_LINE;
				sptr_connect_resp->EIA_status = 0;
			}
		}
		else if (sptr_line_info->line_status != AGLS_LINE_FREE)
		{
			sptr_connect_resp->connect_status = AGCR_NOT_CONNECTED;

			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Connect request rejected as line %u is in use\n\r", i);
		}
		else if (port_status != AG_PORT_AVAILABLE)
		{
			sptr_connect_resp->connect_status = AGCR_NOT_CONNECTED;

			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Connect request rejected as line %u is in use or bad type\n\r", i);
		}
		else
		{
			/* Fill in the fields of the response packet */
			memcpy(sptr_connect_resp->general_name, connect_request.general_name, GENERAL_NAME_LENGTH);
			memcpy(sptr_connect_resp->specific_name, connect_request.specific_name, SPECIFIC_NAME_LENGTH);
			sptr_connect_resp->channel_number = i;
			sptr_connect_resp->connect_status = AGCR_CONNECTED;
			sptr_connect_resp->EIA_status = get_EIA_status((USHORT)i);

/*       This is commented out by Naveen, this is because at this point 
         line information contains old values and it is inited to default
         after this statement. That's why this statement is moved after
         calling setup_port_for_dial_out.

			set_connect_vars_info(&sptr_connect_resp->connect_vars_info, &sptr_line_info->line_vars); 
*/

			/* Update session table entry information to reflect connection status, etc */
			sptr_session_entry->line_in_use = i;
			sptr_session_entry->session_status = AG_SESS_CONNECTED;
			sptr_session_entry->user_entry = user_entry;

			/* Update the line information entry */
			delete_line_from_inbound_list(sptr_line_info);

			sptr_line_info->connect_mode = connect_request.connect_type;
			set_line_status(sptr_line_info, AGLS_CONNECTING);
			sptr_line_info->sptr_session_entry = sptr_session_entry;
			sptr_line_info->inactivity_timer = sptr_line_info->line_vars.idle_time;

			sptr_line_vars = &sptr_line_info->line_vars;
			sptr_line_vars->current_baud_rate = sptr_line_vars->default_baud_rate;
			sptr_line_vars->current_data_bits = sptr_line_vars->default_data_bits;
			sptr_line_vars->current_parity = sptr_line_vars->default_parity;
			sptr_line_vars->current_stop_bits = sptr_line_vars->default_stop_bits;
			sptr_line_vars->current_max_packet_size = sptr_line_vars->default_max_packet_size;

			if (setup_port_for_dialout(sptr_line_info) == FAIL)
			{
				sptr_connect_resp->connect_status = AGCR_NOT_CONNECTED;

				/* Clear off information stored */
				
				sptr_session_entry->line_in_use = UNKNOWN_LINE;
				sptr_session_entry->session_status = AG_SESS_ACTIVE;
				sptr_session_entry->user_entry = NULL;

				sptr_line_info->connect_mode = sptr_line_info->line_vars.line_type;
				set_line_status(sptr_line_info, AGLS_LINE_FREE);
				sptr_line_info->sptr_session_entry = NULL;
				sptr_line_info->inactivity_timer = sptr_line_info->line_vars.idle_time;
			}
			else
			{
				ag_printf(AG_MCSI_OUT_PRINTF, "AG: Setup outbound connection on line %u\n\r", i);

/* The followinmg statement is added by Naveen.P.N. 09/07/1997 */
   			set_connect_vars_info(&sptr_connect_resp->connect_vars_info, &sptr_line_info->line_vars); 

				sptr_line_info->stats.total_dialout_calls++;
				get_current_date_time(&sptr_line_info->stats.connect_time);

				ag.stats.number_of_connections++;
				ag.stats.outbound_connections++;
				ag.stats.total_dialout_calls++;
			}
		}
	}
	
/* Added by Naveen...*/
			old_modem_control_byte[requested_channel] = 0;
/* ... Naveen 20/1/1997*/
	change_endian(sptr_ag_packet);

	send_network_packet(sptr_session_entry, sptr_ag_packet, (USHORT)(sizeof(CONNECT_RESP_TYPE) + sizeof(BYTE)), cb_ptr);

	return TRUE;
}


STATIC enum BOOLEAN process_status_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE *user_entry;
	BYTE next_line_id;
	USHORT lines_considered;
	char user_name_buffer[41];		/* From database, we get 40 + 1 char name */
	LINE_INFO_ENTRY *sptr_line_info;
	LINE_STAT_INFO_TYPE *sptr_line_status;
	STATUS_REQ_TYPE status_request_packet;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a GET STATUS request\n\r");

	change_endian(sptr_ag_packet);
	
	status_request_packet = sptr_ag_packet->packet_type.status_request;
	status_request_packet.user_name[USER_NAME_LENGTH - 1] = '\0';

	/* From here packet is for OUT purposes only */

	sptr_ag_packet->packet_class = AG_STATUS_RESP;
	memset(&sptr_ag_packet->packet_type.status_response, 0, 
		((sizeof(STATUS_RESP_TYPE) - sizeof(LINE_STAT_INFO_TYPE)) + (MAX_LINES_PER_STATUS_PACKET * sizeof(LINE_STAT_INFO_TYPE))));
	
	lines_considered = 0;
/*
	LOOK AT THIS AGAIN !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	Looks like, there is a bug here. The packet that has been memset to 0
	is being looked at for request parameters.
*/
	next_line_id = status_request_packet.last_line_id;
	if (next_line_id < ag.number_of_lines)
		sptr_line_info = &ag.line_info_array[next_line_id];
	sptr_line_status = sptr_ag_packet->packet_type.status_response.line_status_array;
	while ((next_line_id < ag.number_of_lines) && (lines_considered < MAX_LINES_PER_STATUS_PACKET))
	{
		memcpy(sptr_line_status->general_name, sptr_line_info->line_vars.general_name, GENERAL_NAME_LENGTH);
		memcpy(sptr_line_status->specific_name, sptr_line_info->line_vars.specific_name, SPECIFIC_NAME_LENGTH);

/*
	Sachin added the second contition on 14/11/1996.
	If the port is already in use by a different module, it is to be reported
	as busy. For now, it may be in use for ras or router (OWNED_BY_PPP in both
	cases). At a later stage, it may be owned by the terminal server.
*/
		/* if ((sptr_line_info->line_status == AGLS_LINE_BUSY) || (get_wan_port_owner(next_line_id) != OWNED_BY_NONE)) */
		if (ag_get_port_availability (next_line_id) != AG_PORT_AVAILABLE)
		{
			sptr_line_status->line_status = AGLS_LINE_BUSY;
			sptr_line_status->line_class = sptr_line_info->connect_mode;
			if (sptr_line_info->sptr_session_entry->user_entry != NULL && ag_get_port_availability (next_line_id) == OWNED_BY_AG)
			{
				get_ras_user_name(sptr_line_info->sptr_session_entry->user_entry, user_name_buffer);
				user_name_buffer[USER_NAME_LENGTH - 1] = '\0';	/* Truncate to 16 chars */
				strcpy(sptr_line_status->line_owner, user_name_buffer);
			}
		}
		else
		{
			sptr_line_status->line_status = AGLS_LINE_FREE;
			sptr_line_status->line_class = sptr_line_info->line_vars.line_type;
		}
		user_entry = is_valid_ras_user(status_request_packet.user_name);
		if (user_entry == NULL || ag_get_port_availability (next_line_id) != AG_PORT_AVAILABLE)
			sptr_line_status->access = FALSE;
		else
			sptr_line_status->access = has_dialout_access(user_entry, next_line_id);
/* Added by Naveen... */
		if (!get_ras_number_of_users() &&  ag_get_port_availability (next_line_id) == AG_PORT_AVAILABLE)
			sptr_line_status->access = TRUE;
/* ... Naveen */
		fill_client_style_line_parameters_info(&sptr_line_status->line_parameters, &sptr_line_info->line_vars);
		big_to_little_endian_ushort(&sptr_line_status->line_parameters.baud_divisor);

		lines_considered++;
		next_line_id++;
		sptr_line_status++;
		sptr_line_info++;
	}

	sptr_ag_packet->packet_type.status_response.number_of_lines = ag.number_of_lines;
	sptr_ag_packet->packet_type.status_response.last_line_id = next_line_id;

	change_endian(sptr_ag_packet);

	ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending STATUS RESPONSE packet \n\r");

	send_network_packet(sptr_session_entry, sptr_ag_packet, (USHORT)((sizeof(STATUS_RESP_TYPE) - sizeof(LINE_STAT_INFO_TYPE)) + (lines_considered * sizeof(LINE_STAT_INFO_TYPE)) + sizeof(BYTE)), cb_ptr);
	return TRUE;
}


STATIC enum BOOLEAN process_disconnect_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE line_id;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a DISCONNECT request\n\r");

	change_endian(sptr_ag_packet);
	
	line_id = sptr_ag_packet->packet_type.disconnect_request.channel_number;

	/* Since this terminate request is from a client end, we do not need
	** to inform client again in anyway. We will simply terminate the
	** SPX connection and cleanup up any associated line.
	*/

	/* Terminate the network connection. Terminating will abort all pending 
	** sends and receives. So post routines will be able to free up buffers.
	** NOTE: We need to wait until the post routine before finishing the full
	** cleanup which will be done in the timer.
	*/
	if (line_id == UNKNOWN_LINE)
	{
		ag_printf(AG_SESS_PRINTF, "AG: Starting terminating of session without allocated line\n\r");
		terminate_network_connection(sptr_session_entry);
	}
	else if (line_id < ag.number_of_lines)
	{
		ag_printf(AG_SESS_PRINTF, "AG: Starting termination of session using line %u\n\r", (USHORT)line_id);

		if (ag.line_info_array[line_id].sptr_session_entry != NULL)
			terminate_network_connection(ag.line_info_array[line_id].sptr_session_entry);
		else
			terminate_network_connection(sptr_session_entry);
	}
	else
	{
		ag_printf(AG_ALARM_PRINTF, "AG: Got a disconnect for invalid line %u\n\r", (USHORT)line_id);
	}

	return FALSE;
}


STATIC enum BOOLEAN process_change_parameters(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE line_id;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a CHANGE PARAMETERS request\n\r");

	change_endian(sptr_ag_packet);
	
	line_id = sptr_ag_packet->packet_type.change_parameters_request.channel_number;

	if (line_id != UNKNOWN_LINE && line_id < ag.number_of_lines && ag.owned_by_module[line_id] != UNKNOWN_MODULE)
		change_line_parameters(line_id, &sptr_ag_packet->packet_type.change_parameters_request.connect_vars_info);

	ag_printf(AG_MCSI_OUT_PRINTF, "AG: Sending a CHANGE PARAMETERS response\n\r");

	return FALSE;
}


STATIC enum BOOLEAN process_inbound_request(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE i, line_id;
	BYTE *user_entry;
	LINE_INFO_ENTRY *sptr_line_info;
	enum BOOLEAN has_access, valid_line;
	CONNECT_RESP_TYPE *sptr_connect_resp;
	INBOUND_CONNECT_REQ_TYPE inbound_request;
		
	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got an INBOUND CONNECT request\n\r");

	change_endian(sptr_ag_packet);
	
	inbound_request = sptr_ag_packet->packet_type.inbound_connect_request;
	inbound_request.user_name[USER_NAME_LENGTH - 1] = '\0';

	/* From now IN packet will be used for OUT purposes */
	sptr_ag_packet->packet_class = AG_CONNECT_RESP;
	sptr_connect_resp = &sptr_ag_packet->packet_type.connect_response;
	memset(sptr_connect_resp, 0, sizeof(CONNECT_RESP_TYPE));

	user_entry = is_valid_ras_user(inbound_request.user_name);
	if (user_entry == NULL)
		sptr_connect_resp->connect_status = AGCR_INVALID_USER_NAME;
	else
	{
		sptr_connect_resp->channel_number = UNKNOWN_LINE;
		sptr_connect_resp->EIA_status = 0;

		has_access = valid_line = FALSE;

		for (i = 0; i < inbound_request.number_of_channels; i++)
		{
			line_id = inbound_request.services_array[i];
			if (line_id >= ag.number_of_lines)
				continue;

/* Naveen, 01/02/1996 */

#if 0
			if (ag_get_port_availability(line_id) != AG_PORT_AVAILABLE)
				continue;
#endif
			if ((ag_get_port_availability (line_id) == AG_PORT_DISABLED) ||
			    (ag_get_port_availability (line_id) == AG_PORT_NOT_ASYNC))
				continue ;

/* Naveen, 01/02/1996 */

			valid_line = TRUE;			/* Atleast one valid line */

			if (has_dialout_inbound_access(user_entry, line_id) == 0)
				continue;
			has_access = TRUE;


			sptr_line_info = &ag.line_info_array[line_id];
			add_inbound_session_to_line(sptr_line_info, sptr_session_entry);

			if (sptr_line_info->line_status == AGLS_LINE_FREE)
				add_line_to_inbound_list(sptr_line_info);
		}

		if (valid_line == FALSE)
		{
 			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Requested line does not exist\n\r");
			sptr_connect_resp->connect_status = AGCR_INVALID_LINE_NAME;
		}
		else if (has_access == FALSE)
		{
			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Access denied for the requested line \n\r");
			sptr_connect_resp->connect_status = AGCR_ACCESS_DENIED;
		}
		else
		{
			ag_printf(AG_MCSI_OUT_PRINTF, "AG: Connection established for inbound request \n\r");
			sptr_session_entry->user_entry = user_entry;
			
			ag.stats.total_dialin_users++;

			sptr_connect_resp->connect_status = AGCR_CONNECTED;
			sptr_connect_resp->channel_number = UNKNOWN_LINE;
			sptr_connect_resp->EIA_status = 0;
		}
	}

	change_endian(sptr_ag_packet);

	send_network_packet(sptr_session_entry, sptr_ag_packet, (USHORT)(sizeof(CONNECT_RESP_TYPE) + sizeof(BYTE)), cb_ptr);
	return TRUE;
}


STATIC enum BOOLEAN transfer_data_to_line(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE line_id;
	LINE_INFO_ENTRY *sptr_line_info;

#ifdef DEBUG
	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a SEND DATA request\n\r");
#endif

	change_endian(sptr_ag_packet);
	
	line_id = sptr_ag_packet->packet_type.data_packet.channel_number;
	if (line_id >= ag.number_of_lines)
	{
		ag_printf(AG_ALARM_PRINTF, "AG: Data send requested on invalid line %u\n\r", (USHORT)line_id);
		return FALSE;
	}

	sptr_line_info = &ag.line_info_array[line_id];
	if (sptr_line_info->line_status != AGLS_LINE_BUSY)
		return FALSE;

	/* Since we need a pointer to the 'cb_ptr' variable when the serial
	** send complete is called, do some jugglery and put it just ahead
	** of the AG packet.
	*/
	*((void **)(((BYTE *)sptr_ag_packet) - sizeof(void *))) = cb_ptr;
	data_size -= (sizeof(DATA_TYPE) - sizeof(BYTE) + sizeof(BYTE));

#ifdef DEBUG
	if (data_size != sptr_ag_packet->packet_type.data_packet.packet_size)
		printf("AG: Strange! Network receive did not receive all that was sent\n\r");
#endif /* DEBUG */

	/* Explanation. We need to give the packet to the serial driver now. 
	** But if during previous calls, we had queued up some packets here 
	** (since the serial driver could not accept it at that time), and those
	** packets have not yet been sent out (by the timer code), we need to 
	** continue queueing so that packets are not transmitted out of sequence.
	** End of explanation. Actually a bug fix. Not originally thought of.
	*/
	if (are_there_retained_packets(sptr_session_entry) == TRUE)
	{
		/* Take a shot at sending the retained entries here and now. */
		send_delayed_serial_packets(sptr_session_entry);
		
		if (are_there_retained_packets(sptr_session_entry) == TRUE)
		{
			/* Too bad -- serial tx is still stuck. So queue up. */
			if (retain_rxed_packet(sptr_session_entry, sptr_ag_packet, data_size) == FAIL)
				return FALSE;
			return TRUE;
		}
		else 
		{
			/* Good, we managed to flush the retained list, so try to send.
			** If send fails, then again retain.
			*/
			if (serial_tx_packet(line_id, sptr_ag_packet->packet_type.data_packet.data, data_size, APPLICATION) == FAIL)
			{
#ifdef DEBUG
				printf("AG: Delaying send on the serial side\n\r");
#endif /* DEBUG */
				if (retain_rxed_packet(sptr_session_entry, sptr_ag_packet, data_size) == FAIL)
					return FALSE;
				return TRUE;
			}
		}
	}
	else 
	{
		/* No retained packets in the first place. So send this new packet
		** out. If send fails, queue up.
		*/
		if (serial_tx_packet(line_id, sptr_ag_packet->packet_type.data_packet.data, data_size, APPLICATION) == FAIL)
		{
#ifdef DEBUG
			printf("AG: Delaying send on the serial side\n\r");
#endif /* DEBUG */
			if (retain_rxed_packet(sptr_session_entry, sptr_ag_packet, data_size) == FAIL)
				return FALSE;
			return TRUE;
		}
	}

	/* Reset the inactivity timer -- some action on the net side */
	sptr_line_info->inactivity_timer = sptr_line_info->line_vars.idle_time;

	/* Statistics update */
	update_line_line_tx_statistics(line_id, data_size);

	/* We return TRUE because we need to retain the buffer until the serial
	** send finishes. The serial send complete routine will arrange to
	** free the buffer.
	*/
	return TRUE;
}

STATIC enum BOOLEAN process_control_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *cb_ptr)
{
	BYTE line_id;
	BYTE control_value;
	BYTE control_source;

	ag_printf(AG_MCSI_IN_PRINTF, "AG: Got a CONTROL request to ");

	change_endian(sptr_ag_packet);
	
	line_id = sptr_ag_packet->packet_type.control_packet.channel_number;
	control_value = sptr_ag_packet->packet_type.control_packet.control_value;
	control_source = sptr_ag_packet->packet_type.control_packet.control_source;

	if (line_id >= ag.number_of_lines)
		return FALSE;

	switch (control_source)
	{
	case AGCS_SET_LINE_CONTROL:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Set line control register for line %d\n\r", line_id);
		set_line_control((USHORT)line_id, control_value);
		break;

	case AGCS_SET_MODEM_CONTROL:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Set modem control register for line %d\n\r", line_id);
		set_modem_control((USHORT)line_id, control_value);
		break;

	case AGCS_FLUSH_TX_BUFFER:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Flush transmit buffer for line %d\n\r", line_id);
		flush_serial_tx_buffers((USHORT)line_id);
		break;

	case AGCS_FLUSH_RX_BUFFER:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Flush receive buffer for line %d\n\r", line_id);
		flush_serial_rx_buffers((USHORT)line_id);
		break;

	case AGCS_SET_BREAK:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Send break for line %d\n\r", line_id);
		send_break((USHORT)line_id);
		break;

	default:
		ag_printf(AG_MCSI_IN_PRINTF, "AG: Unsupported action requested for line %d\n\r", line_id);
		break;
	}
	return FALSE;
}

/* -- END CODE -------------------------------------------------------- */
