/* AGSPX.C -- Code relate to the SPX interface
** By: Sanjay
** Start: 11, July, 1996
** Done: 30, August, 1996
*/

/* General Notes:
** 1. We use a header here (VSPXSTR.H) that has a structure declaration
**    for an Event Control Block (ECB) that can be used for SPX packets.
**	  This struct declaration is different from the one used for IPX 
**	  purposes.	So take care not to include IPX code in this file.
*/

#include "rtrstd.h"
#include <socklib.h>

#include "kag.h"
#include "vagstr.h"
#include "agpkttyp.h"
#include "agutil.h"
#include "agdebug.h"

#include "..\..\stacks\spx\kspx.h"
#undef IPX_WAN_ENABLED
#include "..\..\stacks\ipx\kipx.h"
#define __SPX_HEADER_FILE__
#include "..\..\stacks\ipx\vipxstr.h"
#include "..\..\stacks\spx\vspxstr.h"

#include "agdebug.h"
#include "agsess.h"
#include "agresp.h"
#include "agintern.h"
#include "agspx.h"

#include "ipx.h"

/* External Prototypes */

/* Local Prototypes */
STATIC void conn_listen_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);
STATIC void ag_spx_session_rx_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);
STATIC void ag_spx_send_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);
STATIC void ag_spx_send_internal_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);
STATIC SPX_PACKET *get_a_spx_fragment_buffer(USHORT buf_size);
STATIC void free_a_spx_fragment_buffer(BYTE *sptr_spx_packet);
STATIC void ag_spx_send_for_serial_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);
STATIC void ag_spx_terminate_connection_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr);

/* Local Data */

/* -- CODE ------------------------------------------------------------- */

void do_ipx_interface_periodic_actions(void)
{
	if (ag.issue_listen == TRUE)
		wait_for_connection();
}

void wait_for_connection()
{
	SESSION_TABLE_ENTRY *sptr_session_entry;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	enum SPX_RETURN_CODE spx_call_return_code;
	SPX_PACKET *sptr_spx_packet, *sptr_disconnect_packet;
	EVENT_CONTROL_BLOCK *sptr_listen_ecb, *sptr_disconnect_ecb;
	

	ag.issue_listen = FALSE;

	ag.sptr_listen_issued_session_entry = sptr_session_entry = get_free_session_entry();
	if (sptr_session_entry == NULL)
	{
		ag.issue_listen = TRUE;
		return;
	}

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	sptr_spx_packet = sptr_ipx_specific_info->sptr_listen_packet;
	sptr_listen_ecb = sptr_ipx_specific_info->sptr_listen_ecb;

	/* Init the IPX/SPX header */
	sptr_spx_packet->ipx_header.packet_type = 5; 		/* SPX packet */

	/* Init the ECB */
	sptr_listen_ecb->fptr_event_service_routine = conn_listen_post_routine;
	sptr_listen_ecb->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	sptr_listen_ecb->fragment_count = 1;
	sptr_listen_ecb->fragment_descriptor[0].size_of_buffer = sizeof(SPX_PACKET);
	sptr_listen_ecb->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;

	/* The "terminate" ECB will also be used as a "disconnect" ECB */
	sptr_disconnect_packet = sptr_ipx_specific_info->sptr_terminate_packet;
	sptr_disconnect_ecb = sptr_ipx_specific_info->sptr_terminate_ecb;

	/* Init the "disconnect" IPX/SPX header */
	sptr_disconnect_packet->ipx_header.packet_type = 5; 		/* SPX packet */

	/* Init the "disconnect" ECB */
	sptr_disconnect_ecb->fptr_event_service_routine = NULL;
	sptr_disconnect_ecb->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	sptr_disconnect_ecb->fragment_count = 1;
	sptr_disconnect_ecb->fragment_descriptor[0].size_of_buffer = sizeof(SPX_PACKET);
	sptr_disconnect_ecb->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_disconnect_packet;

	spx_call_return_code = spx_listen_for_connection(0, TRUE, &sptr_session_entry->connection_ID, sptr_listen_ecb, sptr_disconnect_ecb);
	if (spx_call_return_code != SPX_SUCCESSFUL)
	{
#ifdef DEBUG
		printf("AG: Listen for connection failed, error code %d\n\r", spx_call_return_code);
#endif /* DEBUG */
		free_session_entry(sptr_session_entry);
		
		ag.issue_listen = TRUE;
		return;
	}
}

STATIC void conn_listen_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	SESSION_TABLE_ENTRY *sptr_session_entry;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;


	if (ecb_ptr->completion_code == SPX_OUT_OF_RESOURCES)
	{
		/* The SPX module calls the listen post with a "out of resources"
		** error as a way of informing the app that it needs receive ECB's.
		** If we don't supply any, rx-ed packets will get discarded. We want
		** it this way. So we ignore this error code. The listen will 
		** actually continue to remain pending.
		*/

		return;
	}

	if (ecb_ptr->completion_code != SPX_CONNECTION_STARTED)
	{
#ifdef DEBUG
		printf("AG: Listen posted with error!\n\r");
#endif /* DEBUG */
		free_session_entry(ag.sptr_listen_issued_session_entry);
		wait_for_connection();
		return;
	}

	ag_printf(AG_SESS_PRINTF, "AG: IPX: New session setup with client at node %04X%02X, network %04X, socket %02X\n\r", 
		ecb_ptr->destination_ipx_address.node_address._ulong,
		ecb_ptr->destination_ipx_address.node_address._ushort,
		ecb_ptr->destination_ipx_address.network,
		ecb_ptr->destination_ipx_address.socket);

	sptr_session_entry = ag.sptr_listen_issued_session_entry;
	add_entry_to_list((LINK *)&ag.session_entries_list, (LINK *)&sptr_session_entry->links);
	sptr_session_entry->session_status = AG_SESS_ACTIVE;
	ag.stats.number_of_sessions++;
	sptr_session_entry->line_in_use = UNKNOWN_LINE;

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	memcpy(&sptr_ipx_specific_info->client_ipx_address, &ecb_ptr->destination_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	sptr_ipx_specific_info->client_immediate_address = ecb_ptr->immediate_address;
	sptr_ipx_specific_info->receives_issued = 0;
	
	ag_spx_issue_receives_for_session(sptr_session_entry);

	wait_for_connection();
}

/* NOTE: Below function tries to allocate a ECB and a buffers for receive.
**		 No attempt is made to keep track of these ECB's. Only if they
**		 are posted will we get them back. If we need to do some sort of 
**		 orderly shutdown, we will be in trouble. NOTE that though the
**		 function says "receives_...", only a single receive is issued at
**		 any time. Earlier it used to issue many receives.....
*/
void ag_spx_issue_receives_for_session(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	if (sptr_ipx_specific_info->receives_issued < ag.ipx_interface.receives_per_session)
	{
#ifdef DEBUG
		ag_printf(AG_DIAGNOSTIC_PRINTF, "AG: Current received (%u)\n\r", (USHORT)sptr_ipx_specific_info->receives_issued);
#endif /* DEBUG */
		ecb_ptr = (EVENT_CONTROL_BLOCK *)mbm(sizeof(EVENT_CONTROL_BLOCK));
		if (ecb_ptr == NULL)
		{
			return;
		}
		sptr_spx_packet = get_a_spx_fragment_buffer(ag.ipx_interface.max_net_rx_buffer_size);
		if (sptr_spx_packet == NULL)
		{
			mbf((BYTE *)ecb_ptr);
			return;
		}

		ecb_ptr->in_use = TRUE;
		ecb_ptr->completion_code = 0;
		ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
		ecb_ptr->fptr_event_service_routine = ag_spx_session_rx_post_routine;
		ecb_ptr->fragment_count = 1;
		ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(MAC_HEADER) +
			sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + ag.ipx_interface.max_net_rx_buffer_size;
		ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
		ecb_ptr->session_handle = sptr_session_entry->connection_ID;

		sptr_spx_packet->ipx_header.packet_type = 5;	/* SPX packet */
		spx_listen_for_sequenced_packets(sptr_session_entry->connection_ID, ecb_ptr);

		sptr_ipx_specific_info->receives_issued++;
		sptr_ipx_specific_info->pending_calls++;
	}
#ifdef DEBUG
	else
		printf("AG: Unable to issue network receive at this time (%u)\n\r", 
			(USHORT)sptr_ipx_specific_info->receives_issued);
#endif /* DEBUG */
}

void ag_spx_issue_max_receives_for_session(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	while (sptr_ipx_specific_info->receives_issued < ag.ipx_interface.receives_per_session)
	{
#ifdef DEBUG
		ag_printf(AG_DIAGNOSTIC_PRINTF, "AG: Current received (%u)\n\r", (USHORT)sptr_ipx_specific_info->receives_issued);
#endif /* DEBUG */
		ecb_ptr = (EVENT_CONTROL_BLOCK *)mbm(sizeof(EVENT_CONTROL_BLOCK));
		if (ecb_ptr == NULL)
		{
			return;
		}
		sptr_spx_packet = get_a_spx_fragment_buffer(ag.ipx_interface.max_net_rx_buffer_size);
		if (sptr_spx_packet == NULL)
		{
			mbf((BYTE *)ecb_ptr);
			return;
		}

		ecb_ptr->in_use = TRUE;
		ecb_ptr->completion_code = 0;
		ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
		ecb_ptr->fptr_event_service_routine = ag_spx_session_rx_post_routine;
		ecb_ptr->fragment_count = 1;
		ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(MAC_HEADER) +
			sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + ag.ipx_interface.max_net_rx_buffer_size;
		ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
		ecb_ptr->session_handle = sptr_session_entry->connection_ID;

		sptr_spx_packet->ipx_header.packet_type = 5;	/* SPX packet */
		spx_listen_for_sequenced_packets(sptr_session_entry->connection_ID, ecb_ptr);

		sptr_ipx_specific_info->receives_issued++;
		sptr_ipx_specific_info->pending_calls++;
	}
}

/* In below function, note the management of the session field 
**	'receives_issued'. This field is incremented by 1 (upto a max) each
** time the ag_spx_issue_receives_for_session() function is called.
** The field is decremented by 1 in this function if either there was some
** error or the received packet was not held up in the code. If the packet
** is held up, this count is decremented only at the point where the packet
** is released and another receive is issued in this case.
*/
STATIC void ag_spx_session_rx_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	USHORT data_size;
	ULONG connection_ID;
	volatile BYTE packet_class;
	SPX_PACKET *sptr_spx_packet;
	UNION_AG_PACKET *sptr_ag_packet;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	sptr_spx_packet = ecb_ptr->fragment_descriptor[0].type.vptr_buffer;
	connection_ID = ecb_ptr->session_handle;
	sptr_session_entry = find_session_using_connection_ID(connection_ID);
	if (sptr_session_entry == NULL)
	{
#ifdef DEBUG
		printf("AG: Receive post on a non-existent session...ignoring\n\r");
#endif /* DEBUG */

		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		return;
	}

	if (ecb_ptr->completion_code == SPX_CONNECT_TERMINATED ||
			ecb_ptr->completion_code == SPX_CONNECT_ABORTED)
	{
		sptr_session_entry->session_status = AG_SESS_ABORTED;

		((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->receives_issued--;
		((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		/* Issue no further receives */
		return;
	}

	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

	sptr_ag_packet = (UNION_AG_PACKET *)&(sptr_spx_packet->data[0]);
	if (sptr_ag_packet->packet_class > AG_CONTROL_PACKET)
	{
		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->receives_issued--;

		ag_spx_issue_receives_for_session(sptr_session_entry);

		return;
	}

	/* The received ECB is now free and will be used for transmission.
	** Also the received buffer will be used for send on the network side or
	** the WAN ports.
	*/

	/* Below size includes the packet header and excludes the IPX/SPX headers */
	data_size = ecb_ptr->fragment_descriptor[0].size_of_buffer;

	/* The immediate address in the ECB is what we want it to be so leave it alone */

	sptr_spx_packet->ipx_header.destination = sptr_spx_packet->ipx_header.source;

	/* Call appropriate function via function handler array */
	/* NOTE: That we pass the ECB as a parameter. This ECB is inited 
	** partially. When the general function handlers need to send, they can
	** call a function here passing this ECB back. Also sptr_ag_packet -
	** IPX/SPX/MAC header sizes will give the full buffer.
	*/
	packet_class = sptr_ag_packet->packet_class;
	if ((*packet_function_table[packet_class])(sptr_session_entry, sptr_ag_packet, data_size, ecb_ptr) == FALSE)
	{
		/* Free the original receive buffer and ECB if the called function 
		** tells FALSE (meaning it didn't request a network send)
		*/
		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->receives_issued--;

	  	ag_spx_issue_receives_for_session(sptr_session_entry);
	}

	/* Some special handling in case the packet type is DATA_PACKET */
	if (packet_class == AG_DATA_PACKET)
	{
		/* On receiving data packets, we need to keep issuing receives to
		** maintain a constant stream of received packets.
		*/
		ag_spx_issue_max_receives_for_session(sptr_session_entry);
	}

	update_line_lan_rx_statistics(sptr_session_entry->line_in_use, data_size);
}	

void ag_spx_send_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size, void *passed_parameter)
{
	SPX_PACKET *sptr_spx_packet;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	EVENT_CONTROL_BLOCK *ecb_ptr = (EVENT_CONTROL_BLOCK *)passed_parameter;


	sptr_spx_packet = (SPX_PACKET *)(((BYTE*)sptr_ag_packet) - sizeof(SPX_HEADER) - sizeof(IPX_HEADER) - sizeof(MAC_HEADER));
	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;

	/* Init the ECB */
	ecb_ptr->in_use = TRUE;
	ecb_ptr->completion_code = 0;
	ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	memcpy(&ecb_ptr->destination_ipx_address, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	ecb_ptr->fragment_count = 1;
	ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(MAC_HEADER) +
			sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + data_size;
	ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
	ecb_ptr->fptr_event_service_routine = ag_spx_send_post_routine;
	ecb_ptr->session_handle = sptr_session_entry->connection_ID;

	/* Init the IPX/SPX header */
	sptr_spx_packet->ipx_header.packet_type = 5;
	memcpy(&sptr_spx_packet->ipx_header.destination, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	sptr_spx_packet->spx_header.connection_control._byte = 0;
	sptr_spx_packet->spx_header.datastream_type = 0;

	spx_send_sequenced_packet(sptr_session_entry->connection_ID, ecb_ptr);

	sptr_ipx_specific_info->pending_calls++;
}

STATIC void ag_spx_send_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	ULONG connection_ID;
	SPX_PACKET *sptr_spx_packet;
	enum SPX_RETURN_CODE completion_code;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	completion_code = (enum SPX_RETURN_CODE)ecb_ptr->completion_code;

	sptr_spx_packet = ecb_ptr->fragment_descriptor[0].type.vptr_buffer;
	connection_ID = ecb_ptr->session_handle;
	sptr_session_entry = find_session_using_connection_ID(connection_ID);
	if (sptr_session_entry == NULL)
	{
#ifdef DEBUG
		printf("AG: Rebound Send post on a non-existent session...ignoring\n\r");
#endif /* DEBUG */

		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		return;
	}

	/* The below count would not have been decremented when receive posts.
	** This post refers to send done in response to request packets. So at
	** this time enable issuing receives for future client packets.
	*/
	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->receives_issued--;

	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

	ag_spx_issue_receives_for_session(sptr_session_entry);

	/* Free the buffer and ECB */
	free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
	mbf((BYTE *)ecb_ptr);
	
	if (completion_code == SPX_CONNECT_ABORTED || completion_code == SPX_CONNECT_TERMINATED)
	{
		sptr_session_entry->session_status = AG_SESS_ABORTED;
		
		return;
	}
}

enum TEST ag_spx_send_internal_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size)
{
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;


	ecb_ptr = (EVENT_CONTROL_BLOCK *)mbm(sizeof(EVENT_CONTROL_BLOCK));
	if (ecb_ptr == NULL)
		return FAIL;

	sptr_spx_packet = (SPX_PACKET *)(((BYTE *)sptr_ag_packet) - sizeof(SPX_HEADER) - sizeof(IPX_HEADER) - sizeof(MAC_HEADER));
	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;

	/* Init the ECB */
	ecb_ptr->in_use = TRUE;
	ecb_ptr->completion_code = 0;
	ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	memcpy(&ecb_ptr->destination_ipx_address, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	ecb_ptr->fragment_count = 1;
	ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(MAC_HEADER) +
			sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + data_size;
	ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
	ecb_ptr->fptr_event_service_routine = ag_spx_send_internal_post_routine;
	ecb_ptr->session_handle = sptr_session_entry->connection_ID;

	/* Init the IPX/SPX header */
	sptr_spx_packet->ipx_header.packet_type = 5;
	memcpy(&sptr_spx_packet->ipx_header.destination, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	sptr_spx_packet->spx_header.connection_control._byte = 0;
	sptr_spx_packet->spx_header.datastream_type = 0;

	spx_send_sequenced_packet(sptr_session_entry->connection_ID, ecb_ptr);

	sptr_ipx_specific_info->pending_calls++;

	return PASS;
}

STATIC void ag_spx_send_internal_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	ULONG connection_ID;
	SPX_PACKET *sptr_spx_packet;
	UNION_AG_PACKET *sptr_ag_packet;
	enum SPX_RETURN_CODE completion_code;
	SESSION_TABLE_ENTRY *sptr_session_entry;
	
	
	completion_code = (enum SPX_RETURN_CODE)ecb_ptr->completion_code;

	sptr_spx_packet = ecb_ptr->fragment_descriptor[0].type.vptr_buffer;
	connection_ID = ecb_ptr->session_handle;
	sptr_session_entry = find_session_using_connection_ID(connection_ID);
	if (sptr_session_entry == NULL)
	{
#ifdef DEBUG
		printf("AG: Rebound Send post on a non-existent session...ignoring\n\r");
#endif /* DEBUG */

		free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
		mbf((BYTE *)ecb_ptr);

		return;
	}

	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

	mbf((BYTE *)ecb_ptr);

	sptr_ag_packet = (UNION_AG_PACKET *)(((BYTE *)sptr_spx_packet) + sizeof(SPX_HEADER) + sizeof(IPX_HEADER) + sizeof(MAC_HEADER));
	internal_sends_callback(sptr_session_entry, sptr_ag_packet);
}

STATIC SPX_PACKET *get_a_spx_fragment_buffer(USHORT buf_size)
{
	BYTE *buf_ptr;
	USHORT actual_buf_size;

	actual_buf_size = buf_size + sizeof(SPX_HEADER) + sizeof(IPX_HEADER) + sizeof(MAC_HEADER) + 64;
	buf_ptr = (BYTE *)mbm(actual_buf_size);

	if (buf_ptr)
		buf_ptr += 64;
	return (SPX_PACKET *)buf_ptr;
}

BYTE *get_a_spx_send_buffer(USHORT data_size)
{
	BYTE *bptr;

	bptr = (BYTE *)get_a_spx_fragment_buffer(data_size);
	if (bptr == NULL)
		return NULL;

	return (bptr + sizeof(MAC_HEADER) + sizeof(IPX_HEADER) + sizeof(SPX_HEADER));
}

STATIC void free_a_spx_fragment_buffer(BYTE *bptr)
{
	if (bptr == NULL)
	{
		ag_printf(AG_ALARM_PRINTF, "AG: NULL pointer request to free\n\r");
		return;
	}
	bptr -= 64;
	mbf(bptr);
}

void free_a_spx_send_buffer(BYTE *bptr)
{
	free_a_spx_fragment_buffer(bptr - (sizeof(MAC_HEADER) + sizeof(IPX_HEADER) + sizeof(SPX_HEADER)));
}

void ag_spx_reduce_network_receives(SESSION_TABLE_ENTRY *sptr_session_entry)
{
#if 0
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	
	/* Fake that a receive has been issued. So this will try to do 
	** some flow control by not issuing a receive on the session.
	*/
	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	if (sptr_ipx_specific_info->receives_issued < ag.ipx_interface.receives_per_session)
		sptr_ipx_specific_info->receives_issued++;
#endif /* 0 */
	return;
}

void ag_spx_increase_network_receives(SESSION_TABLE_ENTRY *sptr_session_entry)
{
#if 0
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	
	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	if (sptr_ipx_specific_info->receives_issued > 0)
		sptr_ipx_specific_info->receives_issued--;
#endif /* 0 */
	return;
}


void ag_spx_free_network_stuff(UNION_AG_PACKET *sptr_ag_packet)
{
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;

	ecb_ptr = (EVENT_CONTROL_BLOCK *)(*(void **)(((BYTE *)sptr_ag_packet) - sizeof(void *)));
	sptr_spx_packet = (SPX_PACKET *)(((BYTE *)sptr_ag_packet) - sizeof(SPX_HEADER) - sizeof(IPX_HEADER) - sizeof(MAC_HEADER));
	free_a_spx_fragment_buffer((BYTE *)sptr_spx_packet);
	mbf((BYTE *)ecb_ptr);
}

/* Below function to be called only when a serially received packet has
** to be sent onto the network.
*/
enum TEST ag_spx_send_serial_rx_packet_on_network (SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size)
{
	ULONG connection_ID;
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;

	sptr_spx_packet = (SPX_PACKET *)(((BYTE *)sptr_ag_packet) - sizeof(IPX_HEADER) - sizeof(SPX_HEADER) - sizeof(MAC_HEADER));
	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	connection_ID = sptr_session_entry->connection_ID;
	ecb_ptr = (EVENT_CONTROL_BLOCK *)mbm(sizeof(EVENT_CONTROL_BLOCK));
	if (ecb_ptr == NULL)
	{
		ag_printf(AG_ALARM_PRINTF, "AG: Insufficient memory to allocate ECB for network send of WAN packet\n\r");

		return FAIL;
	}

	/* Fill in the IPX/SPX header */
	sptr_spx_packet->ipx_header.packet_type = 5;
	memcpy(&sptr_spx_packet->ipx_header.destination, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	sptr_spx_packet->spx_header.connection_control._byte = 0;
	sptr_spx_packet->spx_header.datastream_type = 0;

	/* Fill in the ECB */
	ecb_ptr->in_use = TRUE;
	ecb_ptr->completion_code = 0;
	ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	memcpy(&ecb_ptr->destination_ipx_address, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
	ecb_ptr->fptr_event_service_routine = ag_spx_send_for_serial_post_routine;
	ecb_ptr->fragment_count = 1;
	ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(MAC_HEADER) +
			sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + data_size;
	ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
	ecb_ptr->session_handle = connection_ID;

	spx_send_sequenced_packet(connection_ID, ecb_ptr);

	sptr_ipx_specific_info->pending_calls++;
	
	return PASS;
}

STATIC void ag_spx_send_for_serial_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	USHORT send_size;
	ULONG connection_ID;
	BYTE *sptr_rx_buffer;	
	SPX_PACKET *sptr_spx_packet;
	enum SPX_RETURN_CODE completion_code;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	completion_code = (enum SPX_RETURN_CODE)ecb_ptr->completion_code;
	send_size = ecb_ptr->fragment_descriptor[0].size_of_buffer 
			- sizeof(MAC_HEADER) - sizeof(IPX_HEADER) - sizeof(SPX_HEADER);

	sptr_spx_packet = ecb_ptr->fragment_descriptor[0].type.vptr_buffer;
	connection_ID = ecb_ptr->session_handle;
	sptr_session_entry = find_session_using_connection_ID(connection_ID);
	if (sptr_session_entry == NULL)
	{
#ifdef DEBUG
		printf("AG: Serial Send post on a non-existent session...ignoring\n\r");
#endif /* DEBUG */

		return;
	}
	

	/* Return the buffer back to the serial driver */

	sptr_rx_buffer = (BYTE *)(((BYTE *)sptr_spx_packet) + sizeof(IPX_HEADER) + sizeof(SPX_HEADER) + sizeof(MAC_HEADER) + sizeof(DATA_TYPE) - sizeof(BYTE) + sizeof(BYTE));
	ag.fptr_rx_complete((USHORT)sptr_session_entry->line_in_use, sptr_rx_buffer);

	mbf((BYTE *)ecb_ptr);

	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

	if (completion_code == SPX_CONNECT_TERMINATED || completion_code == SPX_CONNECT_ABORTED)
	{
		sptr_session_entry->session_status = AG_SESS_ABORTED;
		
		return;
	}

	update_line_lan_tx_statistics(sptr_session_entry->line_in_use, send_size);
}


void ag_spx_terminate_connection(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	SPX_PACKET *sptr_spx_packet;
	EVENT_CONTROL_BLOCK *ecb_ptr;
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	ecb_ptr = sptr_ipx_specific_info->sptr_terminate_ecb;
	sptr_spx_packet = sptr_ipx_specific_info->sptr_terminate_packet;

	/* Fill in the IPX/SPX header */
	sptr_spx_packet->ipx_header.packet_type = 5;
	sptr_spx_packet->spx_header.connection_control._byte = 0;
	sptr_spx_packet->spx_header.datastream_type = 0xFE;
	memcpy(&sptr_spx_packet->ipx_header.destination, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));

	/* Initialize the ECB */
	ecb_ptr->in_use = TRUE;
	ecb_ptr->completion_code = 0;
	ecb_ptr->socket_number = swap(AG_IPX_SERVICE_SOCKET);
	ecb_ptr->fptr_event_service_routine = ag_spx_terminate_connection_post_routine;
	ecb_ptr->fragment_count = 1;
	ecb_ptr->fragment_descriptor[0].size_of_buffer = sizeof(SPX_PACKET);
	ecb_ptr->fragment_descriptor[0].type.vptr_buffer = (void *)sptr_spx_packet;
	ecb_ptr->immediate_address = sptr_ipx_specific_info->client_immediate_address;
	ecb_ptr->session_handle = sptr_session_entry->connection_ID;
	memcpy(&ecb_ptr->destination_ipx_address, &sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));

	spx_terminate_connection(sptr_session_entry->connection_ID, ecb_ptr);

	sptr_ipx_specific_info->pending_calls++;
}

STATIC void ag_spx_terminate_connection_post_routine(EVENT_CONTROL_BLOCK *ecb_ptr)
{
	ULONG connection_ID;
	SPX_PACKET *sptr_spx_packet;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	sptr_spx_packet = ecb_ptr->fragment_descriptor[0].type.vptr_buffer;
	connection_ID = ecb_ptr->session_handle;
	sptr_session_entry = find_session_using_connection_ID(connection_ID);
	if (sptr_session_entry == NULL)
	{
#ifdef DEBUG
		printf("AG: Terminate post on a non-existent session...ignoring\n\r");
#endif /* DEBUG */

		return;
	}

	((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls--;

	sptr_session_entry->session_status = AG_SESS_ABORTED;
}

enum BOOLEAN ag_spx_says_ok_to_cleanup(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	if (((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls == 0)
		return TRUE;
	return FALSE;
}

enum BOOLEAN ag_spx_says_cancels_done(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	if (((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->pending_calls <= 
		ag.ipx_interface.receives_per_session)
		return TRUE;
	return FALSE;
}

void ag_spx_cancel_pending_calls(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	/* ?? For now say done as usually when this is called, no call will
	** ?? be pending. 
	*/

	return;
}

enum TEST ag_spx_allocate_protocol_structure(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;
	void *sptr_listen_ecb, *sptr_terminate_ecb;
	void *sptr_listen_packet, *sptr_terminate_packet;


	sptr_ipx_specific_info = (IPX_SPECIFIC_INFO *)mbm(sizeof(IPX_SPECIFIC_INFO));
	sptr_listen_ecb = (void *)mbm(sizeof(EVENT_CONTROL_BLOCK));
	sptr_terminate_ecb = (void *)mbm(sizeof(EVENT_CONTROL_BLOCK));
	sptr_listen_packet = (void *)get_a_spx_fragment_buffer(1);
	sptr_terminate_packet = (void *)get_a_spx_fragment_buffer(1);

	if (sptr_ipx_specific_info == NULL || sptr_listen_ecb == NULL ||
		sptr_terminate_ecb == NULL || sptr_listen_packet == NULL ||
		sptr_terminate_packet == NULL)
	{
		if (sptr_ipx_specific_info != NULL) mbf((BYTE *)sptr_ipx_specific_info);
		if (sptr_listen_ecb != NULL) mbf(sptr_listen_ecb);
		if (sptr_terminate_ecb != NULL) mbf(sptr_terminate_ecb);
		if (sptr_listen_packet != NULL) free_a_spx_fragment_buffer(sptr_listen_packet);
		if (sptr_terminate_packet != NULL) free_a_spx_fragment_buffer(sptr_terminate_packet);

		return FAIL;
	}
	
	memset(sptr_ipx_specific_info, 0, sizeof(IPX_SPECIFIC_INFO));
	sptr_ipx_specific_info->sptr_listen_ecb = sptr_listen_ecb;
	sptr_ipx_specific_info->sptr_terminate_ecb = sptr_terminate_ecb;
	sptr_ipx_specific_info->sptr_listen_packet = sptr_listen_packet;
	sptr_ipx_specific_info->sptr_terminate_packet = sptr_terminate_packet;
	sptr_session_entry->protocol_specific_info = sptr_ipx_specific_info;

	return PASS;
}

void ag_spx_free_protocol_structure(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;

	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;

	free_a_spx_fragment_buffer((BYTE *)sptr_ipx_specific_info->sptr_terminate_packet);
	free_a_spx_fragment_buffer((BYTE *)sptr_ipx_specific_info->sptr_listen_packet);
	mbf((void *)sptr_ipx_specific_info->sptr_terminate_ecb);
	mbf((void *)sptr_ipx_specific_info->sptr_listen_ecb);
	mbf((BYTE *)sptr_ipx_specific_info);
}

ULONG ag_spx_get_client_network_number(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	return ((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->client_ipx_address.network;
}
ULONG ag_spx_get_client_node_number_ulong(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	return ((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->client_ipx_address.node_address._ulong;
}
USHORT ag_spx_get_client_node_number_ushort(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	return ((IPX_SPECIFIC_INFO *)(sptr_session_entry->protocol_specific_info))->client_ipx_address.node_address._ushort;
}

void ag_spx_fill_user_address(SESSION_TABLE_ENTRY *sptr_session_entry, BYTE *sptr_address)
{
	IPX_SPECIFIC_INFO *sptr_ipx_specific_info;


	sptr_ipx_specific_info = sptr_session_entry->protocol_specific_info;
	memcpy(sptr_address, (BYTE *)&sptr_ipx_specific_info->client_ipx_address, sizeof(IPX_ADDRESS_DEFINED));
}

/* -- END CODE -------------------------------------------------------- */
