/*--------------------------------------------------------------------------
	File         : stmc.c
	Synopsis     : The TFTP operation is organised as a finite state machine.
		            The handlers for each state that a connect descriptor
						can be in are included in this file. From the foreground,
						all that is required	to be done is call the StateHandler
						for each descriptor (there are MAX_CONNECTIONS of them)
						that is in use. The handlers take care of	processing in
						that state and moving the descriptor to the next state
	Date         : 21.8.95
	Edit plan    :
		1. Look at the length of each packet received and discard packets with
		   bad lengths.
		2. Translate the fopen(), fclose(), fread(), fwrite()... to suit our
		   environment.
		3. Everytime TFTPServer() is called, decide after how long it has been
		   called using the interval marker and record it in a variable. Decrement
		   the timers by that value.
		4. In WaitForDataHandler() and WaitForAckHandler() discard packets
		   even if they dont come from the same protocol port.
	Edit history :
		1. (25.08.1995) If the no. of bytes read in an fread() is 0, send
		   the 0-length packet otherwise the receiver, having received a
			512-byte packet, keeps waiting for more data. This change is made in
			SendDataStateHandler().
---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include "tftp.h"

int nop_state_handler(int descriptor);
int listen_state_handler(int descriptor);
int send_data_state_handler(int descriptor);
int wait_for_ack_state_handler(int descriptor);
int retransmit_state_handler(int descriptor);
int wait_for_data_state_handler(int descriptor);
int send_ack_state_handler(int descriptor);
int abort_connection_state_handler(int descriptor);
int close_socket_state_handler(int descriptor);

/* the following is to support hangup and redial when roucon is over IP
connection */
extern	void	ppp_hangup_and_redial_wan_port(int port_number);

extern char *	strstr(const char *, const char *);
extern char *	strchr(const char *, int);

/* Jo 27/04/99 */
const int (*tftp_state_machine_handlers[])(int descriptor) =
{
	nop_state_handler,         /* descriptors that are not in use           */
	listen_state_handler,      /* descriptor corresponding to socket 69     */
	send_data_state_handler,    /* Desc. corresponding to a connection
									    created in response to a GET_FILE request */
	wait_for_ack_state_handler,  /* The above Desc., waiting for an ACK       */
	retransmit_state_handler,  /* The Desc., after it has been in the above
									    state for some time(RETRANSMISSION_PERIOD)*/
	wait_for_data_state_handler, /* Desc. corresponding to a connection
									 	 created in response to a PUT_FILE request */
	send_ack_state_handler,     /* The above Desc., after it gets data       */
	close_socket_state_handler, /* A Desc., after it has completed the file 
									    transfer or aborted the connection because
									 	 of data or ack not received for some time */
	abort_connection_state_handler,   /* A Desc. corresponding to a connection
									    which has not received data/ack for a
									 	 certain number of retries                 */
} ;

#ifdef TRACE /* Jo 25/05/99 */
const char Msg1[] = "Error packet sent to client" ;
const char Msg3[] = "\nTFTP : No traffic, connection aborted after " ;
#endif /* Jo 25/05/99 */
const char Msg2[] = "NETASCII mode not supported" ;
									  

/*---------------------------------------------------------------------------
	Function : nop_state_handler
	Synopsis : corresponds to a state where the descriptor is not in use.
	           This is called only for descriptors that are not in use.
---------------------------------------------------------------------------*/
int nop_state_handler(int descriptor)
{
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : listen_state_handler
	Synopsis : called with the descriptor which corresponds to the well known
	           socket for TFTP/UDP (69). If there is no request, it returns.
				  Otherwise it opens a new socket, binds an arbitrary port to it
				  and does the following :
			. Allocates a new connection info. block and inits it.
			. If there is an error in opening the file, sends an error packet.
			. If the request is a GET_FILE request, puts the descriptor in
			  SEND_DATA state and returns.
			. If the request is a PUT_FILE request and there is write-permission
			  for the file, it sends ack #0 and puts the descriptor in the
			  WAIT_FOR_DATA state.
---------------------------------------------------------------------------*/
int listen_state_handler(int descriptor)
{
	ULONG client_ip_address ;
	int i, new_desc, return_value, client_port, length ;
	TFTP_PACKET *tftp_packet ;
	SOCKADDR_IN my_ip_address ;
	BYTE *file_name, *open_mode ;
	FILE_HANDLE *file_pointer ;
#ifdef TRACE
	unsigned char *temp_ptr ;
	int c1, c2, c3, c4 ;
#endif

	return_value = receive_data(&client_ip_address, &client_port,
	                     connect_info_table[descriptor].socket,
								connect_info_table[descriptor].received_packet, &length) ;
	if (return_value != 0)
	{
		return NO_REQUEST ;
		/* No client has sought connection */
	}
	tftp_packet = (TFTP_PACKET *)connect_info_table[descriptor].received_packet ;
	connect_info_table[descriptor].connection_abort_time = tftp_connection_abort_time ;
	switch (net_to_host_short(tftp_packet->opcode))
	{
		case (OPCODE_READ) :
		{
			new_desc = get_connect_descriptor() ;
			if (new_desc == -1)
			{
				return CONNECTION_TABLE_FULL ;
				/* No more than MAX_CONNECTIONS clients are supported */
			}
			connect_info_table[new_desc].socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP) ;
			if (connect_info_table[new_desc].socket == -1)
			{
				free_connect_descriptor(new_desc) ;
				send_error_packet(client_ip_address, client_port,
				                  connect_info_table[descriptor].socket,
										"TFTP Server Error : Socket Table Full", ERR_UNDEFINED) ;
				return CANNOT_OPEN_SOCKET ; /* cannot open any more sockets */
			}

			associate_application_with_socket (connect_info_table[new_desc].socket,
			                                 tftp.application_id) ;

			my_ip_address.sin_family = AF_INET ;
			my_ip_address.sin_port = get_free_port() ;
			my_ip_address.sin_addr.s_addr = my_node_ip_address ;
			return_value = bind(connect_info_table[new_desc].socket,
			     (struct sockaddr *)&my_ip_address, sizeof(SOCKADDR_IN)) ;
			if (return_value == -1)
			{
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return BIND_FAILED ;
			}
			file_name = tftp_packet->message_type.read_write_request.file_name_and_mode ;
			i = 0 ;
			while (file_name[i]) /* convert to uppercase */
			{
				if ((file_name[i] >= 'a') && (file_name[i] <= 'z'))
					file_name[i] -= ('a' - 'A') ;
				i++ ;
			}

			open_mode = file_name ;
			while (*open_mode++) ;
			i = 0 ;
			while (open_mode[i])
			{
				if ((open_mode[i] >= 'a') && (open_mode[i] <= 'z'))
					open_mode[i] -= ('a' - 'A') ;
				i++ ;
			}

#ifdef TRACE
			temp_ptr = (unsigned char *)&client_ip_address ;
			c1 = *(temp_ptr+0) ;
			c2 = *(temp_ptr+1) ;
			c3 = *(temp_ptr+2) ;
			c4 = *(temp_ptr+3) ;
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : GET_FILE request from %u.%u.%u.%u, port : %d\n\t\tFile : %s\n\t\tMode : %s\r",
			        c1, c2, c3, c4,
					  net_to_host_short(client_port), file_name, open_mode) ;
#endif
			if (strcmp(open_mode, "OCTET"))
			{
				send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket, Msg2,
				                  ERR_ACCESS_VIOLATION) ;
#ifdef TRACE
				tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return FILE_OPEN_ERROR ;
			}

			file_pointer = tftp_open(file_name, OPEN_FOR_READ) ;
			if (file_pointer == NULL)
			{
				send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket, "File not found", ERR_NO_FILE) ;
#ifdef TRACE
				tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return FILE_OPEN_ERROR ;
			}
			connect_info_table[new_desc].send_packet = (BYTE *) malloc(MAX_TFTP_PACKET_SIZE) ;
			connect_info_table[new_desc].received_packet = (BYTE *) malloc(MAX_TFTP_PACKET_SIZE) ;
			if ((connect_info_table[new_desc].send_packet == NULL) ||
			    (connect_info_table[new_desc].received_packet == NULL))
			{
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return NO_MEMORY ;
			}
			connect_info_table[new_desc].current_state = STATE_SEND_DATA ;
			connect_info_table[new_desc].operation = OPCODE_READ ;
			connect_info_table[new_desc].peer_protocol_port = client_port ;
			connect_info_table[new_desc].peer_ip_address = client_ip_address ;
			connect_info_table[new_desc].sequence_number = 1 ;
			connect_info_table[new_desc].file_pointer = file_pointer ;
			connect_info_table[new_desc].smoothed_rtt = (tftp.timeout_period * tftp.timer_class.clock_ticks_per_second) ;
			connect_info_table[new_desc].retransmission_timer = connect_info_table[new_desc].smoothed_rtt ;
			connect_info_table[new_desc].smoothed_rtt *= 2 ;
			connect_info_table[new_desc].retry_count = 0 ;
			connect_info_table[new_desc].connection_abort_time = tftp_connection_abort_time ;
			return 0 ;
		} /* of OPCODE_READ request */


		case (OPCODE_WRITE) :
		{
			/* vidy moved the filename parsing code to this place. this is
			to support the command file name "HUPANDREDIAL" and mode "PORT" */

			file_name = tftp_packet->message_type.read_write_request.file_name_and_mode ;
			i = 0 ;
			while (file_name[i])
			{
				if ((file_name[i] >= 'a') && (file_name[i] <= 'z'))
					file_name[i] -= ('a' - 'A') ;
				i++ ;
			}

			open_mode = file_name ;
			i = 0 ;
			while (*open_mode++) ;
			while (open_mode[i])
			{
				if ((open_mode[i] >= 'a') && (open_mode[i] <= 'z'))
					open_mode[i] -= ('a' - 'A') ;
				i++ ;
			}

			/* hangup nad redial command comes as "HUPANDREDIAL_x" where x
			is a single digit decimal port number */
#if 1
/*			printf ("TFTP : File Name received is %s\n",file_name); */
			if (strstr(file_name, "HUPANDREDIAL"))
			{
/*				printf ("TFTP : Hupand redial string received\n"); */
				open_mode = strchr(file_name,'_');
				if (open_mode == NULL)
				{
					printf ("TFTP: Wrong HUP port option\n");
					return 0;
				}

				open_mode += 1;		/* point to the port number digit */
				printf ("TFTP : Hanging up port %d\n", (*open_mode- '0'));
				ppp_hangup_and_redial_wan_port(*open_mode - '0');
				return 0;
			}
#endif

			new_desc = get_connect_descriptor() ;
			if (new_desc == -1)
			{
				return CONNECTION_TABLE_FULL ;
				/* No more than MAX_CONNECTIONS clients are supported */
			}
			connect_info_table[new_desc].socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP) ;
			if (connect_info_table[new_desc].socket == -1)
			{
				free_connect_descriptor(new_desc) ;
				send_error_packet(client_ip_address, client_port,
				                  connect_info_table[descriptor].socket,
										"TFTP Server Error : Socket Table Full", ERR_UNDEFINED) ;
				return CANNOT_OPEN_SOCKET ; /* cannot open any more sockets */
			}

			associate_application_with_socket (connect_info_table[new_desc].socket,
			                                 tftp.application_id) ;

			my_ip_address.sin_family = AF_INET ;
			my_ip_address.sin_port = get_free_port() ;
			my_ip_address.sin_addr.s_addr = my_node_ip_address ;
			return_value = bind(connect_info_table[new_desc].socket,
			     (struct sockaddr *)&my_ip_address, sizeof(SOCKADDR_IN)) ;
			if (return_value == -1)
			{
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return BIND_FAILED ;
			}

#ifdef TRACE
			temp_ptr = (unsigned char *)&client_ip_address ;
			c1 = *(temp_ptr+0) ;
			c2 = *(temp_ptr+1) ;
			c3 = *(temp_ptr+2) ;
			c4 = *(temp_ptr+3) ;
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : PUT_FILE request from %u.%u.%u.%u, port : %d \n\t\tFile : %s\n\t\tMode : %s\r",
			        c1, c2, c3, c4,
					  net_to_host_short(client_port), file_name, open_mode) ;
#endif

			if (strcmp(open_mode, "OCTET"))
			{
				send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket, Msg2,
				                  ERR_ACCESS_VIOLATION) ;
#ifdef TRACE
				tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return FILE_OPEN_ERROR ;
			}

			file_pointer = tftp_open(file_name, OPEN_FOR_WRITE) ;
			if (file_pointer == NULL)
			{
				switch (tftp_fileio_error_code)
				{
					case (FILE_NOT_FOUND) :
#ifdef TRACE
                  tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
						send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket, "File not found", ERR_NO_FILE) ;
						break ;

					case (WRITE_ACCESS_DENIED) :
#ifdef TRACE
                  tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
						send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket, 
                                    "TFTP Server Error : File locked for writing",
						ERR_UNDEFINED) ;
						break ;

					case (HANDLE_TABLE_FULL) :
#ifdef TRACE
                  tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %s\r", Msg1) ;
#endif
						send_error_packet(client_ip_address, client_port, connect_info_table[new_desc].socket,
                                    "TFTP Server Error : Handle table full", ERR_UNDEFINED) ;
						break ;

					default :
						break ;

				}
				
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return FILE_OPEN_ERROR ;
			}
			connect_info_table[new_desc].received_packet = (BYTE *) malloc(MAX_TFTP_PACKET_SIZE) ;
			connect_info_table[new_desc].send_packet = (BYTE *) malloc(MAX_TFTP_PACKET_SIZE) ;
			if ((connect_info_table[new_desc].received_packet == NULL) ||
			    (connect_info_table[new_desc].send_packet == NULL))
			{
				tftp_close(file_pointer) ;
				closesocket(connect_info_table[new_desc].socket) ;
				free_connect_descriptor(new_desc) ;
				return NO_MEMORY ;
			}
			connect_info_table[new_desc].current_state = STATE_SEND_ACK ;
			connect_info_table[new_desc].operation = OPCODE_WRITE ;
			connect_info_table[new_desc].peer_protocol_port = client_port ;
			connect_info_table[new_desc].peer_ip_address = client_ip_address ;
			connect_info_table[new_desc].sequence_number = 0 ;
			connect_info_table[new_desc].file_pointer = file_pointer ;
			connect_info_table[new_desc].smoothed_rtt = (tftp.timeout_period * tftp.timer_class.clock_ticks_per_second) ;
			connect_info_table[new_desc].retransmission_timer = connect_info_table[new_desc].smoothed_rtt ;
			connect_info_table[new_desc].smoothed_rtt *= 2 ;
			connect_info_table[new_desc].retry_count = 0 ;
			connect_info_table[new_desc].connection_abort_time = tftp_connection_abort_time ;
			return 0 ;
		} /* of OPCODE_WRITE request */

		default :
			return INVAID_OPCODE ;
			/* actually it is required to send an error packet with code
			   ERR_ILLEGAL_TFTP_OP. That means opening a socket, sending
				an error packet and closing the socket */
	}
}




/*---------------------------------------------------------------------------
	Function : send_data_state_handler
	Synopsis : Reads the next block of data from the file in to a "hold on"
	           buffer (for the purpose of retransmission), send it to the
				  client, update the descriptor feilds and put the descriptor in
				  STATE_WAIT_FOR_ACK state
---------------------------------------------------------------------------*/
int send_data_state_handler(int descriptor)
{
	int num_bytes_read ;
#ifdef TRACE
	unsigned char *temp_ptr = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1 = *temp_ptr, c2 = *(temp_ptr+1), c3 = *(temp_ptr+2), c4 = *(temp_ptr+3) ;
#endif

	if (connect_info_table[descriptor].connection_abort_time == 0)
	{
		connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "%s%d seconds.....\r", Msg3, tftp_connection_abort_time / tftp.timer_class.clock_ticks_per_second) ;
#endif
		return CONNECTION_BROKEN ;
	}

	num_bytes_read = tftp_read(((TFTP_PACKET *)connect_info_table[descriptor].send_packet)->message_type.read_write_data.data,
	                     TFTP_BLOCK_SIZE, connect_info_table[descriptor].file_pointer) ;
/*	if (num_bytes_read == 0)
	{
		The whole file has been read 
		connect_info_table[descriptor].current_state = STATE_CLOSE_SOCKET ;
		return FT_COMPLETE ;
	}*/
	((TFTP_PACKET *)connect_info_table[descriptor].send_packet)->message_type.read_write_data.data_block_number = 
	   host_to_net_short(connect_info_table[descriptor].sequence_number) ;
	((TFTP_PACKET *)connect_info_table[descriptor].send_packet)->opcode = host_to_net_short(OPCODE_DATA) ;

	connect_info_table[descriptor].packet_length = num_bytes_read + 2+2;
	connect_info_table[descriptor].smoothed_rtt = (tftp.timeout_period * tftp.timer_class.clock_ticks_per_second) ;
	connect_info_table[descriptor].retransmission_timer = connect_info_table[descriptor].smoothed_rtt ;
	connect_info_table[descriptor].smoothed_rtt *= 2 ;
	connect_info_table[descriptor].retry_count = 0 ;
	send_data_packet(connect_info_table[descriptor].peer_ip_address,
	               connect_info_table[descriptor].peer_protocol_port,
						connect_info_table[descriptor].socket,
						connect_info_table[descriptor].send_packet,
						connect_info_table[descriptor].packet_length) ;
#ifdef TRACE
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : Block #%d, %d bytes sent to %u.%u.%u.%u\r", connect_info_table[descriptor].sequence_number,
	        connect_info_table[descriptor].packet_length-2-2, c1, c2, c3, c4) ;
#endif
	connect_info_table[descriptor].current_state = STATE_WAIT_FOR_ACK ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : wait_for_ack_state_handler
	Synopsis : It is called with those descriptors that are waiting for a
	           positive ACK, having sent a data packet. If an ACK is received,
				  the descriptor is driven to the SEND_DATA state. Otherwise
				  the Retransmission counter is decremented and if it has died
				  out, the descriptor is driven to the RETRANSMIT_STATE.
	Note     : If an ACK is received and the length < TFTP_PAKET_LENGTH,
	           the descriptor can be sent straight to the CLOSE_SOCKET state.
---------------------------------------------------------------------------*/
int wait_for_ack_state_handler(int descriptor)
{
	int return_value, client_port, packet_length ;
	ULONG client_ip_address ;
	TFTP_PACKET *tftp_packet ;
#ifdef TRACE
	unsigned char *temp_ptr = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1 = *temp_ptr, c2 = *(temp_ptr+1), c3 = *(temp_ptr+2), c4 = *(temp_ptr+3) ;
#endif

	if (connect_info_table[descriptor].connection_abort_time == 0)
	{
		connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "%s%d seconds.....\r", Msg3, tftp_connection_abort_time / tftp.timer_class.clock_ticks_per_second) ;
#endif
		return CONNECTION_BROKEN ;
	}

	if (connect_info_table[descriptor].retransmission_timer == 0)
	{
		connect_info_table[descriptor].retry_count++ ;
		if (connect_info_table[descriptor].retry_count >= tftp.number_of_retransmissions+1)
		{
			connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
#ifdef TRACE
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : %d retries made...\r", tftp.number_of_retransmissions) ;
#endif
			return CONNECTION_BROKEN ;
		}
		connect_info_table[descriptor].current_state = STATE_RETRANSMIT ;
		return RETRANSMIT_PACKET ;
	}

	return_value = receive_data(&client_ip_address, &client_port,
	              connect_info_table[descriptor].socket,
	              connect_info_table[descriptor].received_packet, &packet_length) ;
	if (return_value)
	{
		return NO_CHANGE ;
	}

	/* ACK received */
	connect_info_table[descriptor].connection_abort_time = tftp_connection_abort_time ;
	tftp_packet = (TFTP_PACKET *)connect_info_table[descriptor].received_packet ;
	if ((net_to_host_short(tftp_packet->opcode) != (OPCODE_ACK)) ||
	    (net_to_host_short(tftp_packet->message_type.ack.acked_block_number) != connect_info_table[descriptor].sequence_number))
	{
		return NO_CHANGE ;
	}
#ifdef TRACE
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : ACK #%d received from %u.%u.%u.%u\r",
	            connect_info_table[descriptor].sequence_number, c1, c2, c3, c4) ;
#endif
	if ((connect_info_table[descriptor].sequence_number != 0) &&
	    (connect_info_table[descriptor].packet_length < TFTP_BLOCK_SIZE+2+2))
	{
		connect_info_table[descriptor].current_state = STATE_CLOSE_SOCKET ;
		tftp_close(connect_info_table[descriptor].file_pointer) ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : File transfer complete\r") ;
#endif
		return 0 ;
	}
	connect_info_table[descriptor].sequence_number++ ;
	connect_info_table[descriptor].current_state = STATE_SEND_DATA ;
	connect_info_table[descriptor].connection_abort_time = tftp_connection_abort_time ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : retransmit_state_handler
	Synopsis : The descriptor moves to this state after it has stayed in the
	           WAITING_FOR_ACK state for RETRANSMISSION_PERIOD amount of time.
				  All this is expected to do is send a block of data to the client
	           the way send_data_state_handler() does except that
				  (a). the block is not read from the file but from the "hold on"
				       buffer
				  (b). the retry count feild of the descriptor block is not touched
				  Having done this it moves the desc. to the WAIT_FOR_ACK state.
---------------------------------------------------------------------------*/
int retransmit_state_handler(int descriptor)
{

#ifdef TRACE
	unsigned char *temp_ptr = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1 = *temp_ptr, c2 = *(temp_ptr+1), c3 = *(temp_ptr+2), c4 = *(temp_ptr+3) ;
#endif

	if (connect_info_table[descriptor].connection_abort_time == 0)
	{
		connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "%s%d seconds.....\r", Msg3, tftp_connection_abort_time / tftp.timer_class.clock_ticks_per_second) ;
#endif
		return CONNECTION_BROKEN ;
	}

	connect_info_table[descriptor].retransmission_timer = connect_info_table[descriptor].smoothed_rtt ;
	connect_info_table[descriptor].smoothed_rtt *= 2 ;
	send_data_packet(connect_info_table[descriptor].peer_ip_address,
	               connect_info_table[descriptor].peer_protocol_port,
						connect_info_table[descriptor].socket,
						connect_info_table[descriptor].send_packet,
						connect_info_table[descriptor].packet_length) ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : Block #%d, %d bytes re-tx.ed to %u.%u.%u.%u after %d secs [Retry #%d]\r",
		     connect_info_table[descriptor].sequence_number, connect_info_table[descriptor].packet_length-2-2,
			  c1, c2, c3, c4, connect_info_table[descriptor].retransmission_timer / (2 * tftp.timer_class.clock_ticks_per_second),
			  connect_info_table[descriptor].retry_count) ;
#endif
	connect_info_table[descriptor].current_state = STATE_WAIT_FOR_ACK ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : wait_for_data_state_handler
	Synopsis : This is related to the PUT_FILE request from the client.
	           It looks for the next block of data from the client. If the
				  block arrives and arrives in sequence, it is written to the
	           file and an ACK is sent with the correct sequence_number.
---------------------------------------------------------------------------*/
int wait_for_data_state_handler(int descriptor)
{
	ULONG client_ip_address ;
	int return_value, client_port, packet_length ;
	TFTP_PACKET *tftp_packet ;
#ifdef TRACE
	unsigned char *temp_ptr = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1 = *temp_ptr, c2 = *(temp_ptr+1), c3 = *(temp_ptr+2), c4 = *(temp_ptr+3) ;
#endif

	if (connect_info_table[descriptor].connection_abort_time == 0)
	{
		connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "%s%d seconds.....\r", Msg3, tftp_connection_abort_time / tftp.timer_class.clock_ticks_per_second) ;
#endif
		return CONNECTION_BROKEN ;
	}

	return_value = receive_data(&client_ip_address, &client_port,
	              connect_info_table[descriptor].socket,
	              connect_info_table[descriptor].received_packet, &packet_length) ;
	if (return_value)
		return NO_CHANGE ; /* Data not received */

	connect_info_table[descriptor].connection_abort_time = tftp_connection_abort_time ;
	tftp_packet = (TFTP_PACKET *)connect_info_table[descriptor].received_packet ;
	if ((net_to_host_short(tftp_packet->opcode) == OPCODE_DATA) &&
	    (net_to_host_short(tftp_packet->message_type.read_write_data.data_block_number) == connect_info_table[descriptor].sequence_number-1))
	{
		connect_info_table[descriptor].sequence_number-- ;
		connect_info_table[descriptor].current_state = STATE_SEND_ACK ;
		return DUPLICATE_BLOCK ;
		/* This happens if our ACK has got lost. It is required to ACK the
		   duplicate packet again while doing nothing with it */
	}
	if ((net_to_host_short(tftp_packet->opcode) != OPCODE_DATA) ||
	    (net_to_host_short(tftp_packet->message_type.read_write_data.data_block_number) != connect_info_table[descriptor].sequence_number) ||
		 (packet_length > TFTP_BLOCK_SIZE+2+2))
	{
		return NO_CHANGE ;
	}
	if (packet_length-2-2)
	{
		if (tftp_write(connect_info_table[descriptor].received_packet+2+2,
		       packet_length-2-2, connect_info_table[descriptor].file_pointer) == 0)
		{
			connect_info_table[descriptor].error_code = tftp_fileio_error_code ;
			connect_info_table[descriptor].current_state = STATE_ABORT_CONNECTION ;
			return CONNECTION_BROKEN ;
		}
	}
#ifdef TRACE
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : Block #%d, %d bytes received from %u.%u.%u.%u\r",
	        connect_info_table[descriptor].sequence_number, packet_length-2-2,
			  c1, c2, c3, c4) ;
#endif
	connect_info_table[descriptor].packet_length = packet_length ;
	connect_info_table[descriptor].current_state = STATE_SEND_ACK ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : send_ack_state_handler
	Synopsis : All this needs to do is send an acknowledgement. Information
	           such as block # to be acknowledged, destination etc. are
				  present in the descriptor block. Something else that this needs
				  to do is increment the sequence_number and move the descriptor
				  to the WAIT_FOR_DATA state. Note that if this ACK gets lost,
				  the data packet will be retransmitted by the client. But the
				  server, which is in the WAIT_FOR_DATA state detects it as a 
	           duplicate packet, but DOES acknowledge the duplicate by driving
				  the descriptor back to the SEND_ACK state with the sequence_number
				  decremented.
---------------------------------------------------------------------------*/
int send_ack_state_handler(int descriptor)
{
	int return_value ;
#ifdef TRACE
	unsigned char *temp_ptr = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1 = *temp_ptr, c2 = *(temp_ptr+1), c3 = *(temp_ptr+2), c4 = *(temp_ptr+3) ;
#endif

	return_value = send_ack_packet(connect_info_table[descriptor].peer_ip_address,
	                            connect_info_table[descriptor].peer_protocol_port,
										 connect_info_table[descriptor].socket,
										 connect_info_table[descriptor].sequence_number) ;
	if (return_value)
		return SEND_FAILED ;

#ifdef TRACE
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : ACK #%d sent to %u.%u.%u.%u\r",
	       connect_info_table[descriptor].sequence_number, c1, c2, c3, c4) ;
#endif

	if ((connect_info_table[descriptor].packet_length < TFTP_BLOCK_SIZE+2+2) &&
	    (connect_info_table[descriptor].sequence_number != 0))
	{
		/* Last packet */
		connect_info_table[descriptor].current_state = STATE_CLOSE_SOCKET ;
#ifdef TRACE
		tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : File transfer complete\r") ;
#endif
		tftp_close(connect_info_table[descriptor].file_pointer) ;
	}
	else
		connect_info_table[descriptor].current_state = STATE_WAIT_FOR_DATA ;
	connect_info_table[descriptor].sequence_number++ ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : abort_connection_state_handler
	Synopsis : This is similar to the ClosesocketHandler except that the
	           descriptor could have arrived to this state from anywhere
				  and the connection termination is "unclean", in the sense that
				  the connection is being broken because of the absence of
				  traffic for some time.
---------------------------------------------------------------------------*/
int abort_connection_state_handler(int descriptor)
{
#ifdef TRACE
	unsigned char *temp_ptr1 = (unsigned char *)&my_node_ip_address,
	              *temp_ptr2 = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1=*temp_ptr1, c2=*(temp_ptr1+1), c3=*(temp_ptr1+2), c4=*(temp_ptr1+3), c5, c6, c7, c8 ;

	c5 = *(temp_ptr2+0) ;
	c6 = *(temp_ptr2+1) ;
	c7 = *(temp_ptr2+2) ;
	c8 = *(temp_ptr2+3) ;
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : connection %u.%u.%u.%u(ARBITRARY)--%u.%u.%u.%u(%u) Aborted...\r",
			c1, c2, c3, c4,
			c5, c6, c7, c8, net_to_host_short(connect_info_table[descriptor].peer_protocol_port)) ;
#endif
	if (connect_info_table[descriptor].received_packet)
		free(connect_info_table[descriptor].received_packet) ;

	if (connect_info_table[descriptor].send_packet)
		free(connect_info_table[descriptor].send_packet) ;

	switch (connect_info_table[descriptor].error_code)
	{
		case (INCORRECT_MAGIC_NUMBER) :
			send_error_packet(connect_info_table[descriptor].peer_ip_address,
			                  connect_info_table[descriptor].peer_protocol_port,
									connect_info_table[descriptor].socket,
									"Incorrect Magic Number", ERR_UNDEFINED) ;
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP Error : Incorrect Magic Number\r") ;
			break ;

		case (INCORRECT_CHECKSUM) :
			send_error_packet(connect_info_table[descriptor].peer_ip_address,
			                  connect_info_table[descriptor].peer_protocol_port,
									connect_info_table[descriptor].socket,
									"Incorrect Checksum", ERR_UNDEFINED) ;
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP Error : Incorrect Checksum\r") ;
			break ;

		case (MEM_ALLOC_FAILED) :
			send_error_packet(connect_info_table[descriptor].peer_ip_address,
			                  connect_info_table[descriptor].peer_protocol_port,
									connect_info_table[descriptor].socket,
									"Malloc Failed", ERR_UNDEFINED) ;
			tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP Error : Insufficient memory, Malloc failed\r") ;


		default :
			break ;
	}

	closesocket(connect_info_table[descriptor].socket) ;

	if (connect_info_table[descriptor].file_pointer)
		tftp_release(connect_info_table[descriptor].file_pointer) ;

	free_connect_descriptor(descriptor) ;
	return 0 ;
}




/*---------------------------------------------------------------------------
	Function : close_socket_state_handler
	Synopsis : It releases all the buffers that have been alloc'ed for this
	           descriptor, closes the socket and frees the descriptor
				  structure area.
---------------------------------------------------------------------------*/
int close_socket_state_handler(int descriptor)
{
#ifdef TRACE
	unsigned char *temp_ptr1 = (unsigned char *)&my_node_ip_address,
	              *temp_ptr2 = (unsigned char *)&connect_info_table[descriptor].peer_ip_address ;
	int c1=*temp_ptr1, c2=*(temp_ptr1+1), c3=*(temp_ptr1+2), c4=*(temp_ptr1+3), c5, c6, c7, c8 ;
#endif
#ifdef TRACE
	c5 = *(temp_ptr2+0) ;
	c6 = *(temp_ptr2+1) ;
	c7 = *(temp_ptr2+2) ;
	c8 = *(temp_ptr2+3) ;
	tftp_printf(TFTP_TRACE_PRINTF, "\nTFTP : connection %u.%u.%u.%u(ARBITRARY)--%u.%u.%u.%u(%u) Terminated...\r",
			c1, c2, c3, c4,
			c5, c6, c7, c8, net_to_host_short(connect_info_table[descriptor].peer_protocol_port)) ;
#endif

	if (connect_info_table[descriptor].received_packet)
		free(connect_info_table[descriptor].received_packet) ;

	if (connect_info_table[descriptor].send_packet)
		free(connect_info_table[descriptor].send_packet) ;

	if (closesocket(connect_info_table[descriptor].socket))
	{
		printf ("\nTFTP : Socket close failed, will attempt next time around") ;
		return (0) ;
	}
	free_connect_descriptor(descriptor) ;

	return 0 ;
}
