#include <kstart.h>
#include <stdlib.h>
#include <string.h>
#include "scrttype.h"
#include "scrtargs.h"
#include "scrtenum.h"
#include "scrtdefs.h"
#include "scrtfuns.h"

void *ArgAddr ;

/*----------------------------------------------------------------------------
	Name		:	GetIntArg												
	Input		:	a_ptr, pointer to argument.								
	Output	:	Integer value of argument.								
	Synopsis	:	Gets the integer value of the argument.					
----------------------------------------------------------------------------*/
long GetIntArg (ArgType *a_ptr, BYTE PortNum)
{
	ValType *v_ptr ;
	long val ;

	val = 0 ;
	v_ptr = GetArgValueAddr (a_ptr, PortNum) ;

	if (v_ptr)
		val = v_ptr->IntValue ;
	return val ;
}

/*----------------------------------------------------------------------------
	Name		:	GetStrArg												
	Input		:	a_ptr, pointer to argument.								
	Output	:	String value of argument.								
	Synopsis	:	Gets the string value of the argument.					
----------------------------------------------------------------------------*/
BYTE *GetStrArg (ArgType *a_ptr, BYTE PortNum)
{
	ValType *v_ptr ;
	BYTE *val ;

	val = 0 ;
	v_ptr = GetArgValueAddr (a_ptr, PortNum) ;
	if (v_ptr)
		val = v_ptr->StrValue ;
	return val ;
}

/*----------------------------------------------------------------------------
	Name		:	GetArgValueAddr											
	Input		:	a_ptr, pointer to argument description.					
	Output	:	Pointer to value of argument.							
	Synopsis	:	Gets the pointer to argument value.						
----------------------------------------------------------------------------*/
void *GetArgValueAddr (ArgType *a_ptr, BYTE PortNum)
{
	void *val ;
	WORD address ; 
	WORD no_of_args ;

	val = 0 ;
	switch (a_ptr->ArgType) 
	{
		case E_INT_CONST :
		case E_STRING_CONST :
			val = &a_ptr->Val ;
			break ;

		case E_INTEGER :
		case E_STRING :
			switch (a_ptr->StorageClass) 
			{
				case VAR_GLOBAL :
				case VAR_CONDITION :
				case VAR_RETURN :
					ArgAddr = a_ptr->SymVal;
					val = &a_ptr->SymVal->Val;
					break;

				case VAR_PARAM :
					/*	address is 0 relative.	*/
					address = (WORD) a_ptr->IntVal;
					/*	No. of args is four below the frame pointer.	*/
					no_of_args = (WORD) (ScriptPortClass[PortNum].FramePtr - 4)->IntVal;
					val = (ArgType *)(ScriptPortClass[PortNum].FramePtr - 4 - no_of_args + address)->ArgVal;
					break;

				case VAR_LOCAL :
					/*	address is 0 relative.	*/
					address = (WORD) a_ptr->IntVal;
					val = &(ScriptPortClass[PortNum].FramePtr + address)->Val;
					break;

				case VAR_TEMPORARY :
					/*	address is 0 relative.	*/
					address = (WORD) a_ptr->IntVal ;
					ArgAddr = ScriptPortClass[PortNum].TempsPtr + address ; 
					val = &((ArgType *) ArgAddr)->Val ;
					break;
			}
			break;
	}
	return val ;
}


/*----------------------------------------------------------------------------
	Name		:	GetParamAddress											
	Input		:	param_no, parameter number (1 relative),				
					type, allowed types.									
	Output	:	Pointer to value of parameter.							
	Synopsis	:	Gets the address of the parameter for built-in			
					functions, and checks the type.						
----------------------------------------------------------------------------*/
ArgType *GetParamAddress (WORD param_no, WORD type, BYTE PortNum)
{
	ArgType *a_ptr ;

	/*	No. of args is one below the stack pointer.	*/
	if (param_no > (WORD) (ScriptPortClass[PortNum].StackTop - 1)->IntVal)
		goto error_exit ;	  /*	No such parameter.	*/

	a_ptr = ScriptPortClass[PortNum].StackTop - 2 - (ScriptPortClass[PortNum].StackTop - 1)->IntVal + param_no ;
	switch (a_ptr->ArgType) 
	{
		case E_INTEGER :
			if (type & ((WORD) INTEGER))
				break;
			goto error_exit;

		case E_INT_CONST :
			if (type & ((WORD) INT_CONST))
				break;
			goto error_exit;

		case E_STRING :
			if (type & ((WORD) STRING))
				break;
			goto error_exit;

		case E_STRING_CONST :
			if (type & ((WORD) STRING_CONST))
				break;
			goto error_exit;
	}
	return a_ptr ;

error_exit :
	return NULL ;
}


/*----------------------------------------------------------------------------
	Name		:	SetPreDefVariable										
	Input		:	var_name, name of the pre-defined variable to set,		
					a_ptr, pointer to argument.								
	Output  	:	0 if variable set, -1 if error.							
	Synopsis	:	Sets pre-defined variable to the given type and value.	
----------------------------------------------------------------------------*/
int SetPreDefVariable (BYTE *var_name, ArgType *a_ptr, BYTE PortNum)
{
	SymType *sym_ptr ;

	sym_ptr = FindSymbol (var_name, PortNum) ;
	if (!sym_ptr)
		goto error_exit ;	  /*	Symbol not found in symbol table.	*/

	sym_ptr->SymClass = a_ptr->ArgType ;
	switch (a_ptr->ArgType) 
	{
		case E_INTEGER :
			sym_ptr->IntVal = a_ptr->IntVal ;
			break ;

		case E_STRING :
			sym_ptr->StrVal = a_ptr->StrVal ;
			break ;
	}
	return 0 ;

error_exit :
	return -1 ;
}



/*----------------------------------------------------------------------------
	Name		:	UpdateStrVar											
	Input		:	old_val, pointer to old value of string variable.		
					new_val, pointer to new value.							
	Output	:	0 if variable set, -1 if error.							
	Synopsis	:	Updates the string variables value. If the old value	
					is non-null, releases it. Memory is allocated for	
					new value and new value is copied into variable.	
----------------------------------------------------------------------------*/
int UpdateStrVar (BYTE **old_val, BYTE *new_val, BYTE PortNum)
{
	BYTE *ptr ;

	if (ptr = (BYTE *) malloc (strlen (new_val) + 1)) 
	{
		if (*old_val) 
		{
			free (*old_val) ;
			*old_val = 0 ;
		}
		*old_val = ptr ;
		strcpy (ptr, new_val) ;	/*	Copy the new value.	*/
	}
	else 
	{
		script_printf (ALARM_SCRIPT_PRINTF,
					"Memory exhausted, port %d\n\r", PortNum) ;
		return -1 ;
	}
	return 0 ;
}


/*----------------------------------------------------------------------------
	Name		:	PurgeLocalSymbols										
	Input		:	None													
	Output	:	None													
	Synopsis	:	Purges the local symbols defined in the procedure.		
----------------------------------------------------------------------------*/
void PurgeLocalSymbols (BYTE PortNum)
{
	ArgType *a_ptr ;

	/*	Starting from the frame pointer upto temporaries check for	*/
	/*	local variables and if that is a string, release the memory.	*/
	a_ptr = ScriptPortClass[PortNum].FramePtr ;

	while (1) 
	{
		if (a_ptr->StorageClass == VAR_LOCAL 
					&& a_ptr->ArgType == E_STRING 
						&& a_ptr->StrVal) 
		{
			free (a_ptr->StrVal) ;
			a_ptr->StrVal = 0 ;
		}
		if (a_ptr == ScriptPortClass[PortNum].TempsPtr)
			break ;	/*	Reached end of locals.	*/
		a_ptr ++ ;	/*	Move on to next variable.	*/
	}

	#if 1
	/*	Now purge out the strings assigned to temporaries which have	*/
	/*	been released because the execution was terminated due to	*/
	/*	stack overrun, or some other internal error.	*/
	a_ptr = ScriptPortClass[PortNum].TempsPtr ;
	while (a_ptr < ScriptPortClass[PortNum].StackTop) 
	{
		/*	If the temporary is a string variable and the memory	*/
		/*	has not been released, release it and invalidate	*/
		/*	the string pointer to avoid releasing it again.	*/
		if (a_ptr->StorageClass == VAR_TEMPORARY 
				&&	a_ptr->ArgType == E_STRING 
						&& a_ptr->StrVal) 
		{
			free (a_ptr->StrVal) ;
			a_ptr->StrVal = 0 ;
		}
		a_ptr ++ ;	/*	Move onto next temporary.	*/
	}
	#endif
}
