/* SERINIT.C -- Has code to support init and deinit of specified SCC for
**				serial activity. Also has some general purpose routines to
**			    modify serial settings and to obtain some status information.
** By: Sanjay
** Start: 20, July, 1996
** Done: 22, July, 1996
*/

#include <stdio.h>
#include "rtrstd.h"
#include "serial.h"
#include "vserstr.h"
#include "serasm.h"
#include <uartmode.h>

/* External prototypes */
extern void serial_rx_complete(USHORT port_number, BYTE *sptr_buffer);

/* Local prototypes */
STATIC void force_post_pending_sends(USHORT port_number);
STATIC void return_transmit_buffers(USHORT port_number);
STATIC void pass_received_buffers_to_application(USHORT port_number);
STATIC enum TEST initialize_descriptors(USHORT port_number);
STATIC void free_receive_buffers(USHORT port_number, int num_buffers);
STATIC void put_tx_descriptors_on_free_list(port_number);
STATIC void put_rx_descriptors_on_free_list(port_number);
STATIC void init_scc_serial_settings(USHORT port_number, ULONG baud, BYTE data_bits, BYTE parity, BYTE stop_bits);
STATIC void stop_rxtx_on_scc(USHORT port_number);
STATIC void stop_tx_on_scc(USHORT port_number);
STATIC void stop_rx_on_scc(USHORT port_number);
STATIC void start_rx_on_scc(USHORT port_number);
STATIC void start_tx_on_scc(USHORT port_number);
STATIC void reinit_scc_tx_descriptors(port_number);
STATIC void reinit_scc_rx_descriptors(port_number);
STATIC ULONG get_baud_divisor_bits(USHORT port_number, ULONG baud_rate);
STATIC void set_mode_data_bits(USHORT *mode_bits, BYTE data_bits);
STATIC void set_mode_parity(USHORT *mode_bits, BYTE parity);
STATIC void set_mode_stop_bits(USHORT *mode_bits, BYTE stop_bits);
STATIC ULONG get_baud_divisor_from_baud_rate(ULONG baud_rate);

/* Globals */
SERIAL_CLASS serial;

/* Externals */
extern ULONG LocalIOToMove;

WORD	scc2_event_reg_store;
WORD	scc3_event_reg_store;
WORD	scc4_event_reg_store;
BYTE	scc2_next_rx_bd;
BYTE	scc3_next_rx_bd;
BYTE	scc4_next_rx_bd;

/* -- CODE --------------------------------------------------------------- */

/* setup_serial_port_for_use()
** 		port_number is 0 based, 
**		user_parameter is used to associate received data with some session
**		fptr_rx_callback is the function to call on data reception; this
**			function is passed the above user_parameter as a parameter
*/
enum TEST setup_serial_port_for_use(USHORT port_number, void *user_parameter,
	void (*fptr_tx_complete)(USHORT port_number, void *user_parameter, BYTE *sptr_tx_buffer),
	void (*fptr_rx_callback)(USHORT port_number, void *user_parameter, BYTE *buf_ptr, USHORT data_size),
	void (**fptr_rx_complete)(USHORT port_number, BYTE *sptr_buffer),
	ULONG baud_rate, BYTE data_bits, BYTE parity, BYTE stop_bits)
{
	if (serial.activity_enabled[port_number] == TRUE)
	{
		/* Already in use by someone else */
#ifdef DEBUG
		printf("SERIAL: Request to init port (%u) already in use.\n\r", port_number);
#endif /* DEBUG */
		return FAIL;
	}

#ifdef DEBUG
	if (fptr_tx_complete == NULL)
	{
		printf("SERIAL: Serial tx function cannot be null in call to serial init WAN %u\n\r", port_number);
		return FAIL;
	}	
	if (fptr_rx_callback == NULL)
	{
		printf("SERIAL: Serial rx callback cannot be null in call to serial init WAN %u\n\r", port_number);
		return FAIL;
	}	
#endif /* DEBUG */

	if (initialize_descriptors(port_number) == FAIL)
		return FAIL;

	serial.fptr_tx_complete[port_number] = fptr_tx_complete;
	serial.fptr_rx_callback[port_number] = fptr_rx_callback;
	serial.user_parameter[port_number] = user_parameter;

	init_scc_serial_settings(port_number, baud_rate, data_bits, parity, stop_bits);
	
	*fptr_rx_complete = serial_rx_complete;
	serial.activity_enabled[port_number] = TRUE;
#ifdef DEBUG
	printf("SERIAL: WAN %u: Inited for use\n\r", port_number);
#endif /* DEBUG */
	return PASS;
}

enum TEST free_serial_port_from_use(USHORT port_number)
{
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Deinit for port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return FAIL;
	}

	/* Stop transmit and receive on the specified SCC */
	stop_rxtx_on_scc(port_number);

	/* Free all transmit buffers */
	return_transmit_buffers(port_number);

	/* Post all pending send complete routines (the buffers belong to
	** the app that requested the send) 
	*/
	force_post_pending_sends(port_number);

	/* Free all buffers */
	/* NOTE: We assume that all app's have called the receive complete
	** routines. That is, currently we own all receive buffers.
	*/
	free_receive_buffers(port_number, MAX_SERIAL_RX_DESCRIPTORS);

	/* To be safe, raise the DTR in case other code has got it down and
	** then free-d up use of the driver.
	*/
	set_serial_dtr(port_number);

	serial.activity_enabled[port_number] = FALSE;
#ifdef DEBUG
	printf("SERIAL: WAN %u: De-inited\n\r", port_number);
#endif /* DEBUG */
	return PASS;
}


/* serial_foreground()
**	This is a major function called in the foreground loop of the main router
**	code. This checks the 'current_rx_list' to see if any data packets have
**	been received on the SCC's 2, 3 and 4 (serial) and queued up via the 
**	interrupt mechanism. In addition it releases any buffer's malloc-ed from
**	within this driver.
*/
void serial_foreground()
{
	USHORT i;

	for (i = 0; i < MAX_WAN_PORTS; i++)
	{
		if (serial.activity_enabled[i] == FALSE)
			continue;

		/* Return transmit buffers (that have been transmitted) to application.
		*/
		return_transmit_buffers(i);

		/* Check for received packets on all WAN ports */
		pass_received_buffers_to_application(i);
	}
}

enum BOOLEAN is_serial_port_in_use_by_driver(USHORT port_number)
{
	return serial.activity_enabled[port_number];
}

void flush_serial_tx_buffers(USHORT port_number)
{
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Flush serial tx buffers on port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return;
	}

	/* Stop transmit on the specified SCC */
	stop_tx_on_scc(port_number);

	return_transmit_buffers(port_number);

	/* Post all pending send complete routines (the buffers belong to
	** the app that requested the send) 
	*/
	force_post_pending_sends(port_number);
	
	put_tx_descriptors_on_free_list(port_number);

	start_tx_on_scc(port_number);

#ifdef DEBUG
	printf("SERIAL: WAN %u: Handled request to flush TX buffers\n\r", port_number);
#endif	
}

void flush_serial_rx_buffers(USHORT port_number)
{
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Flush serial rx on port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return;
	}

	/* NOTE: That some buffers may be pending with the application. We
	** assume that application does not call this function when it currently
	** owns a buffer belonging to this layer.
	*/

	/* Stop receive on the specified SCC */
	stop_rx_on_scc(port_number);

	put_rx_descriptors_on_free_list(port_number);

	start_rx_on_scc(port_number);

#ifdef DEBUG
	printf("SERIAL: WAN %u: Handled request to flush RX buffers\n\r", port_number);
#endif	
}

void modify_serial_parameters(USHORT port_number, BYTE data_bits, BYTE parity, BYTE stop_bits)
{
	USHORT mode_bits = 0;
	
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Modify serial parameters on port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return;
	}

	set_mode_data_bits(&mode_bits, data_bits);
	set_mode_parity(&mode_bits, parity);
	set_mode_stop_bits(&mode_bits, stop_bits);
	serial.mode_bits[port_number] = mode_bits;
	
	switch (port_number)
	{
	case 0:
		modify_scc2_serial_settings(mode_bits);
		break;
	case 1:
		modify_scc3_serial_settings(mode_bits);
		break;
	case 2:
		modify_scc4_serial_settings(mode_bits);
		break;
	}
#ifdef DEBUG
	printf("SERIAL: WAN %u: Handled request to modify serial parameters\n\r", port_number);
#endif	
}

void modify_serial_baud_rate(USHORT port_number, ULONG baud_rate)
{
	ULONG baud_divisor_bits;

	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Modify serial baud on port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return;
	}

	baud_divisor_bits = get_baud_divisor_bits(port_number, baud_rate);
	switch (port_number)
	{
	case 0:
		modify_scc2_serial_baud_rate(baud_divisor_bits);
		break;
	case 1:
		modify_scc3_serial_baud_rate(baud_divisor_bits);
		break;
	case 2:
		modify_scc4_serial_baud_rate(baud_divisor_bits);
		break;
	}

#ifdef DEBUG
	printf("SERIAL: WAN %u: Handled request to modify baud rate to %lu\n\r", port_number, baud_rate);
#endif	
}

/* modify_serial_rx_buffer_size()
**	This is meant for the on-the-fly modification of the MRBLR. End result is
**	that serial packets passed up will be of this size at most.
*/
USHORT modify_serial_rx_buffer_size(USHORT port_number, USHORT max_rx_buffer_size)
{
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: Modify rx buffer size on port (%u) not in use.\n\r", port_number);
#endif /* DEBUG */
		return MAX_SERIAL_RX_BUFFER_SIZE;
	}

	if (max_rx_buffer_size < 32)
		max_rx_buffer_size = 32;
	else if (max_rx_buffer_size > MAX_SERIAL_RX_BUFFER_SIZE)
		max_rx_buffer_size = MAX_SERIAL_RX_BUFFER_SIZE;

	switch (port_number)
	{
	case 0:
		set_scc2_max_rx_buffer_size(max_rx_buffer_size);
		break;
	case 1:
		set_scc3_max_rx_buffer_size(max_rx_buffer_size);
		break;
	case 2:
		set_scc4_max_rx_buffer_size(max_rx_buffer_size);
		break;
	}

#ifdef DEBUG
	printf("SERIAL: WAN %u: Handled request to modify serial rx buffer size to %u\n\r", port_number, max_rx_buffer_size);
#endif	
	return max_rx_buffer_size;
}

void set_serial_rts(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		turn_on_scc2_rts();
		break;
	case 1:
		turn_on_scc3_rts();
		break;
	case 2:
		turn_on_scc4_rts();
		break;
	}
}

void reset_serial_rts(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		turn_off_scc2_rts();
		break;
	case 1:
		turn_off_scc3_rts();
		break;
	case 2:
		turn_off_scc4_rts();
		break;
	}
}

void set_serial_dtr(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		LocalIOToMove &= ~0x08;
		break;
	case 1:
		LocalIOToMove &= ~0x10;
		break;
	case 2:
		LocalIOToMove &= ~0x20;
		break;
	}
	serial_scc_set_dtr_status(LocalIOToMove);
#ifdef DEBUG
	printf("SERIAL: WAN %u: Asserted DTR\n\r", port_number);
#endif	
}

void reset_serial_dtr(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		LocalIOToMove |= 0x08;
		break;
	case 1:
		LocalIOToMove |= 0x10;
		break;
	case 2:
		LocalIOToMove |= 0x20;
		break;
	}
	serial_scc_set_dtr_status(LocalIOToMove);
#ifdef DEBUG
	printf("SERIAL: WAN %u: De-asserted DTR\n\r", port_number);
#endif	
}

enum BOOLEAN is_dcd_present(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		return serial_is_cd_present_on_scc2();
		break;
	case 1:
		return serial_is_cd_present_on_scc3();
		break;
	case 2:
		return serial_is_cd_present_on_scc4();
		break;
	}
}

USHORT serial_get_tx_time_estimate(USHORT port_number)
{
	/* Return estimate of how long it will take to free the tx buffers.
	** Will return atleast 1.
	*/

	ULONG pc_ticks_required;


	pc_ticks_required = (
						serial.statistics[port_number].tx_pending_bytes *
/*						get_baud_divisor_from_baud_rate(serial.baud_rate[port_number]) * */
						9L *					/* 9 bits per byte tx-ed on an average (8N1) */
						182L *					/* In terms of PC ticks (18.2 times a second) */
						16L						/* UART oversampling amount */
						) / (25000000L * 10);	/* 25MHz clock */

	if (pc_ticks_required == 0)
		pc_ticks_required = 1;

	return pc_ticks_required;
}

void associate_user_parameter_with_serial_driver(USHORT port_number, BYTE *user_parameter)
{
	if (serial.activity_enabled[port_number] == FALSE)
	{
#ifdef DEBUG
		printf("SERIAL: WAN %u: Association of user parameter for inactive port\n\r", port_number);
#endif /* DEBUG */
		return;
	}

#ifdef DEBUG
	if (serial.user_parameter[port_number] != NULL)
		printf("SERIAL: WAN %u: WARNING: Modifying an existing user parameter\n\r", port_number);
#endif /* DEBUG */

	serial.user_parameter[port_number] = user_parameter;
}

enum BOOLEAN is_serial_tx_buffer_empty(USHORT port_number)
{
	if (serial.activity_enabled[port_number] == FALSE)
		return TRUE;

	if (serial.sptr_current_tx_descriptor[port_number] == NULL)
		return TRUE;
	else
		return FALSE;
}

STATIC void stop_rxtx_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		stop_rxtx_on_scc2();
		break;
	case 1:
		stop_rxtx_on_scc3();
		break;
	case 2:
		stop_rxtx_on_scc4();
		break;
	}
}

STATIC void force_post_pending_sends(USHORT port_number)
{
	SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor;

	/* We assume that serial tx and rx has been stopped */

	/* Post the entry currently being transmitted. */

	sptr_tx_descriptor = serial.sptr_current_tx_descriptor[port_number];
	if (sptr_tx_descriptor != NULL)
	{
		serial.sptr_current_tx_descriptor[port_number] == NULL;

		if (sptr_tx_descriptor->tx_buffer_owner == APPLICATION)
			serial.fptr_tx_complete[port_number](port_number, serial.user_parameter[port_number], sptr_tx_descriptor->sptr_buffer);
		else
			buffer_free(sptr_tx_descriptor->sptr_buffer);

		add_entry_to_list(&serial.free_tx_list[port_number], (LINK *)&sptr_tx_descriptor->links);
	}

	/* Post other pending entries. */

	sptr_tx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list((LINK *)&serial.current_tx_list[port_number]);
	while (sptr_tx_descriptor != NULL)
	{
		if (sptr_tx_descriptor->tx_buffer_owner == APPLICATION)
			serial.fptr_tx_complete[port_number](port_number, serial.user_parameter[port_number], sptr_tx_descriptor->sptr_buffer);
		else
			buffer_free(sptr_tx_descriptor->sptr_buffer);

		add_entry_to_list(&serial.free_tx_list[port_number], (LINK *)&sptr_tx_descriptor->links);

		sptr_tx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list((LINK *)&serial.current_tx_list[port_number]);
	}
}

/* return_transmit_buffers()
**	Frees buffers allocated from within the driver and returns app owned
**	buffers to the app via the callback. This function disables interrupts
**	for the duration of its call.
*/
STATIC void return_transmit_buffers(USHORT port_number)
{
	USHORT temp_sr;
	SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor;

	
	temp_sr = _GPL();
	_SPL(7);

	sptr_tx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.tx_complete_list[port_number]);

	_SPL(temp_sr);

	while (sptr_tx_descriptor != NULL)
	{
		if (sptr_tx_descriptor->tx_buffer_owner == APPLICATION)
			serial.fptr_tx_complete[port_number](port_number, serial.user_parameter[port_number], sptr_tx_descriptor->sptr_buffer);
		else
			buffer_free(sptr_tx_descriptor->sptr_buffer);

		temp_sr = _GPL();
		_SPL(7);

		/* Add this descriptor to the free list */
		add_entry_to_list(&serial.free_tx_list[port_number], (LINK *)&sptr_tx_descriptor->links);

		sptr_tx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.tx_complete_list[port_number]);

		_SPL(temp_sr);
	}
}

STATIC void pass_received_buffers_to_application(USHORT port_number)
{
	USHORT temp_sr;
	SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor;


	temp_sr = _GPL();
	_SPL(7);
	sptr_rx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.current_rx_list[port_number]);
	_SPL(temp_sr);

	while (sptr_rx_descriptor != NULL)
	{
		/* Call the registered callback function */

	 	(*(serial.fptr_rx_callback[port_number]))(port_number, 
				serial.user_parameter[port_number], 
				sptr_rx_descriptor->sptr_buffer, sptr_rx_descriptor->length);

		temp_sr = _GPL();
		_SPL(7);
		sptr_rx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.current_rx_list[port_number]);
		_SPL(temp_sr);
	}
}

STATIC enum TEST initialize_descriptors(USHORT port_number)
{
	BYTE *bptr;
	int i, num_buffers;


	/* For the TX part, simply zero out the descriptors and mark various
	** pointers properly.
	*/
	serial.sptr_current_tx_descriptor[port_number] = NULL;
	serial.current_tx_list[port_number].sptr_forward_link = NULL;
	serial.current_tx_list[port_number].sptr_backward_link = NULL;
	serial.tx_complete_list[port_number].sptr_forward_link = NULL;
	serial.tx_complete_list[port_number].sptr_backward_link = NULL;
	serial.free_tx_list[port_number].sptr_forward_link = NULL;
	serial.free_tx_list[port_number].sptr_backward_link = NULL;

	serial.statistics[port_number].tx_pending_bytes = 0;

	for (i = 0; i < MAX_SERIAL_TX_DESCRIPTORS; i++)
	{
		/* Put all TX descriptors on the free list */
		serial.tx_descriptor[port_number][i].length = 0;
		serial.tx_descriptor[port_number][i].sptr_buffer = NULL;
		serial.tx_descriptor[port_number][i].tx_buffer_owner = APPLICATION;
		serial.tx_descriptor[port_number][i].links.sptr_forward_link = NULL;
		serial.tx_descriptor[port_number][i].links.sptr_backward_link = NULL;
		add_entry_to_list(&serial.free_tx_list[port_number], (LINK *)&serial.tx_descriptor[port_number][i].links);
	}

	/* For the receive side, in addition to doing above, also allocate
	** buffers for the receive descriptors.
	*/
	serial.num_free_rx_buf[port_number] = MAX_SERIAL_RX_DESCRIPTORS;
	serial.num_scheduled_for_rx[port_number] = 0;
	serial.current_rx_list[port_number].sptr_forward_link = NULL;
	serial.current_rx_list[port_number].sptr_backward_link = NULL;
	serial.scheduled_for_rx[port_number].sptr_forward_link = NULL;
	serial.scheduled_for_rx[port_number].sptr_backward_link = NULL;
	serial.free_rx_list[port_number].sptr_forward_link = NULL;
	serial.free_rx_list[port_number].sptr_backward_link = NULL;

	for (i = 0; i < MAX_SERIAL_RX_DESCRIPTORS; i++)
	{
		serial.rx_descriptor[port_number][i].length = 0;
		serial.rx_descriptor[port_number][i].links.sptr_forward_link = NULL;
		serial.rx_descriptor[port_number][i].links.sptr_backward_link = NULL;
		bptr = (BYTE *)buffer_malloc(MAX_SERIAL_RX_BUFFER_SIZE + RX_BUFFER_HEADER_SPACE);
		if (bptr == NULL)
		{
			printf("SERIAL: Insufficient memory for receive buffers\n\r");
			free_receive_buffers(port_number, num_buffers);
			return FAIL;
		}
		serial.rx_descriptor[port_number][i].sptr_buffer = bptr + RX_BUFFER_HEADER_SPACE;
		/* NOTE: (The ** is not a bug) Below we convert bptr to a 
		** pointer-to-a-pointer of type SERIAL_RXTX_DESCRIPTOR.
		*/
		(*((SERIAL_RXTX_DESCRIPTOR **)bptr)) = &(serial.rx_descriptor[port_number][i]);
		num_buffers++;
		add_entry_to_list(&serial.free_rx_list[port_number], &serial.rx_descriptor[port_number][i].links);
	}

	return PASS;
}

STATIC void free_receive_buffers(USHORT port_number, int num_buffers)
{
	int i;
	BYTE *sptr_buffer;

	for (i = 0; i < num_buffers; i++)
	{
		sptr_buffer = serial.rx_descriptor[port_number][i].sptr_buffer - RX_BUFFER_HEADER_SPACE;
		buffer_free(sptr_buffer);
	}
}

STATIC void put_tx_descriptors_on_free_list(port_number)
{
	int i;

	serial.sptr_current_tx_descriptor[port_number] = NULL;
	serial.current_tx_list[port_number].sptr_forward_link = NULL;
	serial.current_tx_list[port_number].sptr_backward_link = NULL;
	serial.tx_complete_list[port_number].sptr_forward_link = NULL;
	serial.tx_complete_list[port_number].sptr_backward_link = NULL;
	serial.free_tx_list[port_number].sptr_forward_link = NULL;
	serial.free_tx_list[port_number].sptr_backward_link = NULL;

	serial.statistics[port_number].tx_pending_bytes = 0;

	for (i = 0; i < MAX_SERIAL_TX_DESCRIPTORS; i++)
	{
		/* Put all TX descriptors on the free list */
		serial.tx_descriptor[port_number][i].length = 0;
		serial.tx_descriptor[port_number][i].sptr_buffer = NULL;
		serial.tx_descriptor[port_number][i].links.sptr_forward_link = NULL;
		serial.tx_descriptor[port_number][i].links.sptr_backward_link = NULL;
		serial.tx_descriptor[port_number][i].tx_buffer_owner = APPLICATION;
		add_entry_to_list(&serial.free_tx_list[port_number], &serial.tx_descriptor[port_number][i].links);
	}

	reinit_scc_tx_descriptors(port_number);
}

STATIC void put_rx_descriptors_on_free_list(port_number)
{
	int i;

	/* We need to be careful to synchronise what we think is
	** current and what the serial interface actually uses as
	** current.
	*/

	serial.num_free_rx_buf[port_number] = MAX_SERIAL_RX_DESCRIPTORS;
	serial.num_scheduled_for_rx[port_number] = 0;
	serial.current_rx_list[port_number].sptr_forward_link = NULL;
	serial.current_rx_list[port_number].sptr_backward_link = NULL;
	serial.scheduled_for_rx[port_number].sptr_forward_link = NULL;
	serial.scheduled_for_rx[port_number].sptr_backward_link = NULL;
	serial.free_rx_list[port_number].sptr_forward_link = NULL;
	serial.free_rx_list[port_number].sptr_backward_link = NULL;

	for (i = 0; i < MAX_SERIAL_RX_DESCRIPTORS; i++)
	{
		serial.rx_descriptor[port_number][i].length = 0;
		serial.rx_descriptor[port_number][i].links.sptr_forward_link = NULL;
		serial.rx_descriptor[port_number][i].links.sptr_backward_link = NULL;
		add_entry_to_list(&serial.free_rx_list[port_number], &serial.rx_descriptor[port_number][i].links);
	}

	reinit_scc_rx_descriptors(port_number);
}

STATIC void init_scc_serial_settings(USHORT port_number, ULONG baud_rate, BYTE data_bits, BYTE parity, BYTE stop_bits)
{
	USHORT mode_bits = 0;
	ULONG baud_divisor_bits;


	baud_divisor_bits = get_baud_divisor_bits(port_number, baud_rate);
	set_mode_data_bits(&mode_bits, data_bits);
	set_mode_parity(&mode_bits, parity);
	set_mode_stop_bits(&mode_bits, stop_bits);
	serial.mode_bits[port_number] = mode_bits;

	switch (port_number)
	{
	case 0:
		serial_init_scc2(baud_divisor_bits, MAX_SERIAL_RX_BUFFER_SIZE, mode_bits);
		break;
	case 1:
		serial_init_scc3(baud_divisor_bits, MAX_SERIAL_RX_BUFFER_SIZE, mode_bits);
		break;
	case 2:
		serial_init_scc4(baud_divisor_bits, MAX_SERIAL_RX_BUFFER_SIZE, mode_bits);
		break;
	}
}

STATIC void stop_tx_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		stop_tx_on_scc2();
		break;
	case 1:
		stop_tx_on_scc3();
		break;
	case 2:
		stop_tx_on_scc4();
		break;
	}
}

STATIC void stop_rx_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		stop_rx_on_scc2();
		break;
	case 1:
		stop_rx_on_scc3();
		break;
	case 2:
		stop_rx_on_scc4();
		break;
	}
}

STATIC void start_tx_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		start_tx_on_scc2();
		break;
	case 1:
		start_tx_on_scc3();
		break;
	case 2:
		start_tx_on_scc4();
		break;
	}
}

STATIC void start_rx_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		start_rx_on_scc2();
		break;
	case 1:
		start_rx_on_scc3();
		break;
	case 2:
		start_rx_on_scc4();
		break;
	}
}

void reinit_scc_tx_descriptors(port_number)
{
	switch (port_number)
	{
	case 0:
		scc2_init_tx_descriptor();
		break;
	case 1:
		scc3_init_tx_descriptor();
		break;
	case 2:
		scc4_init_tx_descriptor();
		break;
	}
}

void reinit_scc_rx_descriptors(port_number)
{
	switch (port_number)
	{
	case 0:
		scc2_init_rx_descriptors();
		break;
	case 1:
		scc3_init_rx_descriptors();
		break;
	case 2:
		scc4_init_rx_descriptors();
		break;
	}
}

#if 0
/* there is already a function in WAN driver for getting the baud divisor.
use it. */
/*--------------------------------------------------------------------*/
/* Following functions are helper routines that are very MC68360 QUICC
** specific and baud's are specific to the clock speed (25MHz in this case).
** See QUICC User's Manual for more information.
*/
STATIC ULONG get_baud_divisor_bits(USHORT port_number, ULONG baud_rate)
{
	ULONG baud_divisor;
	ULONG baud_divisor_pattern;


	baud_divisor = get_baud_divisor_from_baud_rate(baud_rate);
	if (baud_divisor == 0)
	{
		/* Passed baud not found, so default to 19200 value */
		baud_rate = 19200;
		baud_divisor = get_baud_divisor_from_baud_rate(baud_rate);
	}

	/* Save baud divisor for our use */
	serial.baud_rate[port_number] = baud_rate;

	baud_divisor_pattern = baud_divisor << 1; 
			/* The divisor value starts from bit 1 onwards, so the shift */
	if (baud_rate <= 300)	/* For baud <= 300, set the Div16 bit */
		baud_divisor_pattern |= 0x00000001;

	return baud_divisor_pattern;
}
#else
extern	int GetBaudDivisor (int ActualBaudRate);
STATIC ULONG get_baud_divisor_bits(USHORT port_number, ULONG baud_rate)
{
	/* Save baud divisor for our use */
	serial.baud_rate[port_number] = baud_rate;

	GetBaudDivisor (baud_rate);
}
#endif


STATIC void set_mode_data_bits(USHORT *mode_bits, BYTE data_bits)
{
	switch (data_bits)
	{
	case FIVE_DATA:
		*mode_bits |= 0x0000;
		break;
	case SIX_DATA:
		*mode_bits |= 0x1000;
		break;
	case SEVEN_DATA:
		*mode_bits |= 0x2000;
		break;
	case EIGHT_DATA:
	default:
		*mode_bits |= 0x3000;
		break;
	}
}
STATIC void set_mode_parity(USHORT *mode_bits, BYTE parity)
{
	switch (parity)
	{
	case ODD:
		*mode_bits |= 0x0010;
		break;
	case EVEN:
		*mode_bits |= 0x001A;
		break;
	case NONE:
	default:
		*mode_bits |= 0x0000;
		break;
	}
}
STATIC void set_mode_stop_bits(USHORT *mode_bits, BYTE stop_bits)
{
	switch (stop_bits)
	{
	case ONE_STOP:
		*mode_bits |= 0x0000;
		break;
	case TWO_STOP:
	default:
		*mode_bits |= 0x4000;
		break;
	}
}
#if 0
/*--------------------------------------------------------------------*/
/* get_baud_divisor_from_baud_rate()
**	Returns baud divisor for baud rate or 0 if invalid baud rate.
*/
STATIC ULONG get_baud_divisor_from_baud_rate(ULONG baud_rate)
{
	struct baud_calc {
		ULONG baud_rate;
		int divisor;
	};
	struct baud_calc baud_calc_array[] =
	{
		{ 50, 1952 },		 	/* This table is computed for 25MHz QUICC */
		{ 75, 1301 },
		{ 110, 886 },
		{ 135, 722 },
		{ 150, 650 },
		{ 300, 324 },
		{ 600, 2603 },
		{ 1200, 1301 },
		{ 1800, 867 },
		{ 2000, 780 },
		{ 2400, 650 },
		{ 3600, 433 },
		{ 4800, 324 },
		{ 7200, 216 },
		{ 9600, 162 },
		{ 19200, 80 },
		{ 38400, 40 },
		{ 57600, 26 },
		{ 115200, 13 }
	};
	int i;

	for (i = 0; i < sizeof(baud_calc_array)/sizeof(struct baud_calc); i++)
	{
		if (baud_rate == baud_calc_array[i].baud_rate)
			return baud_calc_array[i].divisor;	/* Found a matching baud rate */
	}

	return 0;
}
#endif
/* -- END CODE ----------------------------------------------------------- */


