/*---------------------------------------------------------------------------
File        : FLSH.C
Author      : Sachin S Desai
Date        : 8th May 1996
Description : This is the FLASH WRITE SCHEDULER
	Any application wanting to write to the flash simply calls
	schedule_flash_write () with the source address pointer, destination
	address pointer, write-complete function pointer, etc.. The info. gets
	queued in a queue, maintained by the scheduler. From the 10 ms timer
	another function in the scheduler, write_next_page() is called. It is up
	to this function to write the next page, update the queue, etc..
		There are a few problems here. If the application queues up a buffer,
	larger than the page size, the write_next_page () routine has to take care
	to see that only one page is written. But it needs to update the pointers
	and the length feilds and NOT update head and tail indecies of the queue.
	It is only after the last page is written out of the block that the buffer
	is freed and the corresponding write_complete function is called.
		Then, there is the other side of the problem. If an application wants
	to write a buffer with size smaller than the flash-page-size, we have again
	run into hot waters. It is possible if we have flashes of page size 1K...
	We have two solutions in such cases. We could write only the queued up
	buffer (copy the rest of the page into a temporary buffer, complete it with
	the buffer to be written and then write the page. Another solution which
	has lot of headaches is to wait till the buffer builds upto the size of
	a page and then write. Visibly, this will have lot of problems like maintaining
	pointers to all buffers, decision on how long to wait for all buffers that
	comprise a flash page to come, etc..

	change : chetan : 21st July 1997
		Checking for Invalid memory locations
		get_flash_number_from_address changed
---------------------------------------------------------------------------*/
unsigned short crctab[256] ; 
#define updcrc(cp, crc) (crctab[((crc >> 8) & 0xFF)] ^ (crc << 8) ^ ((cp) & 0xFF))

#include	<defs.h>
#include <kstart.h>
#include <flashmgr.h>
#include	<stdlib.h>
#include "..\store\boot.h"

#ifndef SMALL_MEM_MAP

????? flag check. remove when not compiling for small proxy

#define FLASH_MANAGER_DEBUG 1

struct FLASH_WRITE_SCHEDULE_CLASS flash_write_scheduler ;
char flash_write_scheduler_inited = 0 ;
enum FLASH_ACCESSIBILITY flash_accessibility[NUMBER_OF_FLASH_PROMS] = {FLASH_ACCESSIBLE, FLASH_ACCESSIBLE} ;
int flash_page_size[NUMBER_OF_FLASH_PROMS] ;

extern unsigned short Fl1ProId, Fl2ProId ;
extern void new_write_to_flash (char *cptr_src, char *cptr_dest, int number_of_bytes) ;


static int get_flash_number_from_address (unsigned char *address)
{
/* NEW_BOOT */
#if 1
	return ((int) ((unsigned long) address - (unsigned long) FL_ROMM_START) / (unsigned long) FL_PROM_SIZE) ;
#else
	return ((int) ((unsigned long)address / (unsigned long)0x80000)) ;
#endif
/* NEW_BOOT */
}


/*
	Note :
		This is written to work around the problem of having to disable interrupts
		during flash_write. It is VERY, VERY important that this routine be called
		from the 10 ms timer only. The difference between write_to_flash and
		new_write_to_flash is that in new_write_to_flash, we do not wait for 10 ms
		after loading the device, hoping that there is no flash access for the next
		10 ms. Since FLASH-write is done only from here, so far so good. What is
		to be kept in mind is that, when FLASH-write is going on, there cannot be a
		FLASH-read at all.
			It has been arranged to see that new_c_write_to_flash() is never called
		with number_of_bytes > page-size. So the extra code can be carefully taken
		out. No other process has any business to call this routine except
		write_next_page(). This is very, very, very... important. Besides, it cannot
		be called with block length greater than the page size.
*/
static int new_c_write_to_flash (char *cptr_src, char *cptr_dest, int number_of_bytes)
{
	unsigned long to_flash ;
	unsigned short flash_prod_id ;

/* NEW_BOOT */
#if 1
	if ((unsigned long) cptr_dest < (unsigned long) FL_CODE_HDR
			|| (unsigned long) cptr_dest + (unsigned long) number_of_bytes > (unsigned long) FL_PROM_END+1)
	{
#if FLASH_MANAGER_DEBUG
		printf ("Alarm : Attempt to write before the code header\n") ;
#endif
		return (0) ;
	}
#else
	if ((unsigned long)cptr_dest < (unsigned long)FL_CODE_HDR)
	{
#if FLASH_MANAGER_DEBUG
		printf ("Alarm : Attempt to write before the code header\n") ;
#endif
		return (0) ;
	}
#endif
/* NEW_BOOT */

	if (number_of_bytes > 512)
		return (0) ;            /* does not write more than 512 bytes in one shot */

	to_flash = (unsigned long) cptr_dest ;
	to_flash &= 0x80000 ;

	if (to_flash)
		flash_prod_id = Fl2ProId ;
	else
		flash_prod_id = Fl1ProId ;

	if (flash_prod_id == 0x1FA4)    /* AT29C040A, page size = 256 */
	{
		if (number_of_bytes > 256)
		{
			new_write_to_flash (cptr_src, cptr_dest, 256) ;
			new_write_to_flash (cptr_src + 256, cptr_dest + 256, number_of_bytes - 256) ;
		}
		else
			new_write_to_flash (cptr_src, cptr_dest, number_of_bytes) ;
	}
	else                            /* AT29C040, page size = 512 */
		new_write_to_flash (cptr_src, cptr_dest, number_of_bytes) ;

	return (number_of_bytes) ;
}



/*
	This function is to be called before scheduling any block to be written
	to the second flash. It initializes the queue-head and tail, flash page
	size, etc.
*/
void init_flash_write_scheduler (void)
{
	int flash_number ;

	flash_write_scheduler.head_of_Q = 0 ;   /* Place to put the new block */
	flash_write_scheduler.tail_of_Q = -1 ;  /* +1 gives the place to take
	                                           next block from */

	for (flash_number = 0 ; flash_number < NUMBER_OF_FLASH_PROMS ; flash_number++)
	{
		flash_accessibility[flash_number] = FLASH_ACCESSIBLE ;
	}

	flash_write_scheduler_inited = 1 ;

	if (Fl1ProId == 0x1FA4)
		flash_page_size[0] = 256 ;/* AT29C040A */
	else
		flash_page_size[0] = 512 ; /* AT29C040 */

	if (Fl2ProId == 0x1FA4)
		flash_page_size[1] = 256 ;/* AT29C040A */
	else
		flash_page_size[1] = 512 ; /* AT29C040 */

	return ;
}

int schedule_flash_write (char *src_ptr, char *dest_ptr, unsigned short block_length,
                          void (*fptr_block_write_complete)(), char *buffer_to_free)
{
	short place_to_write_new_block ;
	FLASH_WRITE_BLOCKS *ptr_flash_write_block ;

	if (flash_write_scheduler_inited == 0)
	{
		printf ("Flash write scheduler not yet initialized\n") ;
		return (-1) ;
	}

#if FLASH_MANAGER_DEBUG

/* NEW_BOOT */
#if 1
	if ((block_length > 512) 
		|| ((unsigned long) dest_ptr < (unsigned long) FL_CODE_HDR)
			|| (unsigned long) (dest_ptr + (unsigned long) block_length) > (unsigned long) (FL_PROM_END+1))
		return (-1) ;
#else
	if ((block_length > 512) || ((unsigned long)dest_ptr < (unsigned long)FL_CODE_HDR))
		return (-1) ;
#endif
/* NEW_BOOT */

#endif

  	place_to_write_new_block = flash_write_scheduler.head_of_Q ;
	if (place_to_write_new_block == flash_write_scheduler.tail_of_Q)
		return (-1) ;  /* Queue full */

	ptr_flash_write_block = &flash_write_scheduler.flash_write_blocks[place_to_write_new_block] ;
	ptr_flash_write_block->dest_ptr = dest_ptr ;
	ptr_flash_write_block->src_ptr = src_ptr ;
	ptr_flash_write_block->block_length = block_length ;
	ptr_flash_write_block->buffer_to_free = buffer_to_free ;
	ptr_flash_write_block->fptr_block_write_complete = fptr_block_write_complete ;

	if (place_to_write_new_block == (MAX_FLASH_WRITES_PENDING - 1))
		place_to_write_new_block = 0 ;
	else
		place_to_write_new_block++ ;
	flash_write_scheduler.head_of_Q = place_to_write_new_block ;

	return (0) ;
}



/*
	There is a problem here. If a block is queued for flash write and if it
	happens to have a length greater than the page size of the flash, we are
	in with problems. So what this routine then does is simple. It writes a
	single page to the flash and just adjusts the dest-pointer, src-pointer
	and length. When the last of such pages in that block is written,
		(a). it frees the source buffer (if need be)
		(b). it calls the flash_write_complete routine.
*/
extern enum BOOLEAN user_database_updated ;
extern int tftp_udb_write_complete ;
int write_next_page (void)
{
	short next_item = flash_write_scheduler.tail_of_Q ;
	FLASH_WRITE_BLOCKS *ptr_block ;
	int flash_number = get_flash_number_from_address ((unsigned char *)FL_UDB_HDR) ;

	if (flash_write_scheduler_inited == 0)
		return (-1) ;

	if (flash_accessibility[flash_number] == FLASH_WRITE_COMPLETE)
	{
		if (tftp_udb_write_complete == 1)
		{
			tftp_udb_write_complete = 0 ;
			user_database_updated = TRUE ;
		}
	}

	/*
		If control comes here, it means it has been atleast 10 ms since
		the flash page write for ANY flash prom was done. So mark
		the accessibility of all flashes as ACCESSIBLE.
	*/
	for (flash_number = 0 ; flash_number < NUMBER_OF_FLASH_PROMS ; flash_number++)
		flash_accessibility[flash_number] = FLASH_ACCESSIBLE ;
	
	if (next_item == (MAX_FLASH_WRITES_PENDING - 1))
		next_item = 0 ;
	else
		next_item++ ;

	if (next_item == flash_write_scheduler.head_of_Q)
	{
		return (-1) ;  /* Queue empty */
	}

	ptr_block = &flash_write_scheduler.flash_write_blocks[next_item] ;
	flash_number = get_flash_number_from_address (ptr_block->dest_ptr) ;

	if (flash_number >= NUMBER_OF_FLASH_PROMS)
	{
		if (flash_number >= NUMBER_OF_FLASH_PROMS)
#if FLASH_MANAGER_DEBUG
		{
			printf ("Alarm : Attempt to write to flash number %d at %08X\n",
			         flash_number, ptr_block->dest_ptr) ;
		}
#endif

		/*
			Free up the block, call the completion
			routine and update the queue tail index
		*/

		if (ptr_block->buffer_to_free)
			free (ptr_block->buffer_to_free) ;

		flash_write_scheduler.tail_of_Q = next_item ;

		if (ptr_block->fptr_block_write_complete)
			(*ptr_block->fptr_block_write_complete)() ;

		return (-1) ;
	}

	flash_accessibility[flash_number] = FLASH_WRITE_IN_PROGRESS ;
	new_c_write_to_flash (ptr_block->src_ptr, ptr_block->dest_ptr, flash_page_size[flash_number]) ;

	if (ptr_block->block_length > flash_page_size[flash_number])
	{
		ptr_block->src_ptr += flash_page_size[flash_number] ;
		ptr_block->dest_ptr += flash_page_size[flash_number] ;
		ptr_block->block_length -= flash_page_size[flash_number] ;
		return (0) ;
		/* During the next 10 ms timer the next page is written */
	}

	if (ptr_block->buffer_to_free)
		free (ptr_block->buffer_to_free) ;

	flash_write_scheduler.tail_of_Q = next_item ;

	if (ptr_block->fptr_block_write_complete)
		(*ptr_block->fptr_block_write_complete)() ;

	return (0) ;
}
#endif

#define P 0x8408
void	calculate_crc_table (void)
{
	register	unsigned short b, v;
	register unsigned short i;

	for (b = 0 ; ; )
	{
		v = b ;
		for (i = 8 ; i-- ; )
			v = v & 1  ?  (v >> 1) ^ P : v >> 1 ;

		crctab[b] = v & 0xffff ;
		if (++b == 256)
			break ;
	}
}


unsigned short update_crc(unsigned short accumulated_crc, unsigned char *packet, unsigned long packet_length)
{
	unsigned long i;
	for (i = 0 ; i < packet_length ; i++)
		accumulated_crc = updcrc(0xFF & packet[i], accumulated_crc);

	return (accumulated_crc) ;
}

#if SMALL_MEM_MAP


extern	void asmSectorErase(char *sector_base_address);
extern	void	asmCopyToSector(word *SectorAddress, word *Source, dword length);
#define	GET_BASE_ADDR	0
#define	GET_SIZE			1

static const dword SectorBaseAddress[] =		/* AMD 29F400AB flash page sizes table */
{
	0l,					/* PAGE_16K */
	16*1024l,			/* PAGE_8K */
	24*1024l,			/* PAGE_8K */
	32*1024l,			/* PAGE_32K */
	64*1024l,			/* PAGE_64K */
	128*1024l,			/* PAGE_64K */
	192*1024l,	 		/* PAGE_64K */
	256*1024l,			/* PAGE_64K */
	320*1024l,			/* PAGE_64K */
	384*1024l,			/* PAGE_64K */
	448*1024l			/* PAGE_64K */
};

static dword GetSectorSizeOrBaseAddress(dword BaseAddress, int GetBaseAddress)
{
	int	index;
	int	MaximumSectorNumber;

	index = 1;
	if (BaseAddress >= (512*1024))		/* First non-addressable location */
		return 0xFFFFFFFF;					/* return invalid */
	MaximumSectorNumber = sizeof(SectorBaseAddress) /
														sizeof(SectorBaseAddress[0]);
	/* find the first baseaddress above the passed in address */
	for (;index < MaximumSectorNumber; index++)
		if (SectorBaseAddress[index] > BaseAddress)
			break;

	if (GetBaseAddress == GET_BASE_ADDR)
		return SectorBaseAddress[index-1];

	/* return the previous sector base address */
	if (index > 5)
		return (64l*1024);
	return (SectorBaseAddress[index] - SectorBaseAddress[index-1]);
}

enum BOOLEAN	c_write_to_AMD_flash (BYTE *src, BYTE *dest, int count)
{
	int	sector_base_addr;
	int	sector_size;
	int	offset_in_sector;
	char	*sector_buffer;

	/* get sector size and sector base address */
	sector_base_addr = GetSectorSizeOrBaseAddress(dest, GET_BASE_ADDR);
	sector_size = GetSectorSizeOrBaseAddress(dest, GET_SIZE);
	offset_in_sector = (int)dest - sector_base_addr;

	if (sector_base_addr < FL_BOOT_HDR)
	{
		printf("Flash write: can't overwrite boot area\n");
		return FALSE;
	}

	if (offset_in_sector + count > sector_size)
	{
		printf("Flash write: sector boundary error\n");
		return FALSE;
	}
		
	/* allocate memory for reading and storing sector */
	sector_buffer = (char *)malloc(sector_size);
	if (sector_buffer == NULL)
	{
		printf("Flash write: no mem. need %d bytes\n", sector_size);
		return FALSE;
	}

	/* read sector into mem */
	memcpy(sector_buffer, sector_base_addr, sector_size);

	/* erase sector */
	asmSectorErase(sector_base_addr);

	/* modify sector buffer with new info */
	memcpy(sector_buffer + offset_in_sector, src, count);

	/* write sector */
	asmCopyToSector(sector_base_addr, sector_buffer, sector_size);

	/* free sector buffer */
	free(sector_buffer);

	return TRUE;

}

/* Jo 08/07/99 For UDB write */

enum BOOLEAN c_write_user_database_to_AMD_flash (BYTE *src, BYTE *dest, int count)
{
	int	sector_base_addr ;
	int	sector_size ;
	int	offset_in_sector ;
	char	*sector_buffer ;

	/* get sector size and sector base address */
	sector_base_addr = GetSectorSizeOrBaseAddress(dest, GET_BASE_ADDR) ;
	sector_size = GetSectorSizeOrBaseAddress(dest, GET_SIZE) ;
	offset_in_sector = (int)dest - sector_base_addr ;

	if (sector_base_addr < FL_BOOT_HDR)
	{
		printf("Flash write: can't overwrite boot area\n") ;
		return FALSE ;
	}

	if (offset_in_sector + count > sector_size)
	{
		printf ("Flash write: sector boundary error\n") ;
		return FALSE ;
	}

	if (offset_in_sector == 0)
		asmSectorErase (sector_base_addr) ;
		
	/* allocate memory for reading and storing sector */
	sector_buffer = (char *) malloc(count) ; 
	if (sector_buffer == NULL)
	{
		printf ("Flash write: no mem. need %d bytes\n", sector_size) ;
		return FALSE ;
	}

	/* modify sector buffer with new info */
	memcpy (sector_buffer, src, count) ;

	/* write sector */
	asmCopyToSector (sector_base_addr + offset_in_sector, sector_buffer, count) ;

	/* free sector buffer */
	free (sector_buffer) ;

	return TRUE ;
}

#endif
