#include "defs.h"		 
/*---------------------------------------------------------------------------
	file    :       log.c
	Auth    :       vidy
	Date    :       30th Jan 98
	Synop   :       Contains the routines for the log abstract data type
	Modifications : Imran made changes to handle events as records
	Assumptions :
	1. This is for logging events which are not-too-fast. So it
	assumes that the log read is a much faster operation than log write.
	2. log functions are not to be used from ISRs
---------------------------------------------------------------------------*/
#include <stdlib.h>
#include <incall.h>
#include <string.h>
#include "..\applicat\telnet\menusup.h"  /* has get_time_in_hms() */

#include "log.h"
#include <logif.h>

#define TIME_STAMP_CHARS  15 

int     log_enabled = 1;      /* flag to turn on/off logging dynamic */
/* Jo int     time_stamp_log = 1; */

static log_descriptor_type log_descriptors[LOG_MAX_OPENS];
static event_record *log_head_ptr;  	    /* pointer to first record*/
static event_record *log_tail_ptr;   	    /* pointer to last record */
static unsigned long log_number_of_msgs;	 /* number of log msgs */

/* Jo 26/04/99 */
static const char *log_msgs[] = {
                         "Remote PAP authentication failed on WAN",
                         "Events log initialised",
								 "Modem initing on WAN",	
								 "Modem dialing on WAN",
								 "Modem init error on WAN",
								 "No modem resp on WAN",
								 "Modem reports no answer on WAN", 
								 "No connect response from modem on WAN",
								 "Modem connected on WAN",
								 "Modem ring on WAN",
								 "Modem reports NO DIAL TONE on WAN",
								 "Dialing on WAN",
								 "DCD detected on WAN",
								 "DCD lost on WAN",
								 "Changing to PPP on WAN",
								 "Changing to UART on WAN",
								 "Callback arranged on WAN",
			 					 "Callback cancelled on WAN",
								 "CHAP authentication OK on WAN",
								 "CHAP authentication fails on WAN",
								 "Remote PAP authentication OK on WAN",
								 "PAP authentication OK on WAN",
								 "PAP authentication failed on WAN",
								 "Going down by no demand on DOD port",
                         "Coming up by demand on DOD port",	
								 "Network protocols up on WAN",
								 "Dial retries fail on WAN",
								 "Modem reports ERROR on WAN",
      						 "Modem reports BUSY on WAN",
							    "Modem reports NO CARRIER on WAN"		
                         };            

/*---------------------------------------------------------------------------
init the head and tail pointers.Also make all log_descriptors invalid.
To be called only once during init time
---------------------------------------------------------------------------*/
void init_log(void)
{
	int index;
	 
	log_head_ptr = log_tail_ptr = NULL;
	log_number_of_msgs = 0;         /* number of log msgs */

	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		log_descriptors[index].valid = 0;
	}
}

/*---------------------------------------------------------------------------
this will delete all messages,make head and tail pointers null and invalidate
the log_descriptors
---------------------------------------------------------------------------*/
void clean_log(void)
{
	int index;
	event_record *tbuf;

	tbuf = log_head_ptr;
	while(tbuf != NULL)
	{
		log_head_ptr = log_head_ptr->next_event_ptr;
		free(tbuf);
		tbuf = log_head_ptr;
	}

	log_head_ptr = log_tail_ptr = NULL;
	log_number_of_msgs = 0;         
	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		log_descriptors[index].valid = 0;
	}
}

/*---------------------------------------------------------------------------
open_log: open log file for read. returns the index of log_descriptor
sets the read_ptr to log_head_ptr
---------------------------------------------------------------------------*/
int open_log(void)
{
	int index = 0;

	if (!log_enabled)
		return LOG_NOT_LOGGING;
	for(index = 0; index < LOG_MAX_OPENS; index++)
	{
		if (log_descriptors[index].valid == FALSE)
		{
			log_descriptors[index].valid = TRUE;
			log_descriptors[index].read_ptr = log_head_ptr;
			return index;
		}
	}
	return LOG_ERR_MAX_OPEN;
}

/*---------------------------------------------------------------------------
just make the descriptor invalid
---------------------------------------------------------------------------*/
int close_log(int log_descriptor)
{
	if (!log_enabled)
		return LOG_NOT_LOGGING;
	log_descriptors[log_descriptor].valid = FALSE;
}

/*---------------------------------------------------------------------------
get the time stamp to be put into the buffer
---------------------------------------------------------------------------*/
static int get_log_time_stamp(ULONG time,char *buffer)
{
	char tbuf[40];

	get_time_in_hms(time, tbuf);
	sprintf(buffer, "%s-",tbuf);
	return strlen(buffer);
}


/*---------------------------------------------------------------------------
Make space by freeing up some records.makesure that all the read pointers in 
between are invalidated
---------------------------------------------------------------------------*/
static void free_up_event_records(void)
{
	event_record *old_head_ptr;
	int ind,index;

	for(ind = 0;ind < LOG_FREE_UP_QUANTUM;ind++)
	{
		old_head_ptr = log_head_ptr;     	/* keep track of the old_head_ptr */
		log_head_ptr = log_head_ptr->next_event_ptr;

		/* check if any read descriptors have become invalid */
		for(index = 0;index < LOG_MAX_OPENS;index++)
		{
			if ((log_descriptors[index].valid == TRUE)	&&
				 (log_descriptors[index].read_ptr == old_head_ptr))
				 		log_descriptors[index].valid = FALSE; 
		}
		free(old_head_ptr);
		log_number_of_msgs--;
	}
}


/*---------------------------------------------------------------------------
write_log: adds an event record.Keeps the number of records within a certain
maximum limit.
---------------------------------------------------------------------------*/
void write_log(LOG_EVENTS event_type,BYTE port_no)
{

	if (!log_enabled)
		return;

	/* if log_head_ptr does not exist , create it */
	if(log_head_ptr == NULL)
	{
		log_head_ptr = (event_record*)malloc(sizeof(event_record));
		if(log_head_ptr == NULL)	
			return;
		log_tail_ptr = log_head_ptr;
	}
	else
	{
		log_tail_ptr->next_event_ptr = (event_record*)malloc(sizeof(event_record));
		if(log_tail_ptr->next_event_ptr == NULL)
			return;
		log_tail_ptr = log_tail_ptr->next_event_ptr;
	}
	/* prepend time if needed here only */
/*	Jo if (time_stamp_log)
	{	
	 	log_tail_ptr->time_offset = lsl.system_mib.sysUpTime/100;
	} */

 	log_tail_ptr->time_offset = lsl.system_mib.sysUpTime/100;
	log_tail_ptr->event_type     = event_type;
	log_tail_ptr->port_number    = port_no;
	log_tail_ptr->next_event_ptr = NULL;

	log_number_of_msgs++;           /* number of log msgs */		
	if(log_number_of_msgs > LOG_MAX_RECORDS)
		free_up_event_records();

}


/*---------------------------------------------------------------------------
read_log: read  "count" number of bytes
input : int log_desc - log descriptor
			char * buffer - buffer to copy read info
			count - max read count;
Algo
	derive from the write_log
	returns count of chars copied into buffer
	advances the read pointer (in log_descriptor) by number of events read.
---------------------------------------------------------------------------*/
int read_log(int log_descriptor, char *buffer, int count)
{                                                                           
	int log_count,msg_index,max_count,index;
	char time_buf[TIME_STAMP_CHARS],tbuf[LOG_NORMAL_MSG_LEN];
	event_record *read_ptr,*temp_ptr;

	if (!log_enabled)		 
		return LOG_NOT_LOGGING; 

	if (!log_descriptors[log_descriptor].read_ptr) 
		return 0;  /* reading not possible */

	if (log_head_ptr == NULL)
		return LOG_NOT_LOGGING;					  

	if (!log_descriptors[log_descriptor].valid)
			return LOG_INVALID_DESCRIPTOR;					

	read_ptr = log_descriptors[log_descriptor].read_ptr;
	log_count = 0;
	max_count = count;
	*buffer = '\0';

	while((read_ptr != NULL) && (strlen(buffer) < max_count))
	{
		get_log_time_stamp(read_ptr->time_offset,time_buf);
		msg_index = (int) read_ptr->event_type;

		if((int) read_ptr->port_number == (int) 0)
			sprintf(tbuf,"%s%s%c",time_buf,log_msgs[msg_index],CHAR_NEW_LINE);
		else
			sprintf(tbuf,"%s%s %d%c",time_buf,log_msgs[msg_index],read_ptr->port_number,CHAR_NEW_LINE);

		if((strlen(buffer) + strlen(tbuf)) >= max_count)
		{
			break;		/* count should not be exceeded */
		}
		strcat(buffer,tbuf);
		read_ptr = read_ptr->next_event_ptr;
	}
	log_descriptors[log_descriptor].read_ptr = read_ptr;
	if ((strlen(buffer) < (max_count-1)) && (read_ptr))
	{  
		for(index=strlen(buffer); index < (max_count-2) ; index++)
		{
			buffer[index] = '#';   /* padding extra space */
		}
		buffer[index++] = CHAR_NEW_LINE;
		buffer[index] = NULL;
  	} 
	count = strlen(buffer);
	buffer[count] = CHAR_NEW_LINE;   /* Replace NULL by new line char */
	return (count + 1);		
}

/* Jo 05/05/99 */
#ifdef _BIG_PROXY_
/*---------------------------------------------------------------------------
will return the approximate number of bytes and number of messages
present in the log at the moment
---------------------------------------------------------------------------*/
int get_info_log(int *approx_total_size, int *approx_number_of_msgs)
{
	if (!log_enabled)
		return LOG_NOT_LOGGING;

	*approx_number_of_msgs = log_number_of_msgs;
	*approx_total_size = log_number_of_msgs * LOG_NORMAL_MSG_LEN;

	return TRUE;
}
#endif
/* Jo 05/05/99 */

/***********************     last line     ***********************/


