/****************************************************************************
									IP Filtering

Duplicate entries are not to be allowed in the configuration. If allowed, the
later entry will take effect, but the older entry will simply eat up space in
the chain. So, ROUCON has to take care of not allowing duplicate entries.

****************************************************************************/

#include "defs.h"

#ifdef IP_FILTERING
#include	<stddef.h>
#include	<string.h>
#include	<stdio.h>
#include <stdlib.h>
#include "ip.h"	  
#include	<vnvrmstr.h>
#include "ipfilt.h"

/*#define PRINTF 1*/


/* Very important :
   	the "Number of Filtering Database Entries" configuration string should
   appear before any other filter in the ini-file. If filters are added thru'
	ROUCON, there should be no problem. But if the filters are put manually
	in the file, please take care of this
*/


/* local prototypes */

enum BOOLEAN ip_initialize_filtering (int max_filtering_entries) ;
enum TEST ip_set_filtering_table_size (char *cptr_port_number_ip_address_string) ;
enum TEST ip_src_filter_cfg_string (char *cptr_port_number_ip_address_string) ;
enum TEST ip_dest_filter_cfg_string (char *cptr_port_number_ip_address_string) ;

IP_FILTERING_ENTRY *ip_get_free_filtering_entry () ;
void ip_free_filtering_entry (IP_FILTERING_ENTRY *ip_filtering_entry) ;

IP_FILTERING_ENTRY *ip_get_src_filtering_entry (ULONG ip_address, USHORT protocol_port, BYTE protocol) ;
IP_FILTERING_ENTRY *ip_get_dest_filtering_entry (ULONG ip_address, USHORT protocol_port, BYTE protocol) ;

enum BOOLEAN ip_add_entry_to_src_filtering_database (USHORT port_number,
    ULONG lower_ip_address, ULONG higher_ip_address, IP_FILTER_TYPE filter_type) ;
enum BOOLEAN ip_add_entry_to_dest_filtering_database (USHORT port_number,
    ULONG lower_ip_address, ULONG higher_ip_address, IP_FILTER_TYPE filter_type) ;

enum BOOLEAN ip_add_entry_to_src_port_filtering_database (USHORT port_number,
    USHORT lower_port, USHORT higher_port, BYTE protocol, IP_FILTER_TYPE filter_type) ;
enum BOOLEAN ip_add_entry_to_dest_port_filtering_database (USHORT port_number,
    USHORT lower_port, USHORT higher_port, BYTE protocol, IP_FILTER_TYPE filter_type) ;

void display_ip_filtering_table () ;


IP_FILTERING_ENTRY *ip_filtering_table = NULL ;
enum BOOLEAN ip_filtering_inited = FALSE ;
USHORT ip_filter_table_size, ip_number_of_filter_entries = 0, ip_number_of_dest_filter_entries = 0, ip_number_of_src_filter_entries = 0 ;
IP_FILTERING_ENTRY *sptr_ip_src_filter_chain = NULL, *sptr_ip_dest_filter_chain = NULL ;
IP_FILTER_DEFAULT_ACTION ip_filter_default_rx_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;
IP_FILTER_DEFAULT_ACTION ip_filter_default_tx_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;
IP_FILTER_DEFAULT_ACTION ip_filter_default_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;

char *tcp_string = "TCP", *udp_string = "UDP" ;





enum BOOLEAN ip_initialize_filtering (int max_filtering_entries)
{
	int i ;

	ip_filtering_table =
	  (IP_FILTERING_ENTRY *) malloc (max_filtering_entries * sizeof (IP_FILTERING_ENTRY)) ;

	if (ip_filtering_table == NULL)
	{
		ip_filtering_inited = FALSE ;
#ifdef PRINTF
		if (max_filtering_entries)
			ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : malloc failed") ;
#endif
		return (FALSE) ;
	}

	for (i = 0 ; i < max_filtering_entries ; i++)
		ip_filtering_table[i].type = IP_INVALID_FILTER ;

	ip_filtering_inited = TRUE ;
	ip_filter_table_size = max_filtering_entries ;

	return (TRUE) ;
}


enum TEST ip_set_filtering_table_size (char *cptr_port_number_ip_address_string)
{
	ULONG max_size ;

	sscanf (cptr_port_number_ip_address_string, "%3d", &max_size) ;

	ip_initialize_filtering (max_size) ;

	return (PASS) ;
}


#if 0

enum TEST ip_set_filtering_default_rx_action (char *filter_cfg_string)
{
	if (strstr (filter_cfg_string, "forward"))
		ip_filter_default_rx_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;
	else
		ip_filter_default_rx_action = IP_FILTER_DEFAULT_ACTION_FILTER ;

	return (PASS) ;
}


enum TEST ip_set_filtering_default_tx_action (char *filter_cfg_string)
{
	if (strstr (filter_cfg_string, "forward"))
		ip_filter_default_tx_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;
	else
		ip_filter_default_tx_action = IP_FILTER_DEFAULT_ACTION_FILTER ;

	return (PASS) ;
}

#endif


enum TEST ip_set_filtering_default_action (char *filter_cfg_string)
{
	if (strstr (filter_cfg_string, "forward"))
		ip_filter_default_action = IP_FILTER_DEFAULT_ACTION_FORWARD ;
	else
		ip_filter_default_action = IP_FILTER_DEFAULT_ACTION_FILTER ;

	ip_filter_default_rx_action = ip_filter_default_tx_action = ip_filter_default_action ;

	return (PASS) ;
}


IP_FILTERING_ENTRY *ip_get_free_filtering_entry ()
{
	int i ;
	IP_FILTERING_ENTRY *temp_ptr = ip_filtering_table ;

	if (ip_filtering_inited == FALSE)
	{
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\nIP Filtering table not yet initialized") ;
#endif
		return (NULL) ;
	}

	for (i = 0 ; i < ip_filter_table_size ; i++)
	{
		if (temp_ptr->type == IP_INVALID_FILTER)
			return (temp_ptr) ;

		temp_ptr++ ;
	}
	
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\nNo more free filtering entries") ;
#endif

	return (NULL) ;
}

void ip_free_filtering_entry (IP_FILTERING_ENTRY *ip_filtering_entry)
{
	if (ip_filtering_inited == FALSE)
		return ;

	ip_filtering_entry->type = IP_INVALID_FILTER ;

	return ;
}


enum BOOLEAN ip_add_entry_to_src_filtering_database (USHORT port_number,
    ULONG lower_ip_address, ULONG higher_ip_address, IP_FILTER_TYPE filter_type)
{
	IP_FILTERING_ENTRY *new_ip_filter ;

	if (ip_filtering_inited == FALSE)
	{
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\IP Filtering not initialized") ;
#endif
		return (FALSE) ;
	}

	new_ip_filter = ip_get_free_filtering_entry () ;
	if (new_ip_filter == NULL)
	{
		return (FALSE) ;
	}

	new_ip_filter->lower_end.ip_address = lower_ip_address ;
	new_ip_filter->higher_end.ip_address = higher_ip_address ;
	if (filter_type == IP_FILTER_ON_SOURCE_ADDRESS)
		new_ip_filter->higher_end.ip_address = lower_ip_address ;
	new_ip_filter->port = port_number ;
	new_ip_filter->type = filter_type ;
	new_ip_filter->forward_link = sptr_ip_src_filter_chain ;
	sptr_ip_src_filter_chain = new_ip_filter ;

	ip_number_of_filter_entries++ ;
   ip_number_of_src_filter_entries++ ;

#if 0
	printf ("\nInstalled IP source Filter") ;
#endif

	return (TRUE) ;
}



enum BOOLEAN ip_add_entry_to_dest_filtering_database (USHORT port_number,
    ULONG lower_ip_address, ULONG higher_ip_address, IP_FILTER_TYPE filter_type)
{
	IP_FILTERING_ENTRY *new_ip_filter ;

	if (ip_filtering_inited == FALSE)
	{
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\IP Filtering not initialized") ;
#endif
		return (FALSE) ;
	}

	new_ip_filter = ip_get_free_filtering_entry () ;
	if (new_ip_filter == NULL)
	{
		return (FALSE) ;
	}

	new_ip_filter->lower_end.ip_address = lower_ip_address ;
	new_ip_filter->higher_end.ip_address = higher_ip_address ;
	if (filter_type == IP_FILTER_ON_DESTINATION_ADDRESS)
		new_ip_filter->higher_end.ip_address = lower_ip_address ;
	new_ip_filter->port = port_number ;
	new_ip_filter->type = filter_type ;
	new_ip_filter->forward_link = sptr_ip_dest_filter_chain ;
	sptr_ip_dest_filter_chain = new_ip_filter ;

	ip_number_of_filter_entries++ ;
   ip_number_of_dest_filter_entries++ ;

#if 0
	printf ("\nInstalled IP Destination Address Filter") ;
#endif

	return (TRUE) ;
}


enum BOOLEAN ip_add_entry_to_src_port_filtering_database (USHORT port_number,
    USHORT lower_protocol_port, USHORT higher_protocol_port, BYTE protocol, IP_FILTER_TYPE filter_type)
{
	IP_FILTERING_ENTRY *new_ip_filter ;

	if (ip_filtering_inited == FALSE)
	{
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\IP Filtering not initialized") ;
#endif
		return (FALSE) ;
	}

	new_ip_filter = ip_get_free_filtering_entry () ;
	if (new_ip_filter == NULL)
	{
		return (FALSE) ;
	}

	new_ip_filter->lower_end.protocol_port.port_number = lower_protocol_port ;
   new_ip_filter->higher_end.protocol_port.protocol =
   new_ip_filter->lower_end.protocol_port.protocol = protocol ;
	new_ip_filter->higher_end.protocol_port.port_number = higher_protocol_port ;
	if (filter_type == IP_FILTER_ON_SOURCE_PORT)
		new_ip_filter->higher_end.protocol_port.port_number = lower_protocol_port ;
	new_ip_filter->port = port_number ;
	new_ip_filter->type = filter_type ;
	new_ip_filter->forward_link = sptr_ip_src_filter_chain ;
	sptr_ip_src_filter_chain = new_ip_filter ;

	ip_number_of_filter_entries++ ;
   ip_number_of_src_filter_entries++ ;

#if 0
	printf ("\nInstalled IP source Filter") ;
#endif

	return (TRUE) ;
}


enum BOOLEAN ip_add_entry_to_dest_port_filtering_database (USHORT port_number,
    USHORT lower_protocol_port, USHORT higher_protocol_port, BYTE protocol, IP_FILTER_TYPE filter_type)
{
	IP_FILTERING_ENTRY *new_ip_filter ;

	if (ip_filtering_inited == FALSE)
	{
#ifdef PRINTF
		ip_printf (IP_FILTER_PRINTF, "\IP Filtering not initialized") ;
#endif
		return (FALSE) ;
	}

	new_ip_filter = ip_get_free_filtering_entry () ;
	if (new_ip_filter == NULL)
	{
		return (FALSE) ;
	}

	new_ip_filter->lower_end.protocol_port.port_number = lower_protocol_port ;
   new_ip_filter->lower_end.protocol_port.protocol =
   new_ip_filter->lower_end.protocol_port.protocol = protocol ;
	new_ip_filter->higher_end.protocol_port.port_number = higher_protocol_port ;
	if (filter_type == IP_FILTER_ON_DESTINATION_PORT)
		new_ip_filter->higher_end.protocol_port.port_number = lower_protocol_port ;
	new_ip_filter->port = port_number ;
	new_ip_filter->type = filter_type ;
	new_ip_filter->forward_link = sptr_ip_dest_filter_chain ;
	sptr_ip_dest_filter_chain = new_ip_filter ;

	ip_number_of_filter_entries++ ;
   ip_number_of_dest_filter_entries++ ;

#if 0
	printf ("\nInstalled IP Destination Port Filter") ;
#endif

	return (TRUE) ;

}


enum TEST ip_src_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG ip_address ;
	BYTE ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju",
            (int *)&ip_address_array[0], (int *)&ip_address_array[1], (int *)&ip_address_array[2], (int *)&ip_address_array[3]) ;

	   ip_address = net_to_host_long ( *((ULONG *)&ip_address_array[0])) ;

	   if ((ip_add_entry_to_src_filtering_database (port_number, ip_address, ip_address, IP_FILTER_ON_SOURCE_ADDRESS) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_dest_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG ip_address ;
	BYTE ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju",
            (int *)&ip_address_array[0], (int *)&ip_address_array[1], (int *)&ip_address_array[2], (int *)&ip_address_array[3]) ;

      ip_address = net_to_host_long ( *((ULONG *)&ip_address_array[0])) ;

	   if ((ip_add_entry_to_dest_filtering_database (port_number, ip_address, ip_address, IP_FILTER_ON_DESTINATION_ADDRESS) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_src_range_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG low_ip_address, high_ip_address ;
	BYTE low_ip_address_array[10], high_ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
	   sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju,%03ju.%03ju.%03ju.%03ju",
			   (int *)&low_ip_address_array[0], (int *)&low_ip_address_array[1], (int *)&low_ip_address_array[2], (int *)&low_ip_address_array[3],
			   (int *)&high_ip_address_array[0], (int *)&high_ip_address_array[1], (int *)&high_ip_address_array[2], (int *)&high_ip_address_array[3]) ;

	   low_ip_address = net_to_host_long ( *((ULONG *)&low_ip_address_array[0])) ;
	   high_ip_address = net_to_host_long ( *((ULONG *)&high_ip_address_array[0])) ;

	   if ((ip_add_entry_to_src_filtering_database (port_number, low_ip_address, high_ip_address, IP_FILTER_ON_SOURCE_RANGE) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;
      cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_dest_range_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG low_ip_address, high_ip_address ;
	BYTE low_ip_address_array[10], high_ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
	   sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju,%03ju.%03ju.%03ju.%03ju",
			   (int *)&low_ip_address_array[0], (int *)&low_ip_address_array[1], (int *)&low_ip_address_array[2], (int *)&low_ip_address_array[3],
			   (int *)&high_ip_address_array[0], (int *)&high_ip_address_array[1], (int *)&high_ip_address_array[2], (int *)&high_ip_address_array[3]) ;

	   low_ip_address = net_to_host_long ( *((ULONG *)&low_ip_address_array[0])) ;
	   high_ip_address = net_to_host_long ( *((ULONG *)&high_ip_address_array[0])) ;

	   if ((ip_add_entry_to_dest_filtering_database (port_number, low_ip_address, high_ip_address, IP_FILTER_ON_DESTINATION_RANGE) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;
      cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_src_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG ip_address ;
	BYTE ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju",
            (int *)&ip_address_array[0], (int *)&ip_address_array[1], (int *)&ip_address_array[2], (int *)&ip_address_array[3]) ;

      ip_address = net_to_host_long ( *((ULONG *)&ip_address_array[0])) ;

	   if ((ip_add_entry_to_src_filtering_database (port_number, ip_address, ip_address, IP_FORWARD_ON_SOURCE_ADDRESS) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }
	return (PASS) ;
}


enum TEST ip_dest_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG ip_address ;
	BYTE ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju",
            (int *)&ip_address_array[0], (int *)&ip_address_array[1], (int *)&ip_address_array[2], (int *)&ip_address_array[3]) ;

	   ip_address = net_to_host_long ( *((ULONG *)&ip_address_array[0])) ;

	   if ((ip_add_entry_to_dest_filtering_database (port_number, ip_address, ip_address, IP_FORWARD_ON_DESTINATION_ADDRESS) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_src_range_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG low_ip_address, high_ip_address ;
	BYTE low_ip_address_array[10], high_ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
	   sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju,%03ju.%03ju.%03ju.%03ju",
			   (int *)&low_ip_address_array[0], (int *)&low_ip_address_array[1], (int *)&low_ip_address_array[2], (int *)&low_ip_address_array[3],
			   (int *)&high_ip_address_array[0], (int *)&high_ip_address_array[1], (int *)&high_ip_address_array[2], (int *)&high_ip_address_array[3]) ;

	   low_ip_address = net_to_host_long ( *((ULONG *)&low_ip_address_array[0])) ;
	   high_ip_address = net_to_host_long ( *((ULONG *)&high_ip_address_array[0])) ;

	   if ((ip_add_entry_to_src_filtering_database (port_number, low_ip_address, high_ip_address, IP_FORWARD_ON_SOURCE_RANGE) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;
      cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_dest_range_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
	ULONG low_ip_address, high_ip_address ;
	BYTE low_ip_address_array[10], high_ip_address_array[10] ;

   sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;
   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
	   sscanf (cptr_port_number_ip_address_string, "%03ju.%03ju.%03ju.%03ju,%03ju.%03ju.%03ju.%03ju",
			   (int *)&low_ip_address_array[0], (int *)&low_ip_address_array[1], (int *)&low_ip_address_array[2], (int *)&low_ip_address_array[3],
			   (int *)&high_ip_address_array[0], (int *)&high_ip_address_array[1], (int *)&high_ip_address_array[2], (int *)&high_ip_address_array[3]) ;

	   low_ip_address = net_to_host_long ( *((ULONG *)&low_ip_address_array[0])) ;
	   high_ip_address = net_to_host_long ( *((ULONG *)&high_ip_address_array[0])) ;

	   if ((ip_add_entry_to_dest_filtering_database (port_number, low_ip_address, high_ip_address, IP_FORWARD_ON_DESTINATION_RANGE) == FAIL))
		   return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;
      cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') && (*cptr_port_number_ip_address_string != 0))
         cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
      cptr_port_number_ip_address_string++ ;
   }

	return (PASS) ;
}


enum TEST ip_src_port_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
   USHORT protocol_port ;
   char protocol[10] ;
   BYTE protocol_type = 0 ;

	sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;

   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%3s,%hu",
              protocol, &protocol_port) ;

      if (strcmp (protocol, tcp_string) == 0)
         protocol_type = TCP_PROTOCOL ;
      else
         if (strcmp (protocol, udp_string) == 0)
            protocol_type = UDP_PROTOCOL ;
         else
         {
            ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : Cannot recognize protocol %s", protocol) ;
            return (PASS) ;
         }

      if ((ip_add_entry_to_src_port_filtering_database (port_number, protocol_port, protocol_port,
           protocol_type, IP_FILTER_ON_SOURCE_PORT) == FAIL))
		         return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
   }

	return (PASS) ;
}


enum TEST ip_dest_port_filter_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
   USHORT protocol_port ;
   char protocol[10] ;
   BYTE protocol_type = 0 ;

	sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;

   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%3s,%hu",
              protocol, &protocol_port) ;

      if (strcmp (protocol, tcp_string) == 0)
         protocol_type = TCP_PROTOCOL ;
      else
         if (strcmp (protocol, udp_string) == 0)
            protocol_type = UDP_PROTOCOL ;
         else
         {
            ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : Cannot recognize protocol %s", protocol) ;
            return (PASS) ;
         }

      if ((ip_add_entry_to_dest_port_filtering_database (port_number, protocol_port, protocol_port,
           protocol_type, IP_FILTER_ON_DESTINATION_PORT) == FAIL))
		         return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
   }

	return (PASS) ;
}


enum TEST ip_src_port_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
   USHORT protocol_port ;
   char protocol[10] ;
   BYTE protocol_type = 0 ;

	sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;

   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%3s,%hu",
              protocol, &protocol_port) ;

      if (strcmp (protocol, tcp_string) == 0)
         protocol_type = TCP_PROTOCOL ;
      else
         if (strcmp (protocol, udp_string) == 0)
            protocol_type = UDP_PROTOCOL ;
         else
         {
            ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : Cannot recognize protocol %s", protocol) ;
            return (PASS) ;
         }

      if ((ip_add_entry_to_src_port_filtering_database (port_number, protocol_port, protocol_port,
           protocol_type, IP_FORWARD_ON_SOURCE_PORT) == FAIL))
		         return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
   }

	return (PASS) ;
}


enum TEST ip_dest_port_forward_cfg_string (char *cptr_port_number_ip_address_string)
{
	USHORT port_number ;
   USHORT protocol_port ;
   char protocol[10] ;
   BYTE protocol_type = 0 ;

	sscanf (cptr_port_number_ip_address_string, "%02hu", &port_number) ;

   while (*cptr_port_number_ip_address_string != ',')
      cptr_port_number_ip_address_string++ ;
   cptr_port_number_ip_address_string++ ;

   while (1)
   {
      sscanf (cptr_port_number_ip_address_string, "%3s,%hu",
              protocol, &protocol_port) ;

      if (strcmp (protocol, tcp_string) == 0)
         protocol_type = TCP_PROTOCOL ;
      else
         if (strcmp (protocol, udp_string) == 0)
            protocol_type = UDP_PROTOCOL ;
         else
         {
            ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : Cannot recognize protocol %s", protocol) ;
            return (PASS) ;
         }

      if ((ip_add_entry_to_dest_port_filtering_database (port_number, protocol_port, protocol_port,
           protocol_type, IP_FORWARD_ON_DESTINATION_PORT) == FAIL))
		         return (PASS) ;

      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;
      while ((*cptr_port_number_ip_address_string != ',') &&
             (*cptr_port_number_ip_address_string != 0))
               cptr_port_number_ip_address_string++ ;
               cptr_port_number_ip_address_string++ ;

      if (*cptr_port_number_ip_address_string == 0)
         break ;
   }

	return (PASS) ;
}


IP_FILTERING_ENTRY *ip_get_src_filtering_entry (ULONG ip_address, USHORT protocol_port, BYTE protocol)
{
	IP_FILTERING_ENTRY *next_entry = sptr_ip_src_filter_chain ;

	if (ip_filtering_inited == FALSE)
	{
		return (NULL) ;
	}

	while (next_entry != NULL)
	{
		switch (next_entry->type)
		{
			case (IP_FILTER_ON_SOURCE_ADDRESS) :
			case (IP_FORWARD_ON_SOURCE_ADDRESS) :
				if (ip_address == next_entry->lower_end.ip_address)
				{
					return (next_entry) ;
				}
				break ;

         case (IP_FILTER_ON_SOURCE_PORT) :
         case (IP_FORWARD_ON_SOURCE_PORT) :
            if ((protocol_port == next_entry->lower_end.protocol_port.port_number) &&
                (protocol == next_entry->lower_end.protocol_port.protocol))
            {
               return (next_entry) ;
            }
            break ;

			case (IP_FILTER_ON_SOURCE_RANGE) :
			case (IP_FORWARD_ON_SOURCE_RANGE) :
				if ((ip_address >= next_entry->lower_end.ip_address) &&
				    (ip_address <= next_entry->higher_end.ip_address))
				{
					 	return (next_entry) ;
				}
				break ;

			default :
				break ;
		}
		next_entry = next_entry->forward_link ;
	}

	return (NULL) ;
}


IP_FILTERING_ENTRY *ip_get_dest_filtering_entry (ULONG ip_address, USHORT protocol_port, BYTE protocol)
{
	IP_FILTERING_ENTRY *next_entry = sptr_ip_dest_filter_chain ;

	if (ip_filtering_inited == FALSE)
		return (NULL) ;

	while (next_entry != NULL)
	{
		switch (next_entry->type)
		{
			case (IP_FILTER_ON_DESTINATION_ADDRESS) :
			case (IP_FORWARD_ON_DESTINATION_ADDRESS) :
				if (ip_address == next_entry->lower_end.ip_address)
					return (next_entry) ;
				break ;

         case (IP_FILTER_ON_DESTINATION_PORT) :
         case (IP_FORWARD_ON_DESTINATION_PORT) :
            if ((protocol_port == next_entry->lower_end.protocol_port.port_number) &&
                (protocol == next_entry->lower_end.protocol_port.protocol))
            {
               return (next_entry) ;
            }
            break ;

			case (IP_FILTER_ON_DESTINATION_RANGE) :
			case (IP_FORWARD_ON_DESTINATION_RANGE) :
				if ((ip_address >= next_entry->lower_end.ip_address) &&
				    (ip_address <= next_entry->higher_end.ip_address))
					 	return (next_entry) ;
				break ;

			default :
				break ;
		}
		next_entry = next_entry->forward_link ;
	}

	return (NULL) ;
}


enum TEST ip_icmp_filter (char *cptr_port_number_and_string)
{
	USHORT port_number ;
	BYTE *temp_pointer ;
	int i ;
	enum BOOLEAN flag ;

	sscanf (cptr_port_number_and_string, "%02hu", &port_number) ;
	temp_pointer = cptr_port_number_and_string ;

	for (i = 0 ; i < NUMBER_OF_RECOGNIZED_ICMP_PACKET_TYPES ; i++)
	{
		temp_pointer = strstr (temp_pointer, ",") ;
		if (temp_pointer)
			temp_pointer++ ;
		if (strncmp (temp_pointer, "enable", 6) == 0)
			flag = TRUE ;
		else
			flag = FALSE ;

		ip.port[port_number].icmp_packet_type_filters[i] = flag ;
	}
}


#ifdef PRINTF

void display_ip_filtering_table ()
{
	int i, temp1[4], temp2[4] ;
	USHORT no_of_entries_counted = ip_number_of_filter_entries ;

	IP_FILTERING_ENTRY *first_entry = sptr_ip_src_filter_chain ;

	if (ip_filtering_inited == FALSE)
	{
		ip_printf (IP_FILTER_PRINTF, "\nIP Filtering : Filtering not initialized") ;
		return ;
	}

	ip_printf (IP_FILTER_PRINTF, "\nDisplaying IP Filtering configuration.....") ;
	ip_printf (IP_FILTER_PRINTF, "\nNumber of source filters : %d", ip_number_of_src_filter_entries) ;
   ip_printf (IP_FILTER_PRINTF, "\nNumber of destination filters : %d", ip_number_of_dest_filter_entries) ;
	ip_printf (IP_FILTER_PRINTF, "\nIP Filtering default transmit action : %s",
	        (ip_filter_default_tx_action == IP_FILTER_DEFAULT_ACTION_FILTER) ? "Filter" : "Forward") ;
	ip_printf (IP_FILTER_PRINTF, "\nIP Filtering default receive action : %s\n",
	        (ip_filter_default_rx_action == IP_FILTER_DEFAULT_ACTION_FILTER) ? "Filter" : "Forward") ;


	for (i = 0 ; i < 2 ; i++)
	{
      if (i == 0)
         ip_printf (IP_FILTER_PRINTF, "\nSource filter chain.....") ;
      else
         ip_printf (IP_FILTER_PRINTF, "\n\nDestination filter chain.....") ;

		while (first_entry != NULL)
		{
         if (
               (first_entry->type != IP_FILTER_ON_SOURCE_PORT) &&
               (first_entry->type != IP_FORWARD_ON_SOURCE_PORT) &&
               (first_entry->type != IP_FILTER_ON_DESTINATION_PORT) &&
               (first_entry->type != IP_FORWARD_ON_DESTINATION_PORT)
            )
         {
			temp1[0] = *(((BYTE *)(&first_entry->lower_end.ip_address))+0) ;
			temp1[1] = *(((BYTE *)(&first_entry->lower_end.ip_address))+1) ;
			temp1[2] = *(((BYTE *)(&first_entry->lower_end.ip_address))+2) ;
			temp1[3] = *(((BYTE *)(&first_entry->lower_end.ip_address))+3) ;

			temp2[0] = *(((BYTE *)(&first_entry->higher_end.ip_address))+0) ;
			temp2[1] = *(((BYTE *)(&first_entry->higher_end.ip_address))+1) ;
			temp2[2] = *(((BYTE *)(&first_entry->higher_end.ip_address))+2) ;
			temp2[3] = *(((BYTE *)(&first_entry->higher_end.ip_address))+3) ;

			ip_printf (IP_FILTER_PRINTF, "\nIP Filter : %d.%d.%d.%d to %d.%d.%d.%d, port %d",
			        temp1[0], temp1[1], temp1[2], temp1[3],
			        temp2[0], temp2[1], temp2[2], temp2[3], first_entry->port) ;
         }
			switch (first_entry->type)
			{
				case (IP_FILTER_ON_SOURCE_ADDRESS) :
					ip_printf (IP_FILTER_PRINTF, ", type : Filter on src. addr.") ;
					break ;

				case (IP_FILTER_ON_DESTINATION_ADDRESS) :
					ip_printf (IP_FILTER_PRINTF, ", type : Filter on dest. addr.") ;
					break ;

				case (IP_FILTER_ON_DESTINATION_RANGE) :
					ip_printf (IP_FILTER_PRINTF, ", type : Filter on dest. range") ;
					break ;

				case (IP_FILTER_ON_SOURCE_RANGE) :
					ip_printf (IP_FILTER_PRINTF, ", type : Filter on src. range") ;
					break ;

				case (IP_FORWARD_ON_SOURCE_ADDRESS) :
					ip_printf (IP_FILTER_PRINTF, ", type : Forward on src. addr.") ;
					break ;

				case (IP_FORWARD_ON_DESTINATION_ADDRESS) :
					ip_printf (IP_FILTER_PRINTF, ", type : Forward on dest. addr.") ;
					break ;

				case (IP_FORWARD_ON_DESTINATION_RANGE) :
					ip_printf (IP_FILTER_PRINTF, ", type : Forward on dest. range") ;
					break ;

				case (IP_FORWARD_ON_SOURCE_RANGE) :
					ip_printf (IP_FILTER_PRINTF, ", type : Forward on src. range") ;
					break ;

            case (IP_FILTER_ON_SOURCE_PORT) :
               ip_printf (IP_FILTER_PRINTF, "\nIP Filter : Proto. port %d, Proto. %s, Port %d, type : Filter on src. port",
                          first_entry->lower_end.protocol_port.port_number,
                          (first_entry->lower_end.protocol_port.protocol == TCP_PROTOCOL) ? tcp_string : udp_string,
                          first_entry->port) ;
               break ;

            case (IP_FILTER_ON_DESTINATION_PORT) :
               ip_printf (IP_FILTER_PRINTF, "\nIP Filter : Proto. port %d, Proto. %s, Port %d, type : Filter on dest. port",
                          first_entry->lower_end.protocol_port.port_number,
                          (first_entry->lower_end.protocol_port.protocol == TCP_PROTOCOL) ? tcp_string : udp_string,
                          first_entry->port) ;
               break ;

            case (IP_FORWARD_ON_SOURCE_PORT) :
               ip_printf (IP_FILTER_PRINTF, "\nIP Filter : Proto. port %d, Proto. %s, Port %d, type : Forward on src. port",
                          first_entry->lower_end.protocol_port.port_number,
                          (first_entry->lower_end.protocol_port.protocol==TCP_PROTOCOL) ? tcp_string : udp_string,
                          first_entry->port) ;
               break ;

            case (IP_FORWARD_ON_DESTINATION_PORT) :
               ip_printf (IP_FILTER_PRINTF, "\nIP Filter : Proto. port %d, Proto. %s, Port %d, type : Forward on dest. port",
                          first_entry->lower_end.protocol_port.port_number,
                          (first_entry->lower_end.protocol_port.protocol==TCP_PROTOCOL) ? tcp_string : udp_string,
                          first_entry->port) ;
               break ;

            default :
               ip_printf (IP_FILTER_PRINTF, ", type : UNKNOWN") ;
			}
			no_of_entries_counted-- ;
			if (no_of_entries_counted == 0)
				return ;

			first_entry = first_entry->forward_link ;
		}
		first_entry = sptr_ip_dest_filter_chain ;
	}
}


#endif

#endif
