/*
** STSEND.C -- Code related to internally generated sends when errors are
**					encountered or otherwise in the TCP state machine processing.
*/

#include "rtrstd.h"

#include "all.h"
#include "tcpextrn.h"

/*
** send_tcp_control_seg()
**		Format and send a TCP packet with the appropriate code bits set. 
**		This packet should have a sequence number and an ack number as 
**		specified. Also indicate the proper code bits to set.
**	Params:
**		Ptr to connection record for connection to send on
**		Sequence number to use
**		Ack number to use
**		Code field bits to set
** NOTES:
**		In case FIN and SYN control bits are to be used, the caller of this
**			function should take care of updating the sequence numbers to
**			cover these. Also FIN and SYN segment should be added to the
**			retransmit queue at the calling point immediately after this call.
**		No support provided for URG -- this function is for FIN, SYN, ACK and
**			RST.
*/
void 
send_tcp_control_seg(TCP_PER_CONN *ptr_conn_info, TCPSEQ seq_num, 
		TCPSEQ ack_num, USHORT code_bits)
{
	BYTE *ptcpbuf;
	USHORT seg_len;
	TCP_OPTIONS *popt;
	TCP_HEADER tcp_header;


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Control segment requested...");
	tcp_header.code_and_hlen.code_and_hdrlen_word = 0;
	tcp_header.code_and_hlen.code_and_hdrlen.codes = code_bits;
	if (IS_ACK_ON(&tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " ACK");
	if (IS_SYN_ON(&tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " SYN");
	if (IS_FIN_ON(&tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " FIN");
	if (IS_RST_ON(&tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " RST");
	if (IS_ACK_ON(&tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, ", ACK ack no. = %d", ack_num);
	tcp_printf(TCP_DEBUG_PRINTF, "\n");
#endif /* DEBUG */

	seg_len = sizeof(TCP_HEADER);
	if (code_bits & SYN)
	{
		/* For the MSS option -- only thing we use */
		seg_len += sizeof(TCP_MSS_OPTION) + 1;
	}

	ptcpbuf = tcp_get_user_buffer(seg_len);
	if (ptcpbuf == NULL)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to send a control segment due to lack of buffers\n");
		return;
	}

	/* Fill in the header */
	tcp_header.src_port = ptr_conn_info->local_port;
	tcp_header.dst_port = ptr_conn_info->remote_port;
	tcp_header.seq_num = seq_num;
	tcp_header.ack_num = ack_num;
	tcp_header.code_and_hlen.code_and_hdrlen_word = 0;
	tcp_header.code_and_hlen.code_and_hdrlen.hlen = sizeof(TCP_HEADER) >> 2;
	if (code_bits & SYN)
	{
		tcp_header.code_and_hlen.code_and_hdrlen.hlen += 1;
	}
	tcp_header.code_and_hlen.code_and_hdrlen.codes = code_bits;
	if ((IS_RST_ON(&tcp_header) == 0) && (IS_ACK_ON(&tcp_header) != 0))
	{
		ptr_conn_info->conn_flags &= ~TCPF_NEEDACK;
	}
	if (ptr_conn_info != NULL)
		tcp_header.win_size = tcp_recv_wnd(ptr_conn_info);
	else
		tcp_header.win_size = tcp.recv_win_size;
	tcp_header.urg_ptr = 0;

	if (IS_RST_ON(&tcp_header))
	{
		++tcp.mib.tcpOutRsts;
	}

	tcp_hdr_hton((TCP_HEADER *)ptcpbuf, &tcp_header);
	/* Set the MSS option on a SYN segment */
	if (code_bits & SYN)
	{
		popt = (TCP_OPTIONS *)(ptcpbuf + sizeof(TCP_HEADER));
		popt->kind = KIND_MSS;
		popt->opt_union.opt_MSS.len = sizeof(TCP_MSS_OPTION) + 1;
		popt->opt_union.opt_MSS.max_seg_size = host_to_net_short(tcp.max_seg_size);
	}

/* sudhir 28/2/97 */
	send_tcp_packet(ptr_conn_info->seg_ip_params.source_address, 
			ptr_conn_info->seg_ip_params.destination_address, 
			0, ptcpbuf, 
			seg_len, FALSE);
}

/*
** tcp_reset()
** 	Called only when a packet is received and there are no valid 
**		connection records for the packet that came in. 
** Params:
**		Ptr to the incoming tcp segment header in host format
** NOTES:
**		Do not use this function anywhere else.
*/
void
tcp_reset(TCP_HEADER *tcp_header, USHORT tcp_packet_len, IP_INFO *ip_info)
{
	BYTE *ptcpbuf;
	TCP_HEADER out_tcp_header;


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Reset requested\n");
#endif /* DEBUG */

	ptcpbuf = tcp_get_user_buffer(sizeof(TCP_HEADER));
	if (ptcpbuf == NULL)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to send a control segment due to lack of buffers\n");
		return;
	}

	out_tcp_header.dst_port = tcp_header->src_port;
	out_tcp_header.src_port = tcp_header->dst_port;
	if (IS_ACK_ON(tcp_header))
	{
		out_tcp_header.seq_num = tcp_header->ack_num;
	}
	else
	{
		out_tcp_header.seq_num = 0;
	}
	out_tcp_header.ack_num = tcp_header->seq_num + tcp_packet_len - 
			sizeof(TCP_HEADER);
	out_tcp_header.code_and_hlen.code_and_hdrlen_word = 0;
	out_tcp_header.code_and_hlen.code_and_hdrlen.hlen = 
			sizeof(TCP_HEADER) / 4;
	TURN_RST_ON(&out_tcp_header);

	out_tcp_header.win_size = 0;
	out_tcp_header.urg_ptr = 0;

	tcp_hdr_hton((TCP_HEADER *)ptcpbuf, &out_tcp_header);

	send_tcp_packet(ip_info->destination_address, ip_info->source_address,
			DEF_TOS, ptcpbuf, sizeof(TCP_HEADER), FALSE);
}


