/*
** TCPINIT.C -- Has routines called by the routerware LSL init routines as
**					 part of the MTR startup process. Code here will init stuff
**					 related to TCP. Also some control and de-init code is
**					 present.
*/

#include <stdlib.h>
#include <string.h>

#include "rtrstd.h"
#include "all.h"

#include "tcpglob.h"
#include "vnvtcp.h"

extern IPADDR get_ip_address(USHORT);


/* Local prototypes */
void readjust_config(void);
enum TEST tcp_control(enum TRANSPORT_CONTROL_OPERATION, ULONG, ULONG);
void tcp_initialize_socket_protocol_stack_interface(TRANSPORT_INTERFACE *);
void tcp_free_conn(TCP_PER_CONN *);



/*
** initialize_tcp()
**		This function is called from the RouterWare initialization module.
** Params:
**		Clock ticks per second.
*/
enum TEST
initialize_tcp(ULONG clock_ticks_per_second)
{
  	USHORT i;
	TCP_PER_CONN *ptr_conn_info;
	USHORT port_mtu, min_mtu, number_of_physical_ports;


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: TCP initializing...\n");
#endif /* DEBUG */

	if (tcp.enabled == FALSE)
		return PASS;


	tcp.clock_ticks_per_second = clock_ticks_per_second;
	memset(&tcp.mib, 0, sizeof(TCP_MIB));
	memset(&tcp.stats, 0, sizeof(TCP_STATISTICS));

	/* ?? Change the 0 parameter below to something else ?? */
	tcp.local_ip_address = get_ip_address(0);
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Host address = %s\n",
		convert_ip_address_to_dot_format(ip.print_buffer,
				tcp.local_ip_address));
#endif /* DEBUG */

	/* Readjust config info to suit clock_ticks_per_second */
	readjust_config();

	/* Allocate memory for the connection records table */
	if ((tcp.tcp_conn_table = (TCP_PER_CONN *) table_malloc(
			tcp.max_tcp_connection, sizeof(TCP_PER_CONN))) == NULL)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to allocate memory for TCP connection records\n");
		return FAIL;
	}
	ptr_conn_info = tcp.tcp_conn_table;
	for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
	{
		ptr_conn_info->tcp_state = FREE;
	}
	
	/* Send sequence number generator clock */
	tcp.isn_clk_time = 0;

	/* Ephemeral port numbers start from 32768 */
	tcp.next_user_port = 32768;

	/* Register TCP stack with LSL */
#ifdef __LSL__

	if ((enum TEST) lsl_control(REGISTER_TRANSPORT, "TCP Transport", 
			TCP_TRANSPORT, tcp_timer, tcp_control, &tcp.transport_id) == FAIL)
	{
		return FAIL;
	}

#endif /* __LSL__ */

	/* Init some constant SNMP MIB variables */
	tcp.mib.tcpRtoAlgorithm = 4;		/* Van Jacobsons Method is used */
	tcp.mib.tcpMaxConn = tcp.max_tcp_connection;
	tcp.mib.tcpActiveOpens = 0;

	/* Determine the largest MSS that we can accomodate */
	number_of_physical_ports = lsl_control(GET_NUMBER_OF_LAN_PORTS) +
									 		lsl_control(GET_NUMBER_OF_WAN_PORTS);
	min_mtu = 0xFFFF;
	for (i = 0; i < number_of_physical_ports; i++)
	{
		port_mtu = lsl_control(LSL_GET_PORT_MTU, i);
		if (port_mtu < min_mtu)
			min_mtu = port_mtu;
	}
	tcp.max_send_mss = min_mtu - sizeof(TCP_HEADER) - sizeof(IP_HEADER) - sizeof(MAC_HEADER);
	if (tcp.max_send_mss < DEF_TCP_MSS)
		tcp.max_send_mss = DEF_TCP_MSS;

	tcp.timer_enabled = TRUE;

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Initialize successful\n");
	/* Print out standard values */
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Connections = %lu\n", tcp.max_tcp_connection);
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: tcpRtoMin = %lu\n", tcp.min_rtx_timeout);
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: tcpRtoMax = %lu\n", tcp.max_rtx_timeout);
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: 2MSL = %lu\n", tcp.two_msl);
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: ACK delay = %lu\n", tcp.max_ack_delay);
#endif /* DEBUG */

	return PASS;
}

/*
** readjust_config()
**		Called to readjust the configuration info as picked up from the
**		config file to suit usage in code.
** NOTES:
**		All times in config are in milliseconds. These are converted to
**			tick values. Also MSL is stored again as 2 * MSL.
*/
void
readjust_config()
{
	/* Init some constant SNMP MIB variables */
	tcp.mib.tcpRtoMin = tcp.min_rtx_timeout;
	tcp.mib.tcpRtoMax = tcp.max_rtx_timeout;

	tcp.two_msl = 2 * ((tcp.two_msl * tcp.clock_ticks_per_second) / 1000);
	tcp.max_rtx_timeout = (tcp.max_rtx_timeout * tcp.clock_ticks_per_second) /
																						1000;
	tcp.min_rtx_timeout = (tcp.min_rtx_timeout * tcp.clock_ticks_per_second) /
																						1000;
	tcp.max_ack_delay = (tcp.max_ack_delay * tcp.clock_ticks_per_second) / 
																						1000;
	if (tcp.conn_idle_time != 0xFFFFFFFF)
	{
		tcp.conn_idle_time = (tcp.conn_idle_time * tcp.clock_ticks_per_second) / 
																						1000;
	}
}

/*
** tcp_control()
** 	Control called by LSL to provide a sort of interface between the TCP
**		layer and other layers above or below. We support 3 of the 5 control
**		commands, one of which will init the socket library interface.
** Params:
**		Control command
**		Param1 & Param2 - based on the command, cast to desired type
** Returns:
**		PASS if no errors, FAIL otherwise
*/
enum TEST
tcp_control(enum TRANSPORT_CONTROL_OPERATION command, ULONG param_0, 
		ULONG param_1)
{
	USHORT i;
	TCP_PER_CONN *ptr_conn_info;
	TRANSPORT_INTERFACE *sptr_transport_interface;


	PARAMETER_NOT_USED(param_0);

#if defined(DEBUG)
	{
		char *cmd_str;

		switch (command)
		{
		default:
			cmd_str = "UNKNOWN";
			break;
		case OPEN_TRANSPORT:
			cmd_str = "OPEN_TRANSPORT";
			break;
		case CLOSE_TRANSPORT:
			cmd_str = "CLOSE_TRANSPORT";
			break;
		case IS_TRANSPORT_ENABLED:
			cmd_str = "IS_TRANSPORT_ENABLED";
			break;
		case INITIALIZE_SOCKET_TRANSPORT_INTERFACE:
			cmd_str = "INITIALIZE_SOCKET_TRANSPORT_INTERFACE";
			break;
		case INITIALIZE_TRANSPORT_FUNCTION_POINTERS:
			cmd_str = "INITIALIZE_TRANSPORT_FUNCTION_POINTERS";
			break;
		}
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: TCP control function called for %s(%d) function\n",
			cmd_str, command);
	}
#endif /* DEBUG */

	switch (command)
	{
	default:
		break;
	case OPEN_TRANSPORT:
		return PASS;
	case CLOSE_TRANSPORT:

		/* Stop timer */
		tcp.timer_enabled = FALSE;

		/* Close all pending connections and free connection related buffers */

		ptr_conn_info = tcp.tcp_conn_table;
		for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
		{
			if (ptr_conn_info->tcp_state == FREE)
				continue;

			tcp_free_conn(ptr_conn_info);
		}

		/* Free connection buffer */
		table_free((void *)tcp.tcp_conn_table);
		return PASS;
	case IS_TRANSPORT_ENABLED:
		if (tcp.enabled == TRUE)
		{
			*(enum BOOLEAN *) param_1 = TRUE;
			return PASS;
		}
		else
		{
			*(enum BOOLEAN *) param_1 = FALSE;
			return FAIL;
		}
	case INITIALIZE_SOCKET_TRANSPORT_INTERFACE:
		sptr_transport_interface = (TRANSPORT_INTERFACE *)param_1;
		tcp_initialize_socket_protocol_stack_interface(sptr_transport_interface);

		return PASS;
	case INITIALIZE_TRANSPORT_FUNCTION_POINTERS:
		if ((enum TEST) lsl_control(INITIALIZE_SOCKET_FUNCTION_POINTERS,
			SOCKETS_INTERFACE, INITIALIZE_FUNCTION_POINTERS, 
			TRANSPORT_LAYER_TYPE, &tcp.socket_fptrs) == FAIL)
		{
			tcp_printf(TCP_PRINTF, "TCP: Failed to initialize socket related function pointers\n");
			return FAIL;
		}
		return PASS;
	}
	return FAIL;
}

/*
**	tcp_initialize_socket_protocol_stack_interface()
**		Called to initialize pointers to the socket interface entry points.
** Params:
**		Ptr to a socket library structure that will hold the pointers.
** NOTES:
**		TCP software supports only the following functions
**				socket_tcp()
**				socket_tcp_listen()
**				socket_tcp_connect()
**				socket_tcp_receive() 
**				socket_tcp_send() 
**				socket_tcp_close()
**				socket_tcp_status()
**				socket_tcp_get_state_string()
**				tcp_check_ip_address()
**		accept() can be called inspite of this.
*/
void
tcp_initialize_socket_protocol_stack_interface(TRANSPORT_INTERFACE *
									sptr_transport_interface)
{
	sptr_transport_interface->socket =
		(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *, USHORT))	socket_tcp;

	sptr_transport_interface->bind = NULL;
/* 	(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *)) socket_tcp_bind;
*/
	sptr_transport_interface->listen = 
		(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *, USHORT)) socket_tcp_listen;
		
	sptr_transport_interface->connect = 
		(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *)) socket_tcp_connect;

	sptr_transport_interface->receive =
		(USHORT (*) (struct USER_SOCKET *, void *, USHORT, SOCKADDR	*, 
		USHORT *, enum TEST *)) socket_tcp_receive;

	sptr_transport_interface->send =
		(USHORT (*) (struct USER_SOCKET *, void *, USHORT,
		SOCKADDR *, enum TEST *)) socket_tcp_send;

	sptr_transport_interface->queue_length = NULL;
/*		(USHORT (*) (struct USER_SOCKET *, USHORT, enum TEST *))
**		socket_tcp_queue_length;
*/
	sptr_transport_interface->close = 
		(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *)) socket_tcp_close;

	sptr_transport_interface->status = 
		(BYTE_ENUM (TEST) (*) (struct USER_SOCKET *)) socket_tcp_status;

	sptr_transport_interface->check = 
		(BYTE_ENUM (TEST) (*) (SOCKADDR *, USHORT)) tcp_check_ip_address;
}

/*
** tcp_free_conn()
** 	Called to clean up the session, but will not interact with
**		any other layer in the process.
** Params:
**		Ptr to connection record
**	NOTE:
**		Assume connection is valid
*/
void
tcp_free_conn(TCP_PER_CONN *ptr_conn_info)
{
	/* De-couple any links with socket entries */
	if (ptr_conn_info->socket_record_ptr != NULL)
		ptr_conn_info->socket_record_ptr->vptr_protocol_control_block = NULL;

	free_tcp_conn_record(ptr_conn_info);
}

