/*--------------------------------------------------------------------------*/
/*	File		:	CODE.C													*/
/*	Purpose		:	This file contains code generation routines.			*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#include	"windows.h"

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"
#include	"scrtdata.h"
#include	"scrtfuns.h"

#include	<string.h>
#include	<stdlib.h>
#include	<stdio.h>

static	CodeTypeFptr	CodeList;	/*	Head of available code nodes.	*/
static	CodeTypeFptr	CodeHeadPtr;	/*	Address of code space.	*/
#ifdef WIN
static	HANDLE		hCodeSpace;
#endif

int    CodeSize ;
/*--------------------------------------------------------------------------*/
/*	Name		:	EmitExpression											*/
/*	Input		:	Pointer to root of expression tree.						*/
/*	Output		:	Name of temporary where result is available.			*/
/*	Synopsis	:	Prints the expression in postfix form.					*/
/*--------------------------------------------------------------------------*/

byte	*EmitExpression(ExprNodeType *node, CodeTypeFptr *c_ptr)
{
	SymInfoType	*sym_ptr;
	DefnArgType	*formal_arg_ptr;
	CallArgType	*arg_ptr;
	CodeTypeFptr	c_ptr1;
	CodeTypeFptr	c_ptr2;
	int		operator;
	int		arg_count;
	byte	*name1;
	byte	*name2;
	byte	*name3;
	byte	temp_str[10];	/*	To convert constants to string.	*/

	name3 = 0;
	c_ptr1 = c_ptr2 = 0;
	if (node) {
		switch (node->TokClass) {
			case Q_EXPRESSION :
				operator = node->Operator;
				switch (operator) {
					case Q_PLUS :
					case Q_MINUS :
					case Q_TIMES :
					case Q_DIV :
					case Q_LESS_OR_EQUAL :
					case Q_GREATER_OR_EQUAL :
					case Q_LESS :
					case Q_GREATER :
					case Q_IS_EQUAL :
					case Q_IS_NOT_EQUAL :
					case Q_AND :
					case Q_OR :
						name1 = EmitExpression(node->Llink, &c_ptr1);
						name2 = EmitExpression(node->Rlink, &c_ptr2);
						GenerateTempVar(node->Name);
						name3 = node->Name;
						c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
						c_ptr2 = EmitCode(operator,
									node->Llink->ExprType, name1,
									node->Rlink->ExprType, name2,
									name3);
						*c_ptr = MergeCodes(c_ptr1, c_ptr2);
						break;
					case Q_NOT :
					case Q_UNARY_MINUS :
						name1 = EmitExpression(node->Rlink, &c_ptr1);
						GenerateTempVar(node->Name);
						name3 = node->Name;
						c_ptr2 = EmitCode(operator,
									node->ExprType, name1, 0, 0, name3);
						*c_ptr = MergeCodes(c_ptr1, c_ptr2);
						break;
					case Q_LP :
						name1 = EmitExpression(node->Rlink, &c_ptr1);
						GenerateTempVar(node->Name);
						name3 = node->Name;
						c_ptr2 = EmitCode(Q_MOV, node->ExprType, name1,
									node->ExprType, 0, name3);
						*c_ptr = MergeCodes(c_ptr1, c_ptr2);
						break;
					case Q_ID :
					case Q_INT_CONST :
					case Q_REAL_CONST :
					case Q_STRING_CONST :
						name1 = node->Rlink->Name;
						GenerateTempVar(node->Name);
						name3 = node->Name;
						*c_ptr = EmitCode(Q_MOV, node->ExprType, name1,
									node->ExprType, 0, name3);
						if (*c_ptr) {
							switch (operator) {
								case Q_INT_CONST :
									(*c_ptr)->Type1 = E_INT_CONST;
									break;
								case Q_REAL_CONST :
									(*c_ptr)->Type1 = E_REAL_CONST;
									break;
								case Q_STRING_CONST :
									(*c_ptr)->Type1 = E_STRING_CONST;
									break;
							}
						}
						break;
					case Q_FN_CALL :
						sym_ptr = FindSymbol(node->Rlink->Name);
						formal_arg_ptr = 0;
						if (sym_ptr && sym_ptr->ProcInfoPtr)
							formal_arg_ptr = sym_ptr->ProcInfoPtr->ArgList;
						arg_count = 0;
						arg_ptr = node->ArgList;
						while (arg_ptr) {
							/*	If the argument is passed by reference,	*/
							/*	push the argument name itself instead of	*/
							/*	the temporary into which it is moved.	*/
							if (formal_arg_ptr &&
									formal_arg_ptr->Attribute == Q_VAR &&
									(arg_ptr->ExprPtr->Attribute & E_LVALUE))
								name1 = arg_ptr->ExprPtr->Name;
							else {
								name1 = EmitExpression(arg_ptr->ExprPtr,
																	&c_ptr2);
								c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
							}
							c_ptr2 = EmitCode(Q_PUSH,
								arg_ptr->ExprPtr->ExprType, name1, 0, 0, 0);
							c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
							arg_ptr = arg_ptr->NextArgPtr;
							if (formal_arg_ptr->NextArgPtr)
								formal_arg_ptr = formal_arg_ptr->NextArgPtr;
							arg_count++;
						}
						itoa(arg_count, temp_str, 10);
						c_ptr2 = EmitCode(Q_PUSH, E_INT_CONST, temp_str,
																0, 0, 0);
						c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
						c_ptr2 = EmitCode(Q_CALL, 0, node->Rlink->Name,
																0, 0, 0);
						c_ptr1 = MergeCodes(c_ptr1, c_ptr2);

						if (node->ExprType == E_INTEGER || node->ExprType ==
									E_REAL || node->ExprType == E_STRING) {
							GenerateTempVar(node->Name);
							c_ptr2 = EmitCode(Q_MOV, node->ExprType,
										PreDefVars[RETURN_VARIABLE].VarName,
										node->ExprType, 0, node->Name);
							name3 = node->Name;
							c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
						}
						else
							name3 = 0;
						*c_ptr = c_ptr1;
						break;
					default :
						EmitExpression(node->Llink, c_ptr);
						EmitExpression(node->Rlink, c_ptr);
						PutStr("---CHECK---");
						break;
				}
				break;
			default :
				PutStr("___WRONG___");
				break;
		}
	}
	return(name3);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EmitCode												*/
/*	Input		:	None													*/
/*	Output		:	Pointer to the code node.								*/
/*	Synopsis	:	Returns a pointer to code node.							*/
/*--------------------------------------------------------------------------*/

CodeTypeFptr	EmitCode(int operator, int type1, byte *name1, int type2, byte *name2, byte *name3)
{
	CodeTypeFptr	c_ptr;

	c_ptr = GetCodeNode();
	if (!c_ptr) {
		Error(ERR_OUT_OF_MEMORY);
		goto error_exit;
	}
	c_ptr->Arg1 = c_ptr->Arg2 = c_ptr->Arg3 = 0;
	c_ptr->NextCode = 0;
	c_ptr->Operator = (byte) operator;
	c_ptr->Type1 = (byte) type1;
	c_ptr->Type2 = (byte) type2;
	if (name1) {
#ifndef WIN
		c_ptr->Arg1 = malloc(strlen(name1) + 1);
#else
		c_ptr->Arg1 = (byte *) LocalAlloc(LMEM_FIXED, strlen(name1) + 1);
#endif
		if (!c_ptr->Arg1) {
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
		strcpy(c_ptr->Arg1, name1);
	}
	if (name2) {
#ifndef WIN
		c_ptr->Arg2 = malloc(strlen(name2) + 1);
#else
		c_ptr->Arg2 = (byte *) LocalAlloc(LMEM_FIXED, strlen(name2) + 1);
#endif
		if (!c_ptr->Arg2) {
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
		strcpy(c_ptr->Arg2, name2);
	}
	if (name3) {
#ifndef WIN
		c_ptr->Arg3 = malloc(strlen(name3) + 1);
#else
		c_ptr->Arg3 = (byte *) LocalAlloc(LMEM_FIXED, strlen(name3) + 1);
#endif
		if (!c_ptr->Arg3) {
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
		strcpy(c_ptr->Arg3, name3);
	}
	return(c_ptr);

	error_exit :

	if (c_ptr) {
#ifndef WIN
		if (c_ptr->Arg1)
			free(c_ptr->Arg1);
		if (c_ptr->Arg2)
			free(c_ptr->Arg2);
		if (c_ptr->Arg3)
			free(c_ptr->Arg3);
#else
		if (c_ptr->Arg1)
			LocalFree((byte *) c_ptr->Arg1);
		if (c_ptr->Arg2)
			LocalFree((byte *) c_ptr->Arg2);
		if (c_ptr->Arg3)
			LocalFree((byte *) c_ptr->Arg3);
#endif
		FreeCodeNode(c_ptr);
	}
	return(0);	/*	No memory.	*/
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FreeCodes												*/
/*	Input		:	c_ptr, pointer to list of codes.						*/
/*	Output		:	None													*/
/*	Synopsis	:	Frees the code node pointers.							*/
/*--------------------------------------------------------------------------*/

void	FreeCodes(CodeTypeFptr c_ptr)
{
	CodeTypeFptr	c_ptr1;

	while (c_ptr) {
#ifndef WIN
		if (c_ptr->Arg1)
			free(c_ptr->Arg1);
		if (c_ptr->Arg2)
			free(c_ptr->Arg2);
		if (c_ptr->Arg3)
			free(c_ptr->Arg3);
#else
		if (c_ptr->Arg1)
			LocalFree((byte *) c_ptr->Arg1);
		if (c_ptr->Arg2)
			LocalFree((byte *) c_ptr->Arg2);
		if (c_ptr->Arg3)
			LocalFree((byte *) c_ptr->Arg3);
#endif
		c_ptr1 = c_ptr->NextCode;
		FreeCodeNode(c_ptr);
		c_ptr = c_ptr1;
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	MergeCodes												*/
/*	Input		:	c_ptr1, pointer to first list of codes,					*/
/*					c_ptr2, pointer to second list of codes.				*/
/*	Output		:	Pointer to the merged list of codes.					*/
/*	Synopsis	:	Concatenates the second list of codes to the first		*/
/*						list of codes.										*/
/*--------------------------------------------------------------------------*/

CodeTypeFptr	MergeCodes(CodeTypeFptr c_ptr1, CodeTypeFptr c_ptr2)
{
	CodeTypeFptr	c_ptr;

	c_ptr = c_ptr1;
	if (c_ptr) {
		while (c_ptr->NextCode)
			c_ptr = c_ptr->NextCode;
		c_ptr->NextCode = c_ptr2;
		return(c_ptr1);
	}
	else
		return(c_ptr2);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	PrintCodes												*/
/*	Input		:	c_ptr, pointer to code list.							*/
/*	Output		:	None													*/
/*	Synopsis	:	Prints the list of codes.								*/
/*--------------------------------------------------------------------------*/

void	PrintCodes(CodeTypeFptr c_ptr)
{
	static	byte	*OperatorStrs[] = { "neg", "add", "sub", "mul", "div",
		"lesseq", "greatereq", "equal", "notequal", "less", "greater", "not",
		"", "", "and", "or"
	};
	static	byte	*TypeStrs[] = {"INT", "REAL", "STRING",
		"INTCONST", "REALCONST", "STRINGCONST" "", "", "",
	};
	byte	buffer[4 * MAX_TOKEN_LEN];	/*	To form the psuedo code.	*/

	if (OutFileHandle < 0)
		goto ignore_exit;
	while (c_ptr) {
		buffer[0] = 0;
		switch (c_ptr->Operator) {
			case Q_PLUS :
			case Q_MINUS :
			case Q_TIMES :
			case Q_DIV :
			case Q_LESS_OR_EQUAL :
			case Q_GREATER_OR_EQUAL :
			case Q_LESS :
			case Q_GREATER :
			case Q_IS_EQUAL :
			case Q_IS_NOT_EQUAL :
			case Q_AND :
			case Q_OR :
				sprintf(buffer, "\t%s\t%s %s %s %s %s\r\n",
							OperatorStrs[c_ptr->Operator],
							TypeStrs[c_ptr->Type1], c_ptr->Arg1,
							TypeStrs[c_ptr->Type2], c_ptr->Arg2,
							c_ptr->Arg3);
				break;
			case Q_NOT :
			case Q_UNARY_MINUS :
				sprintf(buffer, "\t%s\t%s %s %s\r\n",
						OperatorStrs[c_ptr->Operator],
						TypeStrs[c_ptr->Type1], c_ptr->Arg1, c_ptr->Arg3);
				break;
			case Q_MOV :
				sprintf(buffer, "\tmov\t%s %s %s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1,
						TypeStrs[c_ptr->Type2], c_ptr->Arg3);
				break;
			case Q_LABEL_DEFN :	/*	Label declaration.	*/
				sprintf(buffer, "%s :\r\n", c_ptr->Arg1);
				break;
			case Q_JMP :	/*	Unconditional jump.	*/
				sprintf(buffer, "\tjmp\t%s\r\n", c_ptr->Arg1);
				break;
			case Q_JZ :	/*	Conditional jump.	*/
				sprintf(buffer, "\tjz\t%s %s %s\r\n", TypeStrs[c_ptr->Type1],
											c_ptr->Arg1, c_ptr->Arg3);
				break;
			case Q_JNZ :	/*	Conditional jump.	*/
				sprintf(buffer, "\tjnz\t%s %s %s\r\n", TypeStrs[c_ptr->Type1],
											c_ptr->Arg1, c_ptr->Arg3);
				break;
			case Q_JEQ :	/*	Jump on equal.	*/
				sprintf(buffer, "\tje\t%s %s %s %s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1,
						TypeStrs[c_ptr->Type2], c_ptr->Arg2,
						c_ptr->Arg3);
				break;
			case Q_JNEQ :	/*	Jump on not equal.	*/
				sprintf(buffer, "\tjne\t%s %s %s %s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1,
						TypeStrs[c_ptr->Type2], c_ptr->Arg2,
						c_ptr->Arg3);
				break;
			case Q_PUSH :
				sprintf(buffer, "\tpush\t%s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1);
				break;
			case Q_CALL :
				if (c_ptr->Arg2)
					sprintf(buffer, "\tcall\t%s %s %s\r\n", c_ptr->Arg1,
							TypeStrs[c_ptr->Type2], c_ptr->Arg2);
				else
					sprintf(buffer, "\tcall\t%s\r\n", c_ptr->Arg1);
				break;
			case Q_RET :
				if (c_ptr->Arg1)
					sprintf(buffer, "\tret\t%s %s\r\n",
							TypeStrs[c_ptr->Type1], c_ptr->Arg1);
				else
					sprintf(buffer, "\tret\r\n");
				break;
			case Q_LINENO :	/*	Psuedocode, line number information.	*/
				sprintf(buffer, "; Line %s\r\n", c_ptr->Arg1);
				break;
			case Q_PROCEDURE :	/*	Procedure beginning.	*/
				sprintf(buffer, "PROC\t%s\r\n", c_ptr->Arg1);
				break;
			case Q_ENDPROCEDURE :	/*	Procedure end.	*/
				sprintf(buffer, "ENDPROC\t%s\r\n", c_ptr->Arg1);
				break;
			case Q_PARAM :	/*	Procedure parameter.	*/
				sprintf(buffer, "\tPARAM\t%s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1);
				break;
			case Q_DECLARE :	/*	Variable declaration.	*/
				sprintf(buffer, "\tDECLARE\t%s %s\r\n",
						TypeStrs[c_ptr->Type1], c_ptr->Arg1);
				break;
			case Q_TEMPCOUNT :	/*	Max temporaries used in fn body.	*/
				sprintf(buffer, "\tTEMPCOUNT\t%s\r\n", c_ptr->Arg1);
				break;
			case Q_KILL :	/*	Kill temporary.	*/
				sprintf(buffer, "\tKILL\t%s\r\n", c_ptr->Arg1);
				break;
		}
		if (FWrite(OutFileHandle, buffer, strlen(buffer)) < 0) {
			/*	Write error, ignore further writing.	*/
			Close(OutFileHandle);
			OutFileHandle = -1;		/*	Invalidate the file handle.	*/
			break;
		}
		c_ptr = c_ptr->NextCode;
	}

	ignore_exit :
		;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	InitCodeSpace											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Allocates memory for code pool and initializes it.		*/
/*--------------------------------------------------------------------------*/

int		InitCodeSpace(void)
{
	#define	CODE_SPACE		65536

	CodeTypeFptr	c_ptr;
	int		idx;

	CodeHeadPtr = 0;
#ifndef WIN
	_asm {
		mov		ax, 4800H
		mov		bx, CODE_SPACE / 16
		int		21H
		jc		NoMem
		mov		Word Ptr CodeHeadPtr, 0
		mov		Word Ptr CodeHeadPtr+2, ax
	NoMem :
	}
#else
	hCodeSpace = GlobalAlloc(GHND, CODE_SPACE);
	if (!hCodeSpace)
		CodeHeadPtr = NULL;
	else {
		CodeHeadPtr = (CodeTypeFptr) GlobalLock(hCodeSpace);
		if (!CodeHeadPtr)
			GlobalFree(hCodeSpace);
	}
#endif

	if (CodeHeadPtr) {
		c_ptr = CodeList = CodeHeadPtr;
		for (idx = 0; idx < CODE_SPACE / sizeof(CodeType) - 1; idx++) {
			c_ptr->NextCode = c_ptr + 1;
			c_ptr++;
		}
		c_ptr->NextCode = 0;	/*	End of list.	*/
		return(0);
	}
	else
		return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FreeCodeSpace											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Frees the code node pointers.							*/
/*--------------------------------------------------------------------------*/

int		FreeCodeSpace(void)
{
	if (CodeHeadPtr) {
#ifndef WIN
		_asm {
			mov		ax, 4900H
			mov		es, Word Ptr CodeHeadPtr+2
			int		21H
		}
#else
		GlobalUnlock(hCodeSpace);
		GlobalFree(hCodeSpace);
		hCodeSpace = NULL;
#endif
		CodeHeadPtr = 0;
	}
	return(0);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	GetCodeNode												*/
/*	Input		:	None													*/
/*	Output		:	pointer to code node if node available, else null.		*/
/*	Synopsis	:	Allocates a code node from the available pool.			*/
/*--------------------------------------------------------------------------*/

CodeTypeFptr	GetCodeNode(void)
{
	CodeTypeFptr	c_ptr;

	c_ptr = 0;
	if (CodeList) {
		c_ptr = CodeList;
		CodeList = CodeList->NextCode;
	}
	return(c_ptr);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FreeCodeNode											*/
/*	Input		:	c_ptr, pointer to code node.							*/
/*	Output		:	None													*/
/*	Synopsis	:	Frees the code node and puts it back in available pool.	*/
/*--------------------------------------------------------------------------*/

void	FreeCodeNode(CodeTypeFptr c_ptr)
{
	c_ptr->NextCode = CodeList;
	CodeList = c_ptr;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	NextCodeNode											*/
/*	Input		:	None													*/
/*	Output		:	c_ptr, pointer to next allocatable code node.			*/
/*	Synopsis	:	Returns the address of the next allocatable code node.	*/
/*--------------------------------------------------------------------------*/

CodeTypeFptr	NextCodeNode(void)
{
	return(CodeList);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	KillTemporaries											*/
/*	Input		:	None													*/
/*	Output		:	0 if successful, else -1.								*/
/*	Synopsis	:	Generates code for killing temporaries.					*/
/*--------------------------------------------------------------------------*/

int		KillTemporaries(CodeTypeFptr *c_ptr)
{
	CodeTypeFptr	c_ptr1;
	CodeTypeFptr	c_ptr2;
	int		count;
	byte	name[MAX_TOKEN_LEN];

	c_ptr1 = c_ptr2 = 0;
	count = TempVarCount - 1;
	while (count >= 0) {
		strcpy(name, "$T");
		itoa(count, name + 2, 10);
		count--;
		c_ptr2 = EmitCode(Q_KILL, 0, name, 0, 0, 0);
		c_ptr1 = MergeCodes(c_ptr1, c_ptr2);
	}
	*c_ptr = c_ptr1;
	return(0);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FormArgString											*/
/*	Input		:	type, argument type (integer/real/string/int const, etc.*/
/*					name, pointer to argument name,							*/
/*					buffer, buffer pointer to copy argument string in.		*/
/*	Output		:	length of the argument string.							*/
/*	Synopsis	:	Builds the argument string from the type and name.		*/
/*--------------------------------------------------------------------------*/

int		FormArgString(byte type, byte *name, byte *buffer)
{
	SymInfoType	*sym_ptr;
	int		len;

	buffer[0] = type;
	len = 0;
	switch (type) {
		case E_INT_CONST :
		case E_REAL_CONST :
		case E_STRING_CONST :
			/*	Copy the whole constant (int/real/string.	*/
			strcpy(buffer + 1, name);	/*	Copy constant value.	*/
			len = strlen(buffer + 1) + 2;	/*	type, value, null.	*/
			break;
		case E_INTEGER :
		case E_REAL :
		case E_STRING :
			if (name[0] == '$') {
				if (name[1] == 'T') {
					buffer[1] = VAR_TEMPORARY;
				   sprintf (buffer + 2, "%c%s", (byte) atoi (name + 2), "");
					len = 4;	/*	type(1), storage class(1), address(2).	*/
				}
				else {
					len = 2;
					if (!strcmp(name, PreDefVars[CONDN_VARIABLE].VarName))
						buffer[1] = VAR_CONDITION;
					else
						if (!strcmp(name, PreDefVars[RETURN_VARIABLE].VarName))
							buffer[1] = VAR_RETURN;
						else
							len = 0;
				}
			}
			else {
				if (sym_ptr = FindSymbol(name)) {
					buffer[1] = sym_ptr->SymType;	/*	Global/local/param.	*/
					buffer[2] = sym_ptr->Status;
					len = 4;	/*	type(1), storage class(1), address(2).	*/
				}
				else
					len = 0;	/*	Error !	*/
			}
			break;
	}
	CodeSize += len;

	return(len);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	WriteQuads												*/
/*	Input		:	c_ptr, pointer to code list.							*/
/*	Output		:	0 if code written, else -1.								*/
/*	Synopsis	:	Writes out code for the quads given.					*/
/*--------------------------------------------------------------------------*/

int		WriteQuads(CodeTypeFptr c_ptr)
{
	int		len;	/*	Length of symbol name/constant to write.	*/
	byte	buffer[MAX_TOKEN_LEN];	/*	Max length of argument string.	*/
	word	count;
	int		offset;
	int		result_type;

	if (CodeFileHandle < 0)
		goto ignore_exit;
	count = 0;
	while (c_ptr) {
		buffer[0] = c_ptr->Operator;
		if (FWrite(CodeFileHandle, buffer, sizeof(c_ptr->Operator)) !=
												sizeof(c_ptr->Operator))
			goto error_exit;
		switch (c_ptr->Operator) {
			case Q_PLUS :
			case Q_MINUS :
			case Q_TIMES :
			case Q_DIV :
			case Q_LESS_OR_EQUAL :
			case Q_GREATER_OR_EQUAL :
			case Q_LESS :
			case Q_GREATER :
			case Q_IS_EQUAL :
			case Q_IS_NOT_EQUAL :
			case Q_AND :
			case Q_OR :
				/*	operator, type1, arg1, type2, arg2, dummy type, arg3.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				len = FormArgString(c_ptr->Type2, c_ptr->Arg2, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;

				/*	Get the type of the result from the opearator and	*/
				/*	the two operands.	*/
				result_type = GetResultType(c_ptr->Operator, c_ptr->Type1,
															c_ptr->Type2);
				if (result_type == -1)
					goto error_exit;
				len = FormArgString((byte) result_type, c_ptr->Arg3, buffer);

				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_NOT :
			case Q_UNARY_MINUS :
				/*	operator, type1, arg1, dummy type, arg3.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				/*	Type in the next call is a dummy.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg3, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_MOV :
				/*	operator, type1, arg1, type2, arg3.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				/*	If the first argument is a constant, set the	*/
				/*	second argument to the corresponding type.	*/
				switch (c_ptr->Type1) {
					case E_INT_CONST :
						c_ptr->Type2 = E_INTEGER;
						break;
					case E_REAL_CONST :
						c_ptr->Type2 = E_REAL;
						break;
					case E_STRING_CONST :
						c_ptr->Type2 = E_STRING;
						break;
				}
				len = FormArgString(c_ptr->Type2, c_ptr->Arg3, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_LABEL_DEFN :	/*	Label declaration.	*/
				/*	operator, label name.	*/
				len = strlen(c_ptr->Arg1) + 1;	/*	Label name + null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg1, len) != len)
					goto error_exit;
				break;
			case Q_JMP :	/*	Unconditional jump.	*/
				/*	operator, label offset, label name.	*/
				if ((offset = GetLabelOffset(c_ptr->Arg1)) >= 0) {
					/*	Get label offset relative to current instruction.	*/
					offset -= count;
					if (FWrite(CodeFileHandle, (byte *) &offset,
										sizeof(offset)) != sizeof(offset))
						goto error_exit;
				}
				else
					goto error_exit;

				len = strlen(c_ptr->Arg1) + 1;	/*	Label name + null.	*/
			  
				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg1, len) != len)
					goto error_exit;
				break;
			case Q_JZ :	/*	Conditional jump.	*/
			case Q_JNZ :	/*	Conditional jump.	*/
				/*	operator, type1, arg1, label offset, label name.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				if ((offset = GetLabelOffset(c_ptr->Arg3)) >= 0) {
					/*	Get label offset relative to current instruction.	*/
					offset -= count;
					if (FWrite(CodeFileHandle, (byte *) &offset,
										sizeof(offset)) != sizeof(offset))
						goto error_exit;
				}
				else
					goto error_exit;

				len = strlen(c_ptr->Arg3) + 1;	/*	Label name + null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg3, len) != len)
					goto error_exit;
				break;
			case Q_JEQ :
			case Q_JNEQ :
				/*	operator, type1, arg1, type2, arg2, label offset & name.*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				len = FormArgString(c_ptr->Type2, c_ptr->Arg2, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				if ((offset = GetLabelOffset(c_ptr->Arg3)) >= 0) {
					/*	Get label offset relative to current instruction.	*/
					offset -= count;
					if (FWrite(CodeFileHandle, (byte *) &offset,
										sizeof(offset)) != sizeof(offset))
						goto error_exit;
				}
				else
					goto error_exit;

				len = strlen(c_ptr->Arg3) + 1;	/*	Label name + null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg3, len) != len)
					goto error_exit;
				break;
			case Q_PUSH :
				/*	operator, type1, arg1.	*/
				len = FormArgString(c_ptr->Type1, c_ptr->Arg1, buffer);
				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_CALL :
				/*	operator, proc name.	*/
				len = strlen(c_ptr->Arg1) + 1;

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg1, len) != len)
					goto error_exit;
				break;
			case Q_RET :
				/*	operator.	*/
				break;
			case Q_LINENO :	/*	Psuedocode, line number information.	*/
				/*	operator, line number.	*/
				len = atoi(c_ptr->Arg1);

				/* Executable code size */
				CodeSize += sizeof(len);

				if (FWrite(CodeFileHandle, (byte *) &len, sizeof(len)) !=
															sizeof(len))
					goto error_exit;
				break;
			case Q_PROCEDURE :	/*	Procedure beginning.	*/
				/*	operator, proc name.	*/
				len = strlen(c_ptr->Arg1) + 1;	/*	Proc name + null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg1, len) != len)
					goto error_exit;
				break;
			case Q_ENDPROCEDURE :	/*	Procedure end.	*/
				/*	operator, proc name.	*/
				len = strlen(c_ptr->Arg1) + 1;	/*	Proc name + null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, c_ptr->Arg1, len) != len)
					goto error_exit;
				break;
			case Q_PARAM :	/*	Procedure parameter.	*/
				/*	operator, type1, arg1.	*/
				buffer[0] = c_ptr->Type1;
				strcpy(buffer + 1, c_ptr->Arg1);
				len = strlen(buffer + 1) + 2;	/*	type, name, null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_DECLARE :	/*	Variable declaration.	*/
				/*	operator, type1, arg1.	*/
				buffer[0] = c_ptr->Type1;
				strcpy(buffer + 1, c_ptr->Arg1);
				len = strlen(buffer + 1) + 2;	/*	type, name, null.	*/

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, buffer, len) != len)
					goto error_exit;
				break;
			case Q_TEMPCOUNT :	/*	Max temporaries used in fn body.	*/
				/*	operator, number of temporaries.	*/
				len = atoi(c_ptr->Arg1);

				/* Executable code size */
				CodeSize += len;

				if (FWrite(CodeFileHandle, (byte *) &len, sizeof(len)) !=
														sizeof(len))
					goto error_exit;
				break;
			case Q_KILL :	/*	Kill temporary.	*/
				/*	operator, temporary address.	*/
				sprintf (&buffer[0], "%c%s", (byte) atoi(c_ptr->Arg1 + 2), "");

				/* Executable code size */
				CodeSize += sizeof(int);

				if (FWrite(CodeFileHandle, buffer, sizeof(int)) != sizeof(int))
					goto error_exit;
				break;
		}
		
		/* Operator Size */
		CodeSize += sizeof(c_ptr -> Operator);

		c_ptr = c_ptr->NextCode;
		count++;
	}
	ignore_exit :
	return(0);

	error_exit :
	/*	Flush the file and invalidate handle to prevent further writes.	*/
	Close(CodeFileHandle);
	CodeFileHandle = -1;		/*	Invalidate the file handle.	*/
	return(-1);
}
