/*--------------------------------------------------------------------------*/
/*	File		:	DECODE.C												*/
/*	Purpose		:	This file contains script language decoding routines.	*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#ifndef WIN
#include	"\me_2\inc\misc.h"
#else
#include	"windows.h"
#include	"misc.h"
#endif

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"

#include	<string.h>
#include	<stdlib.h>
#include	<stdio.h>

static	int		InFileHandle = -1;	/*	Input file handle.	*/
static	int		Scope;	/*	Lexical scope level.	*/
static	SymInfoType	*Display[MAX_LEVELS];	/*	Display of symbols.	*/

static	PreDefVarType	PreDefVars[] = {	/*	List of pre-defined vars.	*/
	{"$CONDN_VAR", E_INTEGER},
	{"$RET_VAR", E_INTEGER},
	{"CONNECTED", E_INTEGER},
	{"EOF", E_INTEGER},
	{"SUCCESS", E_INTEGER},
	{"KEYHIT", E_INTEGER},
	{"MONO", E_INTEGER},
	{"MEHOST", E_INTEGER}
};

static	int		NoOfPreDefVars = sizeof(PreDefVars) / sizeof(PreDefVarType);

extern	int		Open(byte *file_name);
extern	int		Read(int file_handle, byte *buffer, int count);
extern	int		FWrite(int file_handle, byte *buffer, int count);
extern	void	Close(int file_handle);
extern	int		AddSymbol(byte *name, int sym_type, SymInfoType **n_ptr);
extern	int		GetSymbolName(SymInfoType *sym_ptr, byte class, int address, byte **name_ptr);
extern	void	PurgeSymbols(SymInfoType *n_ptr);
extern	int		ReadArgument(byte arg_type, byte *buffer);
extern	int		ReadAndPrint(void);

int		main(int argc, char *argv[])
{
	if (argc < 2) {
		printf("Usage : decode <code_file_name>\n");
		goto error_exit;
	}
	InFileHandle = Open(argv[1]);
	if (InFileHandle < 0) {
		printf("Error opening %s\n", argv[1]);
		goto error_exit;
	}
	ReadAndPrint();
	Close(InFileHandle);
	return(0);

	error_exit :
	if (InFileHandle >= 0)
		Close(InFileHandle);
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	Open													*/
/*	Input		:	file_name, file name to open.							*/
/*	Output		:	file handle if file opened, else -1.					*/
/*	Synopsis	:	Opens the file and returns the file handle.				*/
/*--------------------------------------------------------------------------*/

int		Open(byte *file_name)
{
	int		file_handle;

#ifndef WIN
	_asm {
		mov		file_handle, -1
		mov		ax, 3d00H
		mov		dx, file_name
		int		21H
		jc		OpenError
		mov		file_handle, ax
	OpenError :
	}
#else
	file_handle = _lopen(file_name, READ_WRITE);
#endif
	return(file_handle);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	Read													*/
/*	Input		:	file_handle, file handle to read,						*/
/*					buffer, buffer to read into,							*/
/*					count, number of characters to read.					*/
/*	Output		:	number of characters read if no error, else -1.			*/
/*	Synopsis	:	Reads specified number of characters from the file.		*/
/*--------------------------------------------------------------------------*/

int		Read(int file_handle, byte *buffer, int count)
{
#ifndef WIN
	_asm {
		mov		ah, 3fH				; read from file.
		mov		bx, file_handle		; file handle to read from.
		mov		cx, count			; number of characters to read.
		mov		dx, buffer			; buffer address.
		int		21H
		jnc		ReadDone
		mov		ax, -1				; error reading.
	ReadDone :
	}
#else
	int	   read_count;
	read_count = _lread(file_handle, buffer, count);
	return read_count;
#endif
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FWrite													*/
/*	Input		:	file_handle, file handle to write into,					*/
/*					buffer, buffer to write,								*/
/*					count, number of characters to write.					*/
/*	Output		:	number of characters written if no error, else -1.		*/
/*	Synopsis	:	Writes specified number of characters to the file.		*/
/*--------------------------------------------------------------------------*/

int		FWrite(int file_handle, byte *buffer, int count)
{
#ifndef WIN
	_asm {
		mov		ah, 40H				; write to file.
		mov		bx, file_handle		; file handle to write to.
		mov		cx, count			; number of characters to write.
		mov		dx, buffer			; buffer address.
		int		21H
		jnc		WriteDone
		mov		ax, -1				; error writing.
	WriteDone :
	}
#else
	int		write_count;
	write_count = _lwrite(file_handle, buffer, count);
	return write_count;
#endif
}

/*--------------------------------------------------------------------------*/
/*	Name		:	Close													*/
/*	Input		:	file_handle, file handle to close.						*/
/*	Output		:	None													*/
/*	Synopsis	:	Closes the file.										*/
/*--------------------------------------------------------------------------*/

void	Close(int file_handle)
{
#ifndef WIN
	_asm {
		mov		ah, 3eH
		mov		bx, file_handle
		int		21H
	}
#else
	_lclose(file_handle);
#endif
}

/*--------------------------------------------------------------------------*/
/*	Name		:	AddSymbol												*/
/*	Input		:	name, symbol name,										*/
/*					sym_type, symbol type (integer/real/string),			*/
/*					n_ptr, pointer to intialize for node added.				*/
/*	Output		:	0 if symbol added, else -1.								*/
/*	Synopsis	:	Adds the symbol to the symbol table at the specified	*/
/*						level. If the symbol already there n_ptr is set to	*/
/*						the node, and -1 is returned. If out of memory 0 is	*/
/*						returned with n_ptr set to null.					*/
/*--------------------------------------------------------------------------*/

int		AddSymbol(byte *name, int sym_type, SymInfoType **n_ptr)
{
	SymInfoType	**ptr;
	SymInfoType	*sym_ptr;
	int		result;

	if (Scope >= MAX_LEVELS) {
		*n_ptr = 0;	/*	No such level possible.	*/
		goto error_exit;
	}
	sym_ptr = Display[Scope];
	while (sym_ptr) {
		result = strcmp(name, sym_ptr->Name);
		if (!result) {
			*n_ptr = sym_ptr;	/*	Symbol already found.	*/
			goto error_exit;
		}
		if (result < 0) {
			if (sym_ptr->Llink)
				sym_ptr = sym_ptr->Llink;	/*	Look in left subtree.	*/
			else {
				ptr = &sym_ptr->Llink;	/*	Insert here.	*/
				break;
			}
		}
		else {
			if (sym_ptr->Rlink)
				sym_ptr = sym_ptr->Rlink;	/*	Look in right subtree.	*/
			else {
				ptr = &sym_ptr->Rlink;	/*	Insert here.	*/
				break;
			}
		}
	}
	if (!sym_ptr)
		ptr = &Display[Scope];
#ifndef WIN
	sym_ptr = (SymInfoType *) malloc(sizeof(SymInfoType));
#else
	sym_ptr = (SymInfoType *) LocalAlloc(LMEM_FIXED, sizeof(SymInfoType));
#endif
	if (!sym_ptr) {
		*n_ptr = 0;
		goto error_exit;
	}
#ifndef WIN
	sym_ptr->Name = malloc(strlen(name) + 1);
#else
	sym_ptr->Name = LocalAlloc(LMEM_FIXED, strlen(name) + 1);
#endif
	if (!sym_ptr->Name) {
#ifndef WIN
		free((byte *) sym_ptr);
#else
		LocalFree((byte *) sym_ptr);
#endif
		*n_ptr = 0;
		goto error_exit;
	}
	*n_ptr = *ptr = sym_ptr;
	sym_ptr->SymClass = (byte) sym_type;
	sym_ptr->Status = 0;
	strcpy(sym_ptr->Name, name);
	sym_ptr->Llink = sym_ptr->Rlink = 0;
	sym_ptr->ProcInfoPtr = 0;
	return(0);

	error_exit :
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FindSymbol												*/
/*	Input		:	sym_class, root of symbol table subtree,				*/
/*					class, symbol storage class,							*/
/*					address, symbol address,								*/
/*					name_ptr, pointer to pointer to symbol name.			*/
/*	Output		:	true if symbol found else false.						*/
/*	Synopsis	:	Looks up the symbol table and returns a pointer to the	*/
/*						symbol name (depth first search).					*/
/*--------------------------------------------------------------------------*/

int		GetSymbolName(SymInfoType *sym_ptr, byte class, int address, byte **name_ptr)
{
	if (!sym_ptr)
		return(false);
	if (sym_ptr->SymType == class && sym_ptr->Status == address) {
		*name_ptr = sym_ptr->Name;
		return(true);
	}
	if (sym_ptr->Llink && GetSymbolName(sym_ptr->Llink, class, address,
															name_ptr))
		return(true);
	if (sym_ptr->Rlink && GetSymbolName(sym_ptr->Rlink, class, address,
															name_ptr))
		return(true);
	return(false);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	PurgeSymbols											*/
/*	Input		:	n_ptr, root of symbol table to purge.					*/
/*	Output		:	None													*/
/*	Synopsis	:	Purges the symbols from the specified root.				*/
/*--------------------------------------------------------------------------*/

void	PurgeSymbols(SymInfoType *n_ptr)
{
	if (n_ptr) {
		PurgeSymbols(n_ptr->Llink);
		PurgeSymbols(n_ptr->Rlink);
#ifndef WIN
		free(n_ptr->Name);
		free((byte *) n_ptr);
#else
		LocalFree(n_ptr->Name);
		LocalFree((byte *) n_ptr);
#endif
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ReadArgument											*/
/*	Input		:	arg_type, type of argument to read,						*/
/*					buffer, buffer to read argument into.					*/
/*	Output		:	0 if read successfully, else -1.						*/
/*	Synopsis	:	Reads the argument into the buffer.						*/
/*--------------------------------------------------------------------------*/

int		ReadArgument(byte arg_type, byte *buffer)
{
	byte	class;		/*	Storage class of variable.	*/
	int		address;	/*	Address, offset into symbol table.	*/
	byte	*name;

	switch (arg_type) {
		case E_INT_CONST :
		case E_REAL_CONST :
		case E_STRING_CONST :
			while (1) {
				if (Read(InFileHandle, buffer, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (!*buffer)
					break;	/*	Null found, done.	*/
				buffer++;
			}
			break;
		case E_INTEGER :
		case E_REAL :
		case E_STRING :
			if (Read(InFileHandle, &class, sizeof(class)) != sizeof(class))
				goto error_exit;
			switch (class) {
				case VAR_GLOBAL :
					if (Read(InFileHandle, (byte *) &address,
									sizeof(address)) != sizeof(address))
						goto error_exit;
					if (GetSymbolName(Display[0], VAR_GLOBAL, address, &name))
						strcpy(buffer, name);
					else
						buffer[0] = 0;	/*	Symbol not found, error !	*/
					break;
				case VAR_PARAM :
				case VAR_LOCAL :
					if (Read(InFileHandle, (byte *) &address,
									sizeof(address)) != sizeof(address))
						goto error_exit;
					if (GetSymbolName(Display[1], class, address, &name))
						strcpy(buffer, name);
					else
						buffer[0] = 0;	/*	Symbol not found, error !	*/
					break;
				case VAR_TEMPORARY :
					if (Read(InFileHandle, (byte *) &address,
									sizeof(address)) != sizeof(address))
						goto error_exit;
					strcpy(buffer, "$T");
					itoa(address, buffer + 2, 10);
					break;
				case VAR_CONDITION :
					strcpy(buffer, PreDefVars[CONDN_VARIABLE].VarName);
					break;
				case VAR_RETURN :
					strcpy(buffer, PreDefVars[RETURN_VARIABLE].VarName);
					break;
			}
			break;
	}
	return(0);

	error_exit :
	*buffer = 0;
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ReadAndPrint											*/
/*	Input		:	None													*/
/*	Output		:	0 if successfule, else -1.								*/
/*	Synopsis	:	Reads the code file and prints the instructions.		*/
/*--------------------------------------------------------------------------*/

int		ReadAndPrint(void)
{
	static	byte	*OperatorStrs[] = { "neg", "add", "sub", "mul", "div",
		"lesseq", "greatereq", "equal", "notequal", "less", "greater", "not",
		"", "", "and", "or"
	};
	static	byte	*TypeStrs[] = {"INT", "REAL", "STRING",
		"INTCONST", "REALCONST", "STRINGCONST" "", "", "",
	};
	SymInfoType	*sym_ptr;
	int		count;
	int		param_count;
	int		var_count;
	int		address;
	byte	operator;
	byte	type1;
	byte	type2;
	byte	arg1[MAX_TOKEN_LEN];
	byte	arg2[MAX_TOKEN_LEN];
	byte	arg3[MAX_TOKEN_LEN];

	var_count = 1;	/*	No. of globals.	*/
	while (1) {
		count = Read(InFileHandle, &operator, sizeof(byte));
		if (count == -1)
			break;	/*	End of file.	*/
		switch (operator) {
			case Q_PLUS :
			case Q_MINUS :
			case Q_TIMES :
			case Q_DIV :
			case Q_LESS_OR_EQUAL :
			case Q_GREATER_OR_EQUAL :
			case Q_LESS :
			case Q_GREATER :
			case Q_IS_EQUAL :
			case Q_IS_NOT_EQUAL :
			case Q_AND :
			case Q_OR :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				if (Read(InFileHandle, &type2, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type2, arg2) < 0)
					goto error_exit;
				/*	Read off the dummy argument type.	*/
				if (Read(InFileHandle, &type2, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type2, arg3) < 0)
					goto error_exit;
				printf("\t%s\t%s %s %s %s %s\n", OperatorStrs[operator],
						TypeStrs[type1], arg1, TypeStrs[type2], arg2, arg3);
				break;
			case Q_NOT :
			case Q_UNARY_MINUS :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				/*	Read off the dummy type.	*/
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg3) < 0)
					goto error_exit;
				printf("\t%s\t%s %s %s\n", OperatorStrs[operator],
						TypeStrs[type1], arg1, arg3);
				break;
			case Q_MOV :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				if (Read(InFileHandle, &type2, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type2, arg3) < 0)
					goto error_exit;
				printf("\tmov\t%s %s %s %s\n", TypeStrs[type1], arg1,
												TypeStrs[type2], arg3);
				break;
			case Q_LABEL_DEFN :	/*	Label declaration.	*/
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("%s :\n", arg1);
				break;
			case Q_JMP :	/*	Unconditional jump.	*/
				/*	Read off the jump label offset.	*/
				if (Read(InFileHandle, (byte *) &address,
						sizeof(sym_ptr->Status)) != sizeof(sym_ptr->Status))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("\tjmp\t%s\n", arg1);
				break;
			case Q_JZ :	/*	Conditional jump.	*/
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				/*	Read off the jump label offset.	*/
				if (Read(InFileHandle, (byte *) &address,
						sizeof(sym_ptr->Status)) != sizeof(sym_ptr->Status))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg3) < 0)
					goto error_exit;
				printf("\tjz\t%s %s %s\n", TypeStrs[type1], arg1, arg3);
				break;
			case Q_JNZ :	/*	Conditional jump.	*/
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				/*	Read off the jump label offset.	*/
				if (Read(InFileHandle, (byte *) &address,
						sizeof(sym_ptr->Status)) != sizeof(sym_ptr->Status))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg3) < 0)
					goto error_exit;
				printf("\tjnz\t%s %s %s\n", TypeStrs[type1], arg1, arg3);
				break;
			case Q_JEQ :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				if (Read(InFileHandle, &type2, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type2, arg2) < 0)
					goto error_exit;
				/*	Read off the jump label offset.	*/
				if (Read(InFileHandle, (byte *) &address,
						sizeof(sym_ptr->Status)) != sizeof(sym_ptr->Status))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg3) < 0)
					goto error_exit;
				printf("\tje\t%s %s %s %s %s\n", TypeStrs[type1],
						arg1, TypeStrs[type2], arg2, arg3);
				break;
			case Q_JNEQ :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				if (Read(InFileHandle, &type2, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type2, arg2) < 0)
					goto error_exit;
				/*	Read off the jump label offset.	*/
				if (Read(InFileHandle, (byte *) &address,
						sizeof(sym_ptr->Status)) != sizeof(sym_ptr->Status))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg3) < 0)
					goto error_exit;
				printf("\tjne\t%s %s %s %s %s\n", TypeStrs[type1], arg1,
						TypeStrs[type2], arg2, arg3);
				break;
			case Q_PUSH :
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(type1, arg1) < 0)
					goto error_exit;
				printf("\tpush\t%s %s\n", TypeStrs[type1], arg1);
				break;
			case Q_CALL :
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("\tcall\t%s\n", arg1);
				break;
			case Q_RET :
				printf("\tret\n");
				break;
			case Q_LINENO :	/*	Psuedocode, line number information.	*/
				if (Read(InFileHandle, (byte *) &address, sizeof(address)) !=
													sizeof(address))
					goto error_exit;
				printf("; Line %d\n", address);
				break;
			case Q_PROCEDURE :	/*	Procedure beginning.	*/
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("PROC\t%s\n", arg1);
				Scope++;
				param_count = 1;
				var_count = 1;
				break;
			case Q_ENDPROCEDURE :	/*	Procedure end.	*/
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("ENDPROC\t%s\n", arg1);
				PurgeSymbols(Display[Scope]);
				Display[Scope] = 0;
				Scope--;
				break;
			case Q_PARAM :	/*	Procedure parameter.	*/
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("\tPARAM\t%s %s\n", TypeStrs[type1], arg1);
				if (AddSymbol(arg1, type1, &sym_ptr))
					goto error_exit;
				else {
					sym_ptr->SymType = VAR_PARAM;
					sym_ptr->Status = param_count++;
				}
				break;
			case Q_DECLARE :	/*	Variable declaration.	*/
				if (Read(InFileHandle, &type1, sizeof(byte)) != sizeof(byte))
					goto error_exit;
				if (ReadArgument(E_STRING_CONST, arg1) < 0)
					goto error_exit;
				printf("\tDECLARE\t%s %s\n", TypeStrs[type1], arg1);
				if (AddSymbol(arg1, type1, &sym_ptr))
					goto error_exit;
				else {
					sym_ptr->SymType = (byte) (!Scope ? VAR_GLOBAL : VAR_LOCAL);
					sym_ptr->Status = var_count++;
				}
				break;
			case Q_TEMPCOUNT :	/*	Max temporaries used in fn body.	*/
				if (Read(InFileHandle, (byte *) &address, sizeof(address)) !=
													sizeof(address))
					goto error_exit;
				printf("\tTEMPCOUNT\t%d\n", address);
				break;
			case Q_KILL :	/*	Kill temporary.	*/
				if (Read(InFileHandle, (byte *) &address, sizeof(address)) !=
													sizeof(address))
					goto error_exit;
				printf("\tKILL\t$T%d\n", address);
				break;
		}
	}
	return(0);

	error_exit :
	return(-1);
}
