/*--------------------------------------------------------------------------*/
/*	File		:	SCRIPT.C												*/
/*	Purpose		:	This file contains script language routines.			*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#ifndef WIN
#include	"\me_2\inc\misc.h"
#else
#include	"windows.h"
#include	"misc.h"
#endif

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"
#include	"scrtdata.h"
#include	"scrtfuns.h"

#include	<string.h>
#include	<stdlib.h>

#define	SCRIPT_FILE_ERROR	1	/*	Error in script compilation.	*/
#define	FAR_HEAP_ERROR		2	/*	Far heap allocation error.	*/
#define	USAGE_ERROR			3	/*	Usage error.	*/
#define	FILE_NAME_ERROR		4	/*	Error in script file name.	*/
#define	FILE_OPEN_ERROR		5	/*	Error in script file opening.	*/
#define	BUILT_IN_FN_ERROR	6	/*	Error in adding built in functions.	*/


#define MEM_SIZE		0x6000		/*	Memory needed for near neap.	*/

int		compiler_main(int argc, char *argv[])
{
	/*	List of target applications.	*/
	static	byte	*ApplicationList[] = {
		"METERM", "MEREMOTE", "MEWTERM",
	};
	byte	mesg_buffer[100];	/*	To form messages.	*/
	byte	file_name[PATH_NAME_LEN + FILE_NAME_LEN - 1];
	byte	*ptr;
	int		result;
	int		idx;

	result = CheckForMeterm();
	MeTermPresent = (byte) (result == MC_TERM_SIG);
	MeRmtPresent = (byte) (result == MC_RMT_SIG);

	if (argc < 3) {
		result = USAGE_ERROR;		/*	Usage error.	*/
		goto error_exit;
	}
	for (idx = 0; idx < sizeof(ApplicationList) / sizeof(byte *); idx++) {
		if (!stricmp(argv[1], ApplicationList[idx])) {
			Application = (byte) (1 << idx);
			break;
		}
	}
	if (!Application) {
		result = USAGE_ERROR;	/*	Invalid target appln name, usage error.	*/
		goto error_exit;
	}

	idx = 2;	/*	Index into command line parameters.	*/
	if (Optimize = (byte) !stricmp(argv[idx], "/O")) {	/*	Optimize ?	*/
		idx++;
		if (argc == idx) {	/*	Switch given, but no file name.	*/
			result = USAGE_ERROR;		/*	Usage error.	*/
			goto error_exit;
		}
	}
	if (Suppress = (byte) !stricmp(argv[idx], "/S")) {
		idx++;		/*	Suppress psuedo code.	*/
		if (argc == idx) {	/*	Switch given, but no file name.	*/
			result = USAGE_ERROR;		/*	Usage error.	*/
			goto error_exit;
		}
	}
	ptr = argv[idx++];	/*	Script file name.	*/

	/*	If the file name has more than admissible, truncate it.	*/
	if (strlen(ptr) > PATH_NAME_LEN + FILE_NAME_LEN - 1)
		*(ptr + PATH_NAME_LEN + FILE_NAME_LEN - 2) = 0;
	strcpy(FileName, ptr);
	ptr = CheckForExtn(FileName);
	if (ptr) {	/*	There is an extension given.	*/
		if (!stricmp(ptr, ScriptExtn)) {
			/*	Extension is the same as script extension, strip the extn.	*/
			*ptr = 0;	/*	Terminate string at extension.	*/
			strcpy(file_name, FileName);	/*	Script file without extn.	*/
		}
		else {
			ShowMesg(ScrtMesg[SCRIPT_FILE_NAME_ERROR], true);
			result = FILE_NAME_ERROR;
			goto error_exit;
		}
	}
	else {
		/*	Check if adding the extension would overflow file name len.	*/
		if (strlen(FileName) > PATH_NAME_LEN + FILE_NAME_LEN - 5) {
			ShowMesg(ScrtMesg[SCRIPT_FILE_NAME_ERROR], true);
			result = FILE_NAME_ERROR;
			goto error_exit;
		}
		strcpy(file_name, FileName);
	}
	strcat(file_name, ScriptExtn);	/*	Script file to open.	*/
	/*	Open the file in read only, exclusive mode.	*/
	InFileHandle = Open(file_name, READONLY | EXCLUSIVE);
	if (InFileHandle < 0) {
		strcpy(mesg_buffer, "Error opening ");
		strcat(mesg_buffer, file_name);
		ShowMesg(mesg_buffer, false);
		result = FILE_OPEN_ERROR;	/*	File open error.	*/
		goto error_exit;
	}
	if (!Suppress) {
		strcpy(file_name, FileName);
		strcat(file_name, PsuedoCodeExtn);	/*	Psuedo code file name.	*/
		/*	Create the file, close it, and then open it in	*/
		/*	read/write exclusive mode.	*/
		OutFileHandle = Create(file_name);
		if (OutFileHandle >= 0) {
			Close(OutFileHandle);
			OutFileHandle = Open(file_name, READWRITE | EXCLUSIVE);
		}
		if (OutFileHandle < 0) {
			strcpy(mesg_buffer, "Error creating ");
			strcat(mesg_buffer, file_name);
			ShowMesg(mesg_buffer, false);
			result = FILE_OPEN_ERROR;	/*	File open error.	*/
			goto error_exit;
		}
	}
	strcpy(file_name, FileName);
	strcat(file_name, CodeExtn);	/*	Code file name.	*/
	/*	Create the file, close it, and then open it in	*/
	/*	read/write exclusive mode.	*/
	CodeFileHandle = Create(file_name);
	if (CodeFileHandle >= 0) {
		Close(CodeFileHandle);
		CodeFileHandle = Open(file_name, READWRITE | EXCLUSIVE);
	}
	if (CodeFileHandle < 0) {
		strcpy(mesg_buffer, "Error creating ");
		strcat(mesg_buffer, file_name);
		ShowMesg(mesg_buffer, false);
		result = FILE_OPEN_ERROR;	/*	File open error.	*/
		goto error_exit;
	}
	/*	If error file name given, remember error file name.	*/
	if (argc > idx + 1 && (argv[idx][0] == '-' || argv[idx][0] == '/') &&
						(argv[idx][1] == 'e' || argv[idx][1] == 'E')) {
		idx++;
		ErrorFileName = argv[idx];
	}
	if (InitCodeSpace() < 0) {
		ShowMesg(ScrtMesg[FAR_HEAP_ALLOC_ERROR], false);
		result = FAR_HEAP_ERROR;		/*	Far heap alloc error.	*/
		goto error_exit;
	}
	if (AddBuiltInFunctions() < 0) {
		result = BUILT_IN_FN_ERROR;	/*	Mem error or inconsistency.	*/
		goto error_exit;
	}
	Pprogram();
	FreeCodeSpace();

	if (OutFileHandle >= 0) {
		/*	Write out the compilation status to psuedo code file.	*/
		strcpy(mesg_buffer, "\r\n ");
		itoa(ErrCount, mesg_buffer + strlen(mesg_buffer), 10);
		strcat(mesg_buffer, " Error(s) detected\r\n");
		FWrite(OutFileHandle, mesg_buffer, strlen(mesg_buffer));
	}
	if (ErrCount) {
//		ShowMesg(ScrtMesg[ERROR_IN_SCRIPT_FILE], true);
		result = SCRIPT_FILE_ERROR;	/*	Script file error.	*/
		goto error_exit;
	}

	Close(InFileHandle);
	InFileHandle = -1;			/*	Invalidate the file handle.	*/
	if (OutFileHandle >= 0) {
		Close(OutFileHandle);
		OutFileHandle = -1;		/*	Invalidate the file handle.	*/
	}
	if (CodeFileHandle >= 0) {
		Close(CodeFileHandle);
		CodeFileHandle = -1;		/*	Invalidate the file handle.	*/
	}
	if (ErrorFileHandle >= 0) {
		Close(ErrorFileHandle);	/*	Close the error file name.	*/
		ErrorFileHandle = -1;
	}
	if (ErrorFileName)
		Delete(ErrorFileName);	/*	There was no error, so delete the file.	*/
	return(0);

	error_exit :

	FreeCodeSpace();
	if (InFileHandle >= 0)
		Close(InFileHandle);
	if (OutFileHandle >= 0)
		Close(OutFileHandle);
	if (CodeFileHandle >= 0) {
		Close(CodeFileHandle);
		strcpy(file_name, FileName);	/*	Delete the code file.	*/
		strcat(file_name, CodeExtn);
		Delete(file_name);
	}

	if (ErrorFileHandle >= 0)		/*	Close the error file name.	*/
		Close(ErrorFileHandle);

	if (result == USAGE_ERROR) {	/*	Usage error.	*/
		if (!Application)
			PutStr("\r\nMultiExpress Script Compiler Version 1.99d\r\n");
		ShowMesg("Usage : script <application> <file_name>", false);
		if (!Application)
			PutStr("\r\n");
	}
	return(result);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SkipPathname											*/
/*	Input		:	ptr, pointer to full path.								*/
/*	Output		:	Pointer to filename.									*/
/*	Synopsis	:	Skips directory path in the path name upto file name.	*/
/*--------------------------------------------------------------------------*/

byte	*SkipPathname(byte *ptr)
{
	byte	*t_ptr;

	t_ptr = ptr;
	ptr += strlen(ptr);
	while (*ptr != '\\' && *ptr != ':' && t_ptr != ptr)
		ptr--;
	if (*ptr == '\\' || *ptr == ':')
		ptr++;
	return(ptr);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckForExtn											*/
/*	Input		:	file_name, pointer to file name.						*/
/*	Output		:	Pointer to extension.									*/
/*	Synopsis	:	Looks for an extension in the file name given and		*/
/*						returns true if extension found else returns false.	*/
/*--------------------------------------------------------------------------*/

byte	*CheckForExtn(byte *file_name)
{
	word	len;
	byte	*ptr;

	ptr = SkipPathname(file_name);
	len = strlen(ptr);
	ptr = ptr + len - 1;
	if (len > 4)
		len = 4;	/*	No need to check more than 4 chars.	*/
	while (len) {
		if (*ptr != '.') {
			ptr--;
			len--;
		}
		else
			return(ptr);
	}
	return(0);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	AddBuiltInFunctions										*/
/*	Input		:	None													*/
/*	Output		:	0														*/
/*	Synopsis	:	Adds built in functions to symbol table.				*/
/*--------------------------------------------------------------------------*/

int		AddBuiltInFunctions(void)
{
	#define	Q_VALUE		0	/*	Argument passed by value.	*/
	#define	MAX_ARGS	10	/*	Maximum no. of arguments for built-in fns.	*/

	/*	If the argument type is set to zero, it is a don't care type.	*/
	typedef struct
	{
		byte *ProcName ;	/*	Procedure name.	*/
		int Application ;	/*	Apllication is METERM or MEREMOTE.	*/
		byte RetValue ;	/*	Function return type.	*/
		byte MinNoOfArgs ;	/*	Minimum no. of arguments.	*/
		byte MaxNoOfArgs ;	/*	Maximum no. of arguments.	*/
		byte ArgTypes[MAX_ARGS] ;	/*	Integer/real/string.	*/
		byte ArgKind[MAX_ARGS] ;	/*	Passed by value or reference.	*/
	} BuiltInType ;

	static BuiltInType BuiltInFunctions[] =
	{
#if 0
		{"DISPLAY", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {0}, {Q_VALUE}},
		{"RUN", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
#endif
		{"EXIT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 0, 0, {0}, {0}},
#if 0
		{"QUIT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 0, 0, {0}, {0}},
#endif
		{"WAIT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {Q_INTEGER}, {Q_VALUE}},
#if 0
		{"WAITUNTIL", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#endif
		{"TRANSMIT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#if 0
		{"DATE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VAR}},
		{"TIME", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VAR}},
		{"FIND", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 3, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"GETSTR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING, Q_INTEGER}, {Q_VAR, Q_VALUE}},
		{"GETINT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"GETHEXINT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"GETKEY", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"GETPAT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING, Q_INTEGER}, {Q_VAR, Q_VALUE}},
		{"DISPSTRAT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 4, 4, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_STRING}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE}},
		{"GETSTRAT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 4, 4, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_STRING}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VAR}},
		{"DRAWWINDOW", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 5, 5, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE}},
		{"SAVEWINDOW", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 4, 4, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE}},
		{"RESTOREWINDOW", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"DRAWBOX", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 5, 5, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE}},
		{"CLOSEBOX", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"CLS", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 0, 0, {0}, {0}},
		{"TYPE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
/*		{"SCROLLDOWN", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 6, 6, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VA
		{"SCROLLUP", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 6, 6, {Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALU
*/		{"DISPFMT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 6, {Q_STRING}, {Q_VALUE}}, 
		{"CHDIR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"DIR", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 1, {Q_STRING}, {Q_VALUE}},
		{"FCREATE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"FOPEN", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"FREAD", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VAR, Q_VALUE}},
		{"FWRITE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"FCLOSE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"FPUTS", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 3, {Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"FPUTC", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"FGETS", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VAR, Q_VALUE}},
		{"FGETC", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"ISFILE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#endif
		{"ATOI", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"ITOA", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 3, {Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VAR, Q_VALUE}},
		{"STRLEN", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"STRCAT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VAR, Q_VALUE}},
		{"STRCMP", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"STRCPY", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VAR, Q_VALUE}},
		{"STRFMT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 8, {Q_STRING, Q_STRING}, {Q_VAR, Q_VALUE}},
		{"DEC", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VAR}},
		{"INC", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VAR}},
#if 0
		{"MOD", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"DOS", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"EXEC", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SHELL", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"ALARM", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"SENDFILE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"RECVFILE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"TERMINALMODE", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 3, {Q_STRING}, {Q_VALUE}},
		{"ATOF", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_REAL, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"FTOA", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_REAL, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"FLOOR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_REAL}, {Q_VALUE}},
		{"CEIL", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_REAL}, {Q_VALUE}},
		{"DISKFREE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 0, 1, {0}, {0}},
		{"DOSVERSION", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_STRING, 0, 0, {0}, {0}},
		{"FINDFIRST", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 3, 3, {Q_STRING, Q_INTEGER, Q_STRING}, {Q_VALUE, Q_VALUE, Q_VAR}},
		{"FINDNEXT", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VAR}},
		{"FSEEK", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"GETENV", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"PUTENV", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"GETFILEDATE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"GETFILETIME", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"GETFILEATTR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"GETFILESIZE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETFILEDATE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"SETFILETIME", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"SETFILEATTR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 2, 2, {Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"MKDIR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"RMDIR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"INPORT", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"OUTPORT", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
#endif
		{"TOUPPER", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {Q_STRING}, {Q_VAR}},
		{"TOLOWER", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 1, {Q_STRING}, {Q_VAR}},
#if 0
		{"FLUSHKBD", (METERM_FN | MEREMOTE_FN), Q_VOID, 0, 0, {0}, {0}},
		{"GETCURPOS", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VAR, Q_VAR}},
		{"GETCURTYPE", (METERM_FN | MEREMOTE_FN), Q_VOID, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VAR, Q_VAR}},
		{"SETCURPOS", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"SETCURTYPE", (METERM_FN | MEREMOTE_FN), Q_VOID, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"GETVIDCHAR", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"GETVIDATTR", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"PUTVIDCHAR", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"PUTVIDATTR", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"MEMFREE", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"MEMALLOC", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 3, 3, {Q_INTEGER, Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VAR, Q_VAR}},
		{"MEMRELEASE", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 2, 2, {Q_INTEGER, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"MEMREAD", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 4, 4, {Q_INTEGER, Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VAR, Q_VALUE}},
		{"MEMWRITE", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 4, 4, {Q_INTEGER, Q_INTEGER, Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE, Q_VALUE, Q_VALUE}},
		{"DELETE", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SAVESCREEN", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VAR}},
		{"RESTORESCREEN", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"GETCURDIR", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING, Q_STRING}, {Q_VAR, Q_VALUE}},
		{"FDISPLAY", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 1, 6, {Q_STRING}, {Q_VALUE}},
#endif
#if 0
		{"DROPCONNECTION", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_VOID, 0, 0, {0}, {0}},
#endif
#if 0
		{"MONO", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
#endif
		{"GETDCD", (METERM_FN | MEREMOTE_FN | MEWTERM_FN), Q_INTEGER, 0, 0, {0}, {0}},
#if 0
		{"EOF", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
		{"MEHOST", (METERM_FN | MEREMOTE_FN), Q_INTEGER, 0, 0, {0}, {0}},
#endif
		{"BAUDRATE", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {0}, {Q_VALUE}},
#if 0
		{"BLANKON", METERM_FN, Q_VOID, 0, 0, {0}, {0}},
		{"BLINKON", METERM_FN, Q_VOID, 0, 0, {0}, {0}},
		{"BOLDON", METERM_FN, Q_VOID, 0, 0, {0}, {0}},
		{"BOLDOFF", METERM_FN, Q_VOID, 0, 0, {0}, {0}},
		{"CR", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"CURSOR", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"DATABITS", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"DIAL", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {0}, {Q_VALUE}},
		{"DUPLEX", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"EMULATION", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#endif
		{"HANGUP", (METERM_FN | MEWTERM_FN), Q_VOID, 0, 0, {0}, {0}},
#if 0
		{"LF", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"LOG", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING}, {Q_VALUE}},
		{"NORMALON", METERM_FN, Q_VOID, 0, 0, {0}, {0}},
#endif
		{"PARITY", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#if 0
		{"PRINTER", METERM_FN, Q_INTEGER, 0, 1, {Q_STRING}, {Q_VALUE}},
#endif
		{"RGETC", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING}, {Q_VAR}},
		{"RGETS", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING}, {Q_VAR}},
#if 0
		{"SNAPSHOT", METERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
#endif
		{"STOPBITS", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
#if 0
		{"TABWIDTH", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
#endif
		{"WAITFOR", (METERM_FN | MEWTERM_FN), Q_INTEGER, 1, 2, {Q_STRING}, {Q_VALUE}},
#if 0
		{"REBOOT", MEREMOTE_FN, Q_VOID, 0, 0, {0}, {0}},
		{"RENAME", MEWTERM_FN, Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"ASCTOHEX", MEWTERM_FN, Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"HEXTOASC", MEWTERM_FN, Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"FILEPOS", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"FILESEARCH", MEWTERM_FN, Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VAR}},
		{"FILESEEK", MEWTERM_FN, Q_INTEGER, 3, 3, {Q_INTEGER,Q_INTEGER,Q_INTEGER}, {Q_VALUE,Q_VALUE,Q_VALUE}},
		{"FOLLOWCURSOR", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"MOVECURSOR", MEWTERM_FN, Q_VOID, 2, 2, {Q_STRING, Q_INTEGER}, {Q_VALUE, Q_VALUE}},
		{"SETBACKGROUND", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETFOREGROUND", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETSCROLLBUFSIZE", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"TERMCOLUMNS", MEWTERM_FN, Q_INTEGER, 1 , 1, {Q_INTEGER}, {Q_VALUE}},
#endif
		{"BREAK", MEWTERM_FN, Q_INTEGER, 1, {Q_INTEGER}, {Q_VALUE}},
#if 0
		{"CONNECT", MEWTERM_FN, Q_INTEGER, 0, 8, {Q_INTEGER,0,0,0,Q_INTEGER, Q_STRING,Q_INTEGER,Q_STRING},{Q_VALUE}},
		{"CONNECTINBOUND", MEWTERM_FN, Q_INTEGER, 0, 0, {0},{0}},
		{"SETPHONEBOOK", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"CONNECTTIME", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
		{"LOADENTRY", MEWTERM_FN, Q_INTEGER, 1, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"REDIAL", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
#endif
		{"RXFLUSH", MEWTERM_FN, Q_VOID, 0, 0, {0}, {0}},
#if 0
		{"SAVEENTRY", MEWTERM_FN, Q_INTEGER, 0, {0}, {0}},
#endif
		{"TXFLUSH", MEWTERM_FN, Q_VOID, 0, 0, {0}, {0}},
#if 0
		{"SETPROTOCOL", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"GETPROTOCOL", MEWTERM_FN, Q_STRING, 1, 1, {Q_STRING}, {Q_VAR}},
		{"APPENDEXTRAMODEMCMD", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"GETMODEM", MEWTERM_FN, Q_STRING, 1, 1, {Q_STRING}, {Q_VAR}},
		{"SETCONNECTRESP", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETEXTRAMODEMCMD", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETMODEM", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"HIDE", MEWTERM_FN, Q_INTEGER, 1, 4, {Q_STRING}, {Q_VALUE}},
		{"SHOW", MEWTERM_FN, Q_INTEGER, 1, 4, {Q_STRING}, {Q_VALUE}},
		{"MAPKEY", MEWTERM_FN, Q_INTEGER, 2, 2, {Q_STRING, Q_STRING}, {Q_VALUE, Q_VALUE}},
		{"UNMAPKEY", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"INVOKEKEY", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETFILEDIR", MEWTERM_FN, Q_VOID, 1, {Q_STRING}, {Q_VALUE}},
		{"SETSCRIPTDIR", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETPHONEBOOKDIR", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"SETSYSTEMDIR", MEWTERM_FN, Q_VOID, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"DIRLIST", MEWTERM_FN, Q_INTEGER, 0, 1, {Q_STRING}, {Q_VALUE}},
		{"WINEXEC", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"WINVER",MEWTERM_FN, Q_STRING, 0, 0, {0}, {0}},
		{"DOSENV", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VAR}},
		{"FREELOCALMEM", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
		{"FREEGLOBALMEM", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
		{"ALLOCMEM", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"RELEASEMEM", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"WRITEMEM", MEWTERM_FN, Q_INTEGER, 3, 3, {Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE, Q_VALUE, Q_VALUE}},
		{"READMEM", MEWTERM_FN, Q_INTEGER, 3, 3, {Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE, Q_VALUE, Q_VAR}},
		{"DIALOG", MEWTERM_FN, Q_INTEGER, 4, 5, {Q_INTEGER,Q_INTEGER, Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE, Q_VAR}},
		{"DIALOGEND", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
		{"LTEXT", MEWTERM_FN, Q_INTEGER, 5, 5, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"RTEXT", MEWTERM_FN, Q_INTEGER, 5, 5, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"CTEXT", MEWTERM_FN, Q_INTEGER, 5, 5, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"CHECKBOX", MEWTERM_FN, Q_INTEGER, 6, 9, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING, Q_INTEGER}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VAR}},
		{"PUSHBUTTON", MEWTERM_FN, Q_INTEGER, 6, 9, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"LISTBOX", MEWTERM_FN, Q_INTEGER, 6, 9, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VAR,Q_VALUE}},
		{"GROUPBOX", MEWTERM_FN, Q_INTEGER, 5, 5, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"DEFPUSHBUTTON", MEWTERM_FN, Q_INTEGER, 5, 7, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
		{"RADIOBUTTON", MEWTERM_FN, Q_INTEGER, 6, 9, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING,Q_INTEGER}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VAR}},
		{"EDITTEXT", MEWTERM_FN, Q_INTEGER, 6, 9, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VAR}},
		{"ICON", MEWTERM_FN, Q_INTEGER, 5, 5, {Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_INTEGER,Q_STRING}, {Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE,Q_VALUE}},
  		{"MESSAGE", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
		{"GETFONT", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VAR}},
		{"SETFONT", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_STRING}, {Q_VALUE}},
#endif
  		{"THISLAYERUP", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},

		{"SETRTS", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"SETDTR", MEWTERM_FN, Q_INTEGER, 1, 1, {Q_INTEGER}, {Q_VALUE}},
		{"GETCTS", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}},
		{"ACTIVATEDOD", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}}	/* vidy 15/06/98 */
/*		{"GETDSR", MEWTERM_FN, Q_INTEGER, 0, 0, {0}, {0}} */
 	};

	SymInfoType	*sym_ptr;
	DefnArgType	*arg_list;
	DefnArgType	*arg_ptr;
	BuiltInType	*proc_ptr;
	byte	arg_count;
	int		idx;

	for (idx = 0; idx < sizeof(BuiltInFunctions) / sizeof(BuiltInType);
																idx++) {
		proc_ptr = &BuiltInFunctions[idx];
		if ((Application == METERM_FN) &&
								!(proc_ptr->Application & METERM_FN))
			continue;	/*	Function not for METERM.	*/
		if ((Application == MEREMOTE_FN) &&
								!(proc_ptr->Application & MEREMOTE_FN))
			continue;	/*	Function not for MEREMOTE.	*/
		if ((Application == MEWTERM_FN) &&
								!(proc_ptr->Application & MEWTERM_FN))
			continue;	/*	Function not for MEWTERM.	*/

		if (AddSymbol(proc_ptr->ProcName, Q_PROC, &sym_ptr)) {
			if (sym_ptr)
				Error(ERR_ID_ALREADY_DECLARED);
			else {
				Error(ERR_OUT_OF_MEMORY);
				goto error_exit;
			}
		}
#ifndef WIN
		sym_ptr->ProcInfoPtr = (ProcInfoType *) malloc(sizeof(ProcInfoType));
#else
		sym_ptr->ProcInfoPtr = (ProcInfoType *) LocalAlloc(LMEM_FIXED, sizeof(ProcInfoType));
#endif
		if (sym_ptr->ProcInfoPtr) {
			/*	Initialize procedure information.	*/
			sym_ptr->ProcInfoPtr->RetValue = proc_ptr->RetValue;
			sym_ptr->ProcInfoPtr->ArgList = 0;
			sym_ptr->ProcInfoPtr->SymTablePtr = 0;
			sym_ptr->ProcInfoPtr->ParamList = 0;
		}
		else {
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
		sym_ptr->Status |= BUILT_IN_FUNCTION;
		if (proc_ptr->MinNoOfArgs < proc_ptr->MaxNoOfArgs)
			sym_ptr->Status |= OPTIONAL_ARGS;
		for (arg_count = 0; arg_count < proc_ptr->MinNoOfArgs; arg_count++) {
#ifndef WIN
			arg_ptr = (DefnArgType *) malloc(sizeof(DefnArgType));
#else
			arg_ptr = (DefnArgType *) LocalAlloc(LMEM_FIXED, sizeof(DefnArgType));
#endif
			if (!arg_ptr) {
				Error(ERR_OUT_OF_MEMORY);
				goto error_exit;
			}
			arg_ptr->ArgClass = proc_ptr->ArgTypes[arg_count];
			arg_ptr->Attribute = proc_ptr->ArgKind[arg_count];
			arg_ptr->VarPtr = 0;
			arg_ptr->NextArgPtr = 0;
			if (!sym_ptr->ProcInfoPtr->ArgList)
				sym_ptr->ProcInfoPtr->ArgList = arg_list = arg_ptr;
			else {
				arg_list->NextArgPtr = arg_ptr;
				arg_list = arg_ptr;
			}
		}
	}
	return(0);

	error_exit :
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	AddPreDefinedVariables									*/
/*	Input		:	None													*/
/*	Output		:	Number of predefined variables added, -1 if error.		*/
/*	Synopsis	:	Adds predefined variables to symbol table.				*/
/*--------------------------------------------------------------------------*/

int		AddPreDefinedVariables(void)
{
	SymInfoType	*sym_ptr;
	int		var_count;

	/*	Add the predefined globals before adding globals declared	*/
	/*	by user. There sholud be a correspondence between the	*/
	/*	sequence in which they are added here and the sequence in	*/
	/*	which they are added just before reading script file and	*/
	/*	decoding for execution.	*/
	for (var_count = 0; var_count < NoOfPreDefVars; var_count++) {
		if (AddSymbol(PreDefVars[var_count].VarName,
							PreDefVars[var_count].VarType, &sym_ptr)) {
			if (sym_ptr)
				Error(ERR_ID_ALREADY_DECLARED);
			else {
				Error(ERR_OUT_OF_MEMORY);
				goto error_exit;
			}
		}
		else {	/*	Set the variable type to global.	*/
			sym_ptr->SymType = VAR_GLOBAL;
			sym_ptr->Status = var_count;	/*	Symbol offset.	*/
		}
	}
	return(var_count);

	error_exit :
	return(-1);
} 
