/*--------------------------------------------------------------------------*/
/*	File		:	SYMBOL.C												*/
/*	Purpose		:	This file contains symbol table routines.				*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#ifdef WIN
#include	"windows.h"
#endif

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"
#include	"scrtdata.h"
#include	"scrtfuns.h"

#include	<string.h>
#include	<stdlib.h>

/*--------------------------------------------------------------------------*/
/*	Name		:	AddSymbol												*/
/*	Input		:	name, symbol name,										*/
/*					sym_type, symbol type (integer/real/string),			*/
/*					n_ptr, pointer to intialize for node added.				*/
/*	Output		:	0 if symbol added, else -1.								*/
/*	Synopsis	:	Adds the symbol to the symbol table at the specified	*/
/*						level. If the symbol already there n_ptr is set to	*/
/*						the node, and -1 is returned. If out of memory 0 is	*/
/*						returned with n_ptr set to null.					*/
/*--------------------------------------------------------------------------*/

int		AddSymbol(byte *name, int sym_type, SymInfoType **n_ptr)
{
	SymInfoType	**ptr;
	SymInfoType	*sym_ptr;
	int		result;

	if (Scope >= MAX_LEVELS) {
		*n_ptr = 0;	/*	No such level possible.	*/
		goto error_exit;
	}
	sym_ptr = Display[Scope];
	while (sym_ptr) {
		result = strcmp(name, sym_ptr->Name);
		if (!result) {
			*n_ptr = sym_ptr;	/*	Symbol already found.	*/
			goto error_exit;
		}
		if (result < 0) {
			if (sym_ptr->Llink)
				sym_ptr = sym_ptr->Llink;	/*	Look in left subtree.	*/
			else {
				ptr = &sym_ptr->Llink;	/*	Insert here.	*/
				break;
			}
		}
		else {
			if (sym_ptr->Rlink)
				sym_ptr = sym_ptr->Rlink;	/*	Look in right subtree.	*/
			else {
				ptr = &sym_ptr->Rlink;	/*	Insert here.	*/
				break;
			}
		}
	}
	if (!sym_ptr)
		ptr = &Display[Scope];
#ifndef WIN
	sym_ptr = (SymInfoType *) malloc(sizeof(SymInfoType));
#else
	sym_ptr = (SymInfoType *) LocalAlloc(LMEM_FIXED, sizeof(SymInfoType));
#endif
	if (!sym_ptr) {
		Error(ERR_OUT_OF_MEMORY);
		*n_ptr = 0;
		goto error_exit;
	}
#ifndef WIN
	sym_ptr->Name = malloc(strlen(name) + 1);
#else
	sym_ptr->Name = (byte *) LocalAlloc(LMEM_FIXED, strlen(name) + 1);
#endif
	if (!sym_ptr->Name) {
		Error(ERR_OUT_OF_MEMORY);
#ifndef WIN
		free((byte *) sym_ptr);
#else
		LocalFree((SymInfoType *) sym_ptr);
#endif
		*n_ptr = 0;
		goto error_exit;
	}
	*n_ptr = *ptr = sym_ptr;
	sym_ptr->SymClass = (byte) sym_type;
	sym_ptr->SymType = 0;
	sym_ptr->Status = 0;
	strcpy(sym_ptr->Name, name);
	sym_ptr->Llink = sym_ptr->Rlink = 0;
	sym_ptr->ProcInfoPtr = 0;
	return(0);

	error_exit :
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FindSymbol												*/
/*	Input		:	name, symbol name.										*/
/*	Output		:	Pointer to symbol node if found, else null.				*/
/*	Synopsis	:	Looks up the symbol in the symbol table and returns a	*/
/*						pointer to the node is found, else null.			*/
/*--------------------------------------------------------------------------*/

SymInfoType	*FindSymbol(byte *name)
{
	SymInfoType	*sym_ptr;
	int		level;
	int		result;

	for (level = Scope; level >= 0; level--) {
		sym_ptr = Display[level];
		while (sym_ptr) {
			result = strcmp(name, sym_ptr->Name);
			if (!result)
				return(sym_ptr);	/*	Symbol found.	*/
			if (result < 0)
				sym_ptr = sym_ptr->Llink;
			else
				sym_ptr = sym_ptr->Rlink;
		}
	}
	return(0);	/*	Symbol not found.	*/
}

/*--------------------------------------------------------------------------*/
/*	Name		:	PurgeSymbols											*/
/*	Input		:	n_ptr, root of symbol table to purge.					*/
/*	Output		:	None													*/
/*	Synopsis	:	Purges the symbols from the specified root.				*/
/*--------------------------------------------------------------------------*/

void	PurgeSymbols(SymInfoType *n_ptr)
{
	if (n_ptr) {
		PurgeSymbols(n_ptr->Llink);
		PurgeSymbols(n_ptr->Rlink);
#ifndef WIN
		free(n_ptr->Name);
		free((byte *) n_ptr);
#else
		LocalFree((byte *) n_ptr->Name);
		LocalFree((SymInfoType *) n_ptr);
#endif
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	GenerateLabelName										*/
/*	Input		:	Pointer to label name.									*/
/*	Output		:	None.													*/
/*	Synopsis	:	Generates a label name.									*/
/*--------------------------------------------------------------------------*/

void	GenerateLabelName(byte *name_ptr)
{
	strcpy(name_ptr, "$L");
	name_ptr += 2;
	itoa(LabelCount, name_ptr, 10);
	LabelCount++;
}
