/*----------------------------------------------------------------------------
	Module   : Script
	Author   : CFP
	Synopsis	: Gives support for script read and write thro' COM as well 
				  as Ethernet
	Mod		:	16/07/97 Increased and #defined all the timeout values
					Vidy/ Chetan
Jyothi : Made changes to read and write script for small proxy memory map
----------------------------------------------------------------------------*/
// script.cpp : implementation file
//

#include "stdafx.h"
#include "procon.h"
#include "script.h"
#include "boot.h"
#include "cnfread.h"
#include "msgumain.h"
#include "msguview.h"
#include "compress.h"
#include "dnldprog.h"
#include "tftpread.h"
#include "tftpwrit.h"

#include <sys\types.h>
#include <sys\stat.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define ScriptTimerID 125
ScriptHeader ScriptInfo ;

/////////////////////////////////////////////////////////////////////////////
// CScriptDlg dialog


CScriptDlg::CScriptDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CScriptDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CScriptDlg)
	//}}AFX_DATA_INIT

	m_nPortNum = 0 ;
	m_szScriptSource = "" ;
	m_szComprScriptSource = "" ;
	m_szScriptCode = "" ;
	m_szFullScript = "" ;

	m_timeOldScr = 0 ;
}

void CScriptDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CScriptDlg)
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CScriptDlg, CDialog)
	//{{AFX_MSG_MAP(CScriptDlg)
	ON_BN_CLICKED(IDC_BTN_DNLD_SCRIPT, OnBtnDnldScript)
	ON_BN_CLICKED(IDC_BTN_EDIT_SCRIPT, OnBtnEditScript)
	ON_BN_CLICKED(IDC_BTN_COMPILE, OnBtnCompile)
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CScriptDlg message handlers

BOOL CScriptDlg::OnInitDialog()
{
	CDialog::OnInitDialog();
	
	// TODO: Add extra initialization here
	CenterWindow() ;

	char WindowHeader[80], CurrentHeading[80] ;

	GetWindowText (CurrentHeading, sizeof (CurrentHeading)) ;
	ModifyHeader (CurrentHeading, WindowHeader) ;
	SetWindowText (WindowHeader) ;

	if (!EditBmpBtn.LoadBitmaps ("EDIT1", "EDIT2", "EDIT3", "EDIT4"))
		AfxMessageBox ("Failed to load EDIT bitmap");

	VERIFY (EditBmpBtn.SubclassDlgItem (IDC_BTN_EDIT_SCRIPT, this));
	EditBmpBtn.SizeToContent();

	if (!CompileBmpBtn.LoadBitmaps ("COMPILE1", "COMPILE2", "COMPILE3", "COMPILE4"))
		AfxMessageBox ("Failed to load COMPILE bitmap");

	VERIFY (CompileBmpBtn.SubclassDlgItem (IDC_BTN_COMPILE, this));
	CompileBmpBtn.SizeToContent();

	if (!DnldBmpBtn.LoadBitmaps ("DNLD1", "DNLD2", "DNLD3", "DNLD4"))
		AfxMessageBox ("Failed to load DOWNLOAD bitmap");

	VERIFY (DnldBmpBtn.SubclassDlgItem (IDC_BTN_DNLD_SCRIPT, this));
	DnldBmpBtn.SizeToContent();

	if (!CancelBmpBtn.LoadBitmaps ("CLOSE1", "CLOSE2", "CLOSE3", "CLOSE4"))
		AfxMessageBox ("Failed to load CLOSE bitmap");

	VERIFY (CancelBmpBtn.SubclassDlgItem (IDCANCEL, this));
	CancelBmpBtn.SizeToContent();

	if (!ReadScript())
		CreateScrFile() ;
	GetOldScrTime() ;

	UpdateData (FALSE) ;
	return TRUE;  // return TRUE  unless you set the focus to a control
}

/*----------------------------------------------------------------------------
	Function : ReadScript
	Input    : None
	Output   : Boolean stating success of read
	Synopsis : The various file paths are formed
				  Depending on the mode branching is done
				  to read the complete script scr and cod files
----------------------------------------------------------------------------*/
BOOL CScriptDlg::ReadScript()
{
	// For now assume that router is up and running

	m_szScriptSource = LoadPath ;
	m_szComprScriptSource = LoadPath ;
	m_szScriptCode = LoadPath ;
	m_szFullScript = LoadPath ;

	/* File names will be Portn.??? where n = 1, 2, 3 */
	char szBuf[200] ;
	sprintf (szBuf, "%s%d", "PORT", m_nPortNum + 1) ;

	/* Contains both Compressed scr and uncompressed cod */
	m_szFullScript += szBuf ;
	m_szFullScript += ".FIL" ; /* PORTn.FIL */

	/* Contains compressed scr */
	m_szComprScriptSource += szBuf ;
	m_szComprScriptSource += ".SC_" ;  /* PORTn.SC_ */

	/* contains uncompressed scr */
	m_szScriptSource += szBuf ;
	m_szScriptSource += ".SCR" ;       /* PORTn.SCR */

	/* contains cod */
	m_szScriptCode += szBuf ;
	m_szScriptCode += ".COD" ;       /* PORTn.COD */

	if (IsPortIP)
	{
		if (!ReadScriptThroEthernet())
			return FALSE ; 
	}
	else
	{
		if (!ReadScriptThroCom())
			return FALSE ; 
	}
	return TRUE ;
}

/*----------------------------------------------------------------------------
	Function : ReadScriptThroEthernet
	Input    : None
	Output   : Boolean stating sucess of read
	Synopsis : Read made thro' Ethernet port
----------------------------------------------------------------------------*/
BOOL CScriptDlg::ReadScriptThroEthernet()
{
	/* First read the script header */
	if (!GetScriptHeaderEnet())
		return FALSE ;
	
	/* TFTP client requests this file from Server */
	sprintf (TFTPGetRemoteFileName, "PORT%d.FIL", m_nPortNum + 1) ;
	/* Name of file that is written to Hard Diak for a TFTP request */
	strcpy (TFTPGetLocalFileName, (LPCSTR) m_szFullScript) ;	

	tftpread TFTPReadScriptDlg ;
	if (TFTPReadScriptDlg.DoModal() != IDOK)
		return FALSE ;

   if (!SeparateScriptScrAndCod())
		return FALSE ;

	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : GetScriptHeaderEnet
	Input    : None
	Output   : Boolean stating success of read
	Synopsis : Reads the Script header thro' Ethernet
----------------------------------------------------------------------------*/
BOOL CScriptDlg::GetScriptHeaderEnet()
{
	strcpy (TFTPGetRemoteFileName, "SCRIPT.HDR") ;

	strcpy (TFTPGetLocalFileName, LoadPath) ;
	strcat (TFTPGetLocalFileName, "SCRIPT.HDR") ;

	tftpread TFTPScriptHdrDlg ;
	if (TFTPScriptHdrDlg.DoModal() != IDOK)
		return FALSE ;

	FILE *fptr ;
	/* SCRIPT.HDR contains the Script header read thro' Ethernet port */
	fptr = fopen (TFTPGetLocalFileName, "rb") ;
	if (fptr == NULL)
	{
		MessageBox ("File open error", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	/* The script header structure is filled */
	fread (&ScriptInfo, sizeof (ScriptInfo), 1, fptr) ;
	fclose (fptr) ;
	/* Delete SCRIPT.HDR */
	remove (TFTPGetLocalFileName) ;
	// check all scripts for validity
	if (ScriptInfo.MagicNumber != cScriptCom.ltl_endian_word ((WORD) MAGIC_NUM))
	{
		int index;

		for(index = 0; index < MAX_NO_OF_WAN_PORTS; index++)
		{
			ScriptInfo.Ports[index].ScrLen  = 0;
			ScriptInfo.Ports[index].CodLen  = 0;
			ScriptInfo.Ports[index].StartAddr  = 0;
		}
	}
	return TRUE ;
}

/*----------------------------------------------------------------------------
	Function : ReadScriptThroCom
	Input    : None
	Output   : Boolean stating success of read
	Synopsis : Reads script thro' COM Port
----------------------------------------------------------------------------*/
BOOL CScriptDlg::ReadScriptThroCom()
{
	if (!GetScriptHeaderCom())
		return FALSE ;
	if (!GetScriptCom())
		return FALSE ;
	if (!SeparateScriptScrAndCod())
		return FALSE ;

	return TRUE ;
}
#define	SCRIPT_HEADER_TIMEOUT_COM	20000			//20 secs - time to just get header
#define	SCRIPT_BLOCK_READ_TIMEOUT_COM	20000		//20 secs - reads 128 bytes
#define	SCRIPT_WRITE_TIMEOUT_COM	40000			//40 secs - total write time
/*----------------------------------------------------------------------------
	Function : GetScriptHeaderCom
	Input    : None
	Output   : Boolean stating success of read
	Synopsis : Gets the script header thro' com port
----------------------------------------------------------------------------*/
BOOL CScriptDlg::GetScriptHeaderCom()
{
	CTRL_FLAG = 0 ;
	Go_on = 1 ;

	BYTE *pInfo ;
	pInfo = (BYTE *) &ScriptInfo ;

	ReadLocReqType RequestLocation ;
	ReadLocRespType RespondLocation ;

	RequestLocation.Ptype =	READ_REQ_TYPE ;
	// RequestLocation.Length = 128 ;

	/* on 7/1/99 */
	RequestLocation.Length = 64 ;

	/* Script header is 512 bytes 
		With the present scheme of reading 128
		bytes in one packet, 4 packets have to be read */
	//for (int i = 0 ; i < 4 ; i ++)
	//{
		cScriptCom.TimeOut = FALSE ;
		// RequestLocation.Address = 
		//	cScriptCom.little_endian (SCRIPT_HEADER + (DWORD) (128 * i)) ;

		RequestLocation.Address = 
			cScriptCom.little_endian (SCRIPT_HEADER) ;
		
		if (ScriptTimerID != SetTimer (ScriptTimerID, (UINT) SCRIPT_HEADER_TIMEOUT_COM, NULL))
		{
			MessageBox ("No Timer available", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
			return FALSE ;
		}
		while (TRUE)
		{
			/* Request the script header (128 bytes at a time) */
			cScriptCom.snd_packet ((BYTE *) &RequestLocation, 
														sizeof (RequestLocation)) ;

			/* Receive the 128 byte packet of script header 
			if (!cScriptCom.rcv_packet ((BYTE *) &RespondLocation,
											128, (BOOL *) &cScriptCom.TimeOut)) 
			{
				Packet received without error 
				break ;
			} */

			if (!cScriptCom.rcv_packet ((BYTE *) &RespondLocation,
											64, (BOOL *) &cScriptCom.TimeOut)) 
			{
				/* Packet received without error */
				break ;
			}


			/* If time expires then something is wrong 
				So come out */
			if (cScriptCom.TimeOut)
			{
				KillTimer (ScriptTimerID) ;
				MessageBox ("Timed out", "ProxyServer - Script Read", MB_OK | MB_ICONINFORMATION) ;
				return FALSE ;
			}
			continue ;
		}

		/* An error has occured */
		if (RespondLocation.ErrCode)
		{
			KillTimer (ScriptTimerID) ;
			MessageBox ("Error reading Script", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
			return FALSE ;
		}
			
		// memcpy ((pInfo + (128 * i)), (BYTE *) RespondLocation.buffer, 128) ;

		memcpy (pInfo, (BYTE *) RespondLocation.buffer, 64) ;
		KillTimer (ScriptTimerID) ;
   //}

	// check all scripts for validity
	if (ScriptInfo.MagicNumber != cScriptCom.ltl_endian_word ((WORD) MAGIC_NUM))
	{
		int index;

		for(index = 0; index < MAX_NO_OF_WAN_PORTS; index++)
		{
			ScriptInfo.Ports[index].ScrLen  = 0;
			ScriptInfo.Ports[index].CodLen  = 0;
			ScriptInfo.Ports[index].StartAddr  = 0;
		}
	}
	return TRUE ;
}

/*----------------------------------------------------------------------------
	Function : SeparateScriptScrAndCod
	Input    : None
	Output   : Boolean stating success of separating src and cod portions
	Synopsis : Reads the file (PORT?.FIL) which contains compressed scr
				  and uncompressed cod.Separates it into PORT?.SC_ and PORT?.COD
				  PORT?.COD contains uncompressed cod
				  PORT?.SC_ contains compressed scr which is converted into
				  PORT?.SCR containing uncompressed scr
----------------------------------------------------------------------------*/
BOOL CScriptDlg::SeparateScriptScrAndCod()
{
	BYTE *pReadBuf = new BYTE[1024] ;
	if (pReadBuf == NULL)
	{
		MessageBox ("Out of memory", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	FILE *sfp, *dfpComprScript, *dfpCode, *dfpSrc ;

	/* Open the file with compressed script and uncompressed cod (Port?.fil) */
	sfp = fopen ((LPCSTR) m_szFullScript, "rb") ;
	if (sfp == NULL)
	{
		MessageBox ("Error reading file", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		delete [] pReadBuf ;
		return FALSE ;
	}

	/* Open the file for compressed script source (port?.sc_) */
	dfpComprScript = fopen ((LPCSTR) m_szComprScriptSource, "wb") ;
	if (dfpComprScript == NULL)
	{
		MessageBox ("Error creating file", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		delete [] pReadBuf ;
		fclose (sfp) ;
		remove ((LPCSTR) m_szFullScript) ;
		return FALSE ;
	}

	WORD nSrcLen = 
		cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].ScrLen) ;
	int nTimes = nSrcLen / 1024 ;
	int nRemains = nSrcLen % 1024 ;
	for (int i = 0 ; i < nTimes ; i ++)
	{
		/* Read port?.fil into buffer */
		fread (pReadBuf, 1, 1024, sfp) ;
		/* Transfer buffer into port?.sc_ */
		fwrite (pReadBuf, 1, 1024, dfpComprScript) ;
	}
	if (nRemains)
	{
		fread (pReadBuf, 1, nRemains, sfp) ;
		fwrite (pReadBuf, 1, nRemains, dfpComprScript) ;
	}
	fclose (dfpComprScript) ;

	dfpComprScript = fopen ((LPCSTR) m_szComprScriptSource, "rb") ;
	if (dfpComprScript == NULL)
	{
		MessageBox ("Error reading file", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		fclose (sfp) ;
		delete [] pReadBuf ;
		remove ((LPCSTR) m_szFullScript) ;
		remove ((LPCSTR) m_szComprScriptSource) ;
		return FALSE ;
	}

	/* open port?.scr for uncompressed scr */
	dfpSrc = fopen ((LPCSTR) m_szScriptSource, "wb") ;
	if (dfpSrc == NULL)
	{
		MessageBox ("Error creating file", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		fclose (dfpComprScript) ;
		fclose (sfp) ;
		delete [] pReadBuf ;
		remove ((LPCSTR) m_szComprScriptSource) ;
		remove ((LPCSTR) m_szFullScript) ;
		return FALSE ;
	}

	/* do the uncompression of scr */
	LZWExpandFromFile (dfpComprScript, dfpSrc) ;
		
	fclose (dfpSrc) ;
	fclose (dfpComprScript) ;
	remove ((LPCSTR) m_szComprScriptSource) ;

	/* open the port?.cod for cod file */
	dfpCode = fopen ((LPCSTR) m_szScriptCode, "wb") ;
	if (dfpCode == NULL)
	{
		MessageBox ("Error creating file", "ProxyServer - Script Read", MB_OK | MB_ICONHAND) ;
		fclose (sfp) ;
		delete [] pReadBuf ;
		return FALSE ;
	}

	BOOL bFileEnd = FALSE ;
	WORD nRead = 0 ;

	/* the file pointer in the port?.fil is pointing
		to the cod section */
	while (!bFileEnd)
	{
		/* read the cod section from port?.fil into buffer */
		nRead = fread (pReadBuf, 1, 1024, sfp) ;
		if (nRead == 0)
			break ;
		if (nRead < 1024)
			bFileEnd = TRUE ;
		/* write from buffer into port?.cod */
		fwrite (pReadBuf, 1, nRead, dfpCode) ;
	}

	fclose (dfpCode) ;
	fclose (sfp) ;
	remove ((LPCSTR) m_szFullScript) ;

	delete [] pReadBuf ;
	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : GetScriptCom
	Input    : None
	Output   : Boolean stating success of read
	Synopsis : Reads the full script into port?.fil
----------------------------------------------------------------------------*/
BOOL CScriptDlg::GetScriptCom()
{
	CTRL_FLAG = 0 ;
	Go_on = 1 ;

	ReadLocReqType ReadLoc ;
	ReadLocRespType RespLoc ;

	/* Address where the total script starts in the flash prom */
	ReadLoc.Address = 
		cScriptCom.little_endian (SCRIPT_ONE_DNLD_ADDR + 
												(m_nPortNum * SINGLE_SCR_SIZE)) ;
	ReadLoc.Ptype = READ_REQ_TYPE ;
	ReadLoc.Length = 128 ;

	CCNFRead cRead ;
	char szBuf[25] ;

	//	Set the gauge to display progress of read.
	CHorzGauge *wndHorzGauge = new CHorzGauge ;
   CRect rc ;
	cRead.SetWindowText ("Reading Script") ;
	(cRead.GetDlgItem (IDC_STATIC_RECT))->GetWindowRect (&rc) ;
	cRead.ScreenToClient (&rc) ;
   wndHorzGauge->Create ((CWnd *) &cRead, rc) ;

   wndHorzGauge->SetGaugeInfo ("",
										  0,
									   100, 
							 GU_ELEVATED,
							GU_DEPRESSED,
									  TRUE,
									 FALSE,
									 FALSE, 
			  RGB (0xff, 0xff, 0xff),
			  RGB (0x80, 0x80, 0x80), 
			        RGB (0, 0, 0xff),
			  RGB (0xc0, 0xc0, 0xc0), 
						  RGB (0, 0, 0)
			  ) ;
    
   AfxGetApp()->LoadCursor (IDC_WAIT) ;
   AfxGetApp()->DoWaitCursor (1) ;

	CStdioFile ComprScript (m_szFullScript,
		CFile::modeCreate | CFile::modeWrite | CFile::typeBinary) ;

	BOOL bError = FALSE ;
	int nRetry = 0 ;
	BOOL bTimer = TRUE ;
	int nError = 0 ;
	DWORD nRead = 0 ;

   DWORD nSrcLen = 
		cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].ScrLen)
		 + cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].CodLen) ;

	if (nSrcLen > SINGLE_SCR_SIZE)
	{
		bError = TRUE ;
		MessageBox ("Error reading script", "ProxyServer - Script", 
												MB_OK | MB_ICONINFORMATION) ;
		goto TimeUp ;
	}
		
	sprintf (szBuf, "%u", (WORD) nSrcLen) ;
	/* total length of script displayed */
	cRead.m_Length = szBuf ;

	sprintf (szBuf, "0x%X", 
		cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].CRC)) ;
	/* crc of script displayed */
	cRead.m_CRC = szBuf ;

	/* To prevent any errors 
	ScriptInfo.Ports[m_nPortNum].DateStamp[24] = 0x00 ; 
	date stamp for a particular script 
	cRead.m_Date = ScriptInfo.Ports[m_nPortNum].DateStamp ; */

	/*  Brindha on 7/1/99. 
 	    Since, Datestamp is removed from ScriptHeader Structure */

	cRead.m_Date = StrNull ;
  	cRead.m_DateWnd.EnableWindow (FALSE) ;
  	cRead.m_DateWnd.ShowWindow (SW_HIDE) ;
 
	cRead.UpdateData (FALSE) ;

	cScriptCom.CODE_CRC = 0xFFFF ;

	/* If script length = 0, no need to read */
	if (nSrcLen)
	{
		while (nRead < nSrcLen)
		{
			if (nSrcLen)
	      	wndHorzGauge->UpdateProgress ((nRead * 100) / nSrcLen) ;

Retry :
			CTRL_FLAG = 0 ;
			cScriptCom.TimeOut = FALSE ;

			/* Kill the timer after every packet read
				is read successfully 
				and before a new packet is to be read
				Set the timer once again. This is done
				to prevent unnessecary waiting */
			if (ScriptTimerID != SetTimer (ScriptTimerID, (UINT) SCRIPT_BLOCK_READ_TIMEOUT_COM, NULL))
			{
				/* No timer available */
				bTimer = FALSE ;
				goto TimeUp ;
			}

			while (TRUE)
			{
				cScriptCom.snd_packet ((BYTE *) &ReadLoc, sizeof (ReadLoc)) ;
				if (!cScriptCom.rcv_packet ((BYTE *) &RespLoc, 
						sizeof (RespLoc), (BOOL *) &cScriptCom.TimeOut))
				{
					break ;  // No CRC Error
				}
				if (cScriptCom.TimeOut)
					break ;
			}

			if (!cScriptCom.TimeOut)
			{
				/* If not timed out */
				switch (RespLoc.Ptype)
				{
					/* Reply in response to request */
					case READ_RESP_TYPE :
						if (!RespLoc.ErrCode)
						{	/* No error */
							int i ;
							nRead += 128 ;

							if (nRead <= nSrcLen)
							{
								/* Write packet to port?.fil */
								ComprScript.Write (RespLoc.buffer, 128) ;
								/* Maintain local crc for received packet */					
								for (i = 0 ; i < 128 ; i ++)
									cScriptCom.updatecrc (RespLoc.buffer[i]) ;
							}
							else
							{
								/* This is the last packet */
								int nLastRead ;
								nLastRead = 128 - (int) (nRead - nSrcLen) ;
								ComprScript.Write (RespLoc.buffer, nLastRead) ;
								for (i = 0 ; i < nLastRead ; i ++)
									cScriptCom.updatecrc (RespLoc.buffer[i]) ;
							}

							/* update the address of location to read next */
							ReadLoc.Address = 
								cScriptCom.little_endian (ReadLoc.Address) ;
							ReadLoc.Address += 128 ;
							ReadLoc.Address = 
								cScriptCom.little_endian (ReadLoc.Address) ;
						}
						else
						{
							/* Error occured. So update error counter */
							nError ++ ;
							if (nError >= 4)
								goto TimeUp ;
						}
						break ;

					default :
						/* Unknown packet */
						break ;
				}
			}	
			else
			{
				/* Time out */
				cScriptCom.FlushReadBuf() ;
				KillTimer (ScriptTimerID) ;
				/* Update timer retry counter */
				nRetry ++ ;
				if (nRetry <= 2)
					goto Retry ;
				else
				 	goto TimeUp ;
			}
			/* Kill the timer after every packet read
				is read successfully 
				and before a new packet is to be read
				Set the timer once again. This is done
				to prevent unnessecary waiting */
			KillTimer (ScriptTimerID) ;
		}

		/* Is Local CRC != Script Header CRC ? */
		cScriptCom.updatecrc ((BYTE) 0) ;
		cScriptCom.updatecrc ((BYTE) 0) ;
		if (cScriptCom.CODE_CRC !=
				cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].CRC))
		{
			bError = TRUE ;
			cRead.MessageBox ("Error in checksum", "COM Port - Read", MB_OK | MB_ICONHAND) ;
		}
	}	

TimeUp :
	KillTimer (ScriptTimerID) ;
	wndHorzGauge->DestroyGauge() ;
	delete wndHorzGauge ;
	AfxGetApp()->DoWaitCursor (0) ;

	ComprScript.Close() ;

	if (bError)
		return FALSE ;

	if (!bTimer)
	{
		cRead.MessageBox ("No Timer", "Com Port - Read", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}
	if (cScriptCom.TimeOut)
	{
		cRead.MessageBox ("Timed out", "Com Port - Read", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}
	if (nError >= 4)
	{
		cRead.MessageBox ("Error reading Script",	NULL, MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : OnBtnDnldScript
	Input    : None 
	Output   : None
	Synopsis : Downloads the combined script either via com or ethernet
				  Checks for unnecessary download.
				  After necessary action is taken it comes out of Script dialog
----------------------------------------------------------------------------*/
void CScriptDlg::OnBtnDnldScript()
{
	// TODO: Add your control notification handler code here

	if (!IsScrModified())
	{
		if (IsCompilationNecessary())
		{
			MessageBox ("Compilation is necessary", "ProxyServer - Script", MB_OK | MB_ICONINFORMATION) ;
			return ;
		}
		else
			return ;
	}

	if (IsCompilationNecessary())
	{
		MessageBox ("Compilation is necessary", "ProxyServer - Script", MB_OK | MB_ICONINFORMATION) ;
		return ;
	}

	if (!CombineScrAndCod())
	{
		MessageBox ("Script write error", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		CDialog::OnCancel() ;
		return ;
	}

	if (IsPortIP)
		WriteThroEnet() ;
	else
		WriteThroCom() ;

	CDialog::OnCancel() ;
}


/*----------------------------------------------------------------------------
	Function : WriteThroEnet
	Input    : None
	Output   : None
	Synopsis : Writes complete script and header via Ethernet
----------------------------------------------------------------------------*/
void CScriptDlg::WriteThroEnet()
{
	if (!WriteScriptScrAndCodToEnet())
		return ;
	WriteScriptHeaderEnet() ;
}

/*----------------------------------------------------------------------------
	Function : WriteScriptScrAndCodToEnet
	Input    : None
	Output   : Boolean indicating success of write
	Synopsis : Writes complete script via Ethernet
----------------------------------------------------------------------------*/
BOOL CScriptDlg::WriteScriptScrAndCodToEnet()
{
	strcpy (TFTPPutLocalFileName, m_szFullScript) ;
	sprintf (TFTPPutRemoteFileName, "PORT%d.FIL", m_nPortNum + 1) ;
	tftpwrit TFTPWriteScript ;
	if (TFTPWriteScript.DoModal() != IDOK)
		return FALSE ;
	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : WriteScriptHeaderEnet
	Input    : None
	Output   : None
	Synopsis : Writes script header via Ethernet
----------------------------------------------------------------------------*/
void CScriptDlg::WriteScriptHeaderEnet()
{
	if (!CalculateCRCForEtherNet())
		return ;
	remove ((LPCSTR) m_szFullScript) ;

	ScriptInfo.Ports[m_nPortNum].ScrLen =
			cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].ScrLen) ;
	ScriptInfo.Ports[m_nPortNum].CodLen =
		cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].CodLen) ;
	ScriptInfo.Ports[m_nPortNum].StartAddr = 
			cScriptCom.little_endian (
				SCRIPT_ONE_DNLD_ADDR + (m_nPortNum * SINGLE_SCR_SIZE)) ;

	/*  Brindha on 7/1/99. 
 	    Since, Datestamp is removed from ScriptHeader Structure */

	/* cScriptCom.SetDateStamp() ;
	strcpy (ScriptInfo.Ports[m_nPortNum].DateStamp, cScriptCom.DateStamp) ; */
	ScriptInfo.NumberOfWANPorts = cScriptCom.ltl_endian_word ((int) MAX_NO_OF_WAN_PORTS) ;
	ScriptInfo.MagicNumber = cScriptCom.ltl_endian_word ((WORD) MAGIC_NUM) ;

	strcpy (TFTPPutLocalFileName, LoadPath) ;
	strcat (TFTPPutLocalFileName, "SCRIPT.HDR") ;
	strcpy (TFTPPutRemoteFileName, "SCRIPT.HDR") ;

	FILE *fptr ;
	fptr = fopen ((LPCSTR) TFTPPutLocalFileName, "wb") ;
	if (fptr == NULL)
	{
		MessageBox ("File Creation error", "ProxyServer - Script", 
														MB_OK | MB_ICONHAND) ;
		return ;
	}
	fwrite ((BYTE *) &ScriptInfo, 1, sizeof (ScriptInfo), fptr) ;
	fclose (fptr) ;

	tftpwrit TFTPWriteScript ;
	TFTPWriteScript.DoModal() ;
	remove ((LPCSTR) TFTPPutLocalFileName) ;
}

/*----------------------------------------------------------------------------
	Function : WriteThroCom
	Input    : None
	Output   : None
	Synopsis : Writes complete script and its header thro' com port
----------------------------------------------------------------------------*/
void CScriptDlg::WriteThroCom()
{
	if (!WriteScriptScrAndCodToCom())
		return ;

	WriteScriptHeaderCom() ;
}


/*----------------------------------------------------------------------------
	Function : WriteScriptScrAndCodToCom
	Input    : None
	Output   : None
	Synopsis : Writes complete script via COM Port
----------------------------------------------------------------------------*/
BOOL CScriptDlg::WriteScriptScrAndCodToCom()
{
	FILE *fptr ;
	fptr = fopen (m_szFullScript, "rb") ;
	if (fptr == NULL)
	{
		MessageBox ("Cannot open script source and code", 
					"ProxyServer - Script Read",	MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	BYTE *ReadBuf = new BYTE[512] ;
	if (ReadBuf == NULL)
	{
		fclose (fptr) ;
		MessageBox ("Out of memory", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}
	
	Go_on = 1 ;
	CTRL_FLAG = 0 ;

	cScriptCom.TimeOut = FALSE ;
	cScriptCom.CODE_CRC = 0xFFFF ;
	cScriptCom.UDBPktCRC = 0xFFFF ;

	if (ScriptTimerID != SetTimer (ScriptTimerID, (UINT) SCRIPT_WRITE_TIMEOUT_COM, NULL))
	{
		MessageBox ("No Timer", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		delete [] ReadBuf ;
		fclose (fptr) ;
		return FALSE ;
	}

	cScriptCom.PrevComprCodeSize = 0 ;
	cScriptCom.ComprCodeSize = 0 ;

	CDnldProg *p_cDnldProg = new (CDnldProg) ;
	p_cDnldProg->m_AbortButton.EnableWindow (FALSE) ;
	p_cDnldProg->m_AbortButton.ShowWindow (SW_HIDE) ;
	p_cDnldProg->SetWindowText ("Downloading Script") ;
	p_cDnldProg->m_FileName = "Script" ;

	WORD nTotLen =	ScriptInfo.Ports[m_nPortNum].CodLen
			            + ScriptInfo.Ports[m_nPortNum].ScrLen ;
	char szBuf[20] ;
	sprintf (szBuf, "%u", nTotLen) ;
	p_cDnldProg->m_Length = szBuf ;
	p_cDnldProg->UpdateData (FALSE) ;

	CRect rc ;
	CHorzGauge *wndHorzGauge = new (CHorzGauge) ;        
	(p_cDnldProg->GetDlgItem (IDC_STATIC_RECT))->GetWindowRect (&rc) ;
	p_cDnldProg->ScreenToClient (&rc) ;

	wndHorzGauge->Create ((CWnd *) p_cDnldProg, rc) ;
	wndHorzGauge->SetGaugeInfo ("",  0, 100, GU_ELEVATED, GU_DEPRESSED,
    	TRUE, FALSE, FALSE, RGB (0xff, 0xff, 0xff), 
	    	RGB (0x80,0x80, 0x80), RGB (0, 0, 0xff), 
				RGB (0xc0, 0xc0, 0xc0), RGB (0, 0, 0)) ;
	wndHorzGauge->UpdateProgress (0) ;

	BOOL bFileEnd = FALSE ;
	WORD nRead = 0 ;
	DWORD nTotRead = 0 ;
	UDBDnldAddress = SCRIPT_ONE_DNLD_ADDR + (m_nPortNum * SINGLE_SCR_SIZE) ;

	while (!bFileEnd)
	{
		nRead = fread (ReadBuf, 1, 512, fptr) ;

		if (nRead < 512)
			bFileEnd = TRUE ;
		if (nRead == 0)
			break ;
		
		for (int i = 0 ; i < (int)nRead ; i ++)
			cScriptCom.updatecrc (ReadBuf[i]) ;

		cScriptCom.ComprCodeSize += nRead ;
		cScriptCom.UDBActualLen = nRead ;
		cScriptCom.send_data_online (ReadBuf, nRead, &cScriptCom.TimeOut, TRUE) ;

		if (cScriptCom.TimeOut)
			goto TimeOut ;
		nTotRead += (DWORD) nRead ;

		if (nTotLen)
			wndHorzGauge->UpdateProgress ((nTotRead * 100) / nTotLen) ;
	}

TimeOut :
	fclose (fptr) ;
	remove ((LPCSTR) m_szFullScript) ;
	wndHorzGauge->DestroyGauge() ;
	delete wndHorzGauge ;
	delete p_cDnldProg ;

	if (cScriptCom.TimeOut)
	{
		MessageBox ("Timed out", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	return TRUE ;
}

/*----------------------------------------------------------------------------
	Function : WriteScriptHeaderCom
	Input    : None
	Output   : None
	Synopsis : Writes the script header via com port
----------------------------------------------------------------------------*/
void CScriptDlg::WriteScriptHeaderCom()
{
	char TempBuf[50] ;
	cScriptCom.updatecrc ((BYTE) 0) ;
	cScriptCom.updatecrc ((BYTE) 0) ;

	ScriptInfo.Ports[m_nPortNum].CRC = 
			cScriptCom.ltl_endian_word (cScriptCom.CODE_CRC) ;

	ScriptInfo.Ports[m_nPortNum].ScrLen =
			cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].ScrLen) ;			
	ScriptInfo.Ports[m_nPortNum].CodLen =
			cScriptCom.ltl_endian_word (ScriptInfo.Ports[m_nPortNum].CodLen) ;			
	ScriptInfo.Ports[m_nPortNum].StartAddr = 
			cScriptCom.little_endian (
				SCRIPT_ONE_DNLD_ADDR + (m_nPortNum * SINGLE_SCR_SIZE)) ;

	/*  Brindha on 7/1/99. 
 	    Since, Datestamp is removed from ScriptHeader Structure */

	/* cScriptCom.SetDateStamp() ;
	strcpy (ScriptInfo.Ports[m_nPortNum].DateStamp, cScriptCom.DateStamp) ; */
	ScriptInfo.NumberOfWANPorts = cScriptCom.ltl_endian_word ((int) MAX_NO_OF_WAN_PORTS) ;
	ScriptInfo.MagicNumber = cScriptCom.ltl_endian_word ((WORD) MAGIC_NUM) ;

/* Jo */
#if 0
	cScriptCom.ComprCodeSize = 512 ;
	cScriptCom.UDBActualLen = 512 ;
#endif
/* Jo */
	UDBDnldAddress = SCRIPT_HEADER ;
	cScriptCom.PrevComprCodeSize = 0 ;

	cScriptCom.ComprCodeSize = 64 ;
	cScriptCom.UDBActualLen =  64 ;

   cScriptCom.send_data_online ((BYTE *) &ScriptInfo, sizeof (ScriptInfo), 
	  	(BOOL *) &cScriptCom.TimeOut, TRUE) ;

	KillTimer (ScriptTimerID) ;
}

/*----------------------------------------------------------------------------
	Function : OnCancel
	Input    : None
	Output   : None
	Synopsis : If source has been modified verifies
				  if download is to be done	
----------------------------------------------------------------------------*/
void CScriptDlg::OnCancel()
{
	// TODO: Add extra cleanup here

	int nRetVal ;
	if (IsScrModified())
	{
		nRetVal = MessageBox ("Source modified.\nDo you want to download it ?",
								"Script", MB_YESNO | MB_ICONQUESTION) ;
		if (nRetVal == IDYES)
			return ;
	}
	
	CDialog::OnCancel();
}

/*----------------------------------------------------------------------------
	Function : OnBtnEditScript
	Input    : None
	Output   : None
	Synopsis : If the scr file does not exist it is created
				  and it is opened using SCREDIT.EXE
				  If scredit.exe not found in the current dir
				  give warning and come out
----------------------------------------------------------------------------*/
void CScriptDlg::OnBtnEditScript()
{
	// TODO: Add your control notification handler code here
	char szFileCmd[400] ;
	FILE *fptr ;

	/* If file Port?.scr does not exist create it */
	fptr = fopen ((LPCSTR) m_szScriptSource, "rb") ;
	if (fptr == NULL)
		CreateScrFile() ;
	fclose (fptr) ;

	/* Form command line parameter (file to be opened)
		for SCREDIT.EXE */
	lstrcpy ((LPSTR) szFileCmd, LoadPath) ;
	strcat (szFileCmd, "SCREDIT.EXE") ;

	/* First check whether scredit.exe is present */
	int Handle ;
	Handle = _open (szFileCmd, _O_RDONLY) ;
	if (Handle == -1)
	{
		MessageBox ("Cannot find SCREDIT.EXE", "ProxyServer - Script", MB_OK | MB_ICONINFORMATION) ; 
		return ;
	}
	close (Handle) ;

	strcat (szFileCmd, " ") ;  /* space between exe and command line arg */
	strcat (szFileCmd, (LPCSTR) m_szScriptSource) ;

	WinExec (szFileCmd, SW_SHOW) ;
}


/*----------------------------------------------------------------------------
	Function : OnBtnCompile
	Input    : None
	Output   : None
	Synopsis : Checks unnecessary compilation
			     If needed calls SCRCMP.EXE with application name and scr file path	
----------------------------------------------------------------------------*/
void CScriptDlg::OnBtnCompile()
{
	// TODO: Add your control notification handler code here

	if (!IsCompilationNecessary())
		return ;

	char szFileCmd[400] ;
	
	lstrcpy ((LPSTR) szFileCmd, LoadPath) ;
	strcat (szFileCmd, "SCRCMP.EXE") ;
	
	/* check whether sc.exe exists in the current working dir */
	int Handle ;
	Handle = _open (szFileCmd, _O_RDONLY) ;
	if (Handle == -1)
	{
		MessageBox ("SCRCMP.EXE not found", "ProxyServer - Script", MB_OK | MB_ICONINFORMATION) ;
		return ;
	}
	close (Handle) ;

	strcat (szFileCmd, " MEWTERM ") ;
	strcat (szFileCmd, (LPCSTR) m_szScriptSource) ;

	WinExec (szFileCmd, SW_SHOW) ;
}


/*----------------------------------------------------------------------------
	Function : OnTimer
	Input    : Event that needs a timer 
	Output   : None
	Synopsis : Processes timer messages for various windows
----------------------------------------------------------------------------*/
void CScriptDlg::OnTimer(UINT nIDEvent)
{
	// TODO: Add your message handler code here and/or call default
	
	switch (nIDEvent)
	{
		case ScriptTimerID :
 			if (CTRL_FLAG == 0) 
 			{	// Can read char from COM port.
			 	char ch ;
				switch (Go_on)
				{
					case 0 :			// Timeout occured and char couldn't be read.
						Go_on = 2 ;	// Make it true.
						break ;		// So that it comes out of rcv_char.
							
					case 1 :
						Go_on = 0 ;
						if (!cScriptCom.timer_rcv_char (&ch))
						{
							break ;
						}
						if (ExpectingReplyPacket && (ch == (BYTE) 0x7e))
						{
							CTRL_FLAG = 1 ;	// Relinquish control to rcv_packet
						}
						break ;

					case 2 :
						Go_on = 1 ;	// Make it ready for reading char next time.
						break ;
				}
			}

		default :
			KillTimer (nIDEvent) ;
			cScriptCom.TimeOut = TRUE ;
			break ;
	}

	CDialog::OnTimer(nIDEvent);
}


/*----------------------------------------------------------------------------
	Function : CompressSrc
	Input    : Source and Destination file pointers 
	Output   : Boolean indicating success of compression
	Synopsis : 
----------------------------------------------------------------------------*/
BOOL CScriptDlg::CompressSrc (FILE *sfptr, FILE *dfptr)
{
	ScriptInfo.Ports[m_nPortNum].ScrLen = 0 ;
		
	const WORD nReadSize = 8 * 1024 ;
	BYTE *ReadBuf = new BYTE[nReadSize] ;
	if (ReadBuf == NULL)
	{
		MessageBox ("Out of memory", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	const WORD nCompSize = 10 * 1024 ;
	BYTE *CompBuf = new BYTE[nCompSize] ;
	if (CompBuf == NULL)
	{
		delete [] ReadBuf ;
		MessageBox ("Out of memory", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	WORD nRead, nComp, nWrite ;
	BOOL bFileEnd = FALSE ;
	BYTE *ptrWrite ;
	while (!bFileEnd)
	{
		nRead = fread (ReadBuf, 1, nReadSize, sfptr) ;

		if (nRead < nReadSize)
			bFileEnd = TRUE ;
		if (nRead == 0)
			break ;

		nComp = LZ15VCompressBuf (ReadBuf, nRead, 
							CompBuf + sizeof (nCompSize), nCompSize) ;

		if (nComp == 0 || nComp == nRead)
		{
			*((int *) CompBuf) = 
				(signed) cScriptCom.ltl_endian_word ((unsigned int) -(int)nRead) ;
			nWrite = nRead + sizeof (nRead) ;
			memcpy ((BYTE *) ((BYTE *) CompBuf + sizeof (int)),
																ReadBuf, nRead) ;
			ptrWrite = CompBuf ;
		}
		else
		{
			*((int *) CompBuf) = cScriptCom.ltl_endian_word (nComp) ;
			nWrite = nComp + sizeof (nComp) ;
			ptrWrite = CompBuf ;
		}
		fwrite (ptrWrite, 1, nWrite, dfptr) ;
		ScriptInfo.Ports[m_nPortNum].ScrLen += nWrite ;		
	}

	delete [] ReadBuf ;
	delete [] CompBuf ;

	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : IsCompilationNecessary
	Input    : none
	Output   : Boolean 
	Synopsis : checks whether scr is newer than cod
				  (Just like NMAKE facility)	
----------------------------------------------------------------------------*/
BOOL CScriptDlg::IsCompilationNecessary()
{
	struct _stat FileStatusScr ;
	struct _stat FileStatusCod ;

	if (_stat ((LPCSTR) m_szScriptSource, &FileStatusScr) == ENOENT) 
	{
		MessageBox ("No source", "ProxyServer - Script", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	if (_stat ((LPCSTR) m_szScriptCode, &FileStatusCod) == ENOENT)
		return TRUE ;

	return (FileStatusScr.st_mtime > FileStatusCod.st_mtime) ;
}


/*----------------------------------------------------------------------------
	Function : GetOldScrTime
	Input    : None
	Output   : None
	Synopsis : Gets the modification time of SCR file 
----------------------------------------------------------------------------*/
void CScriptDlg::GetOldScrTime()
{
	struct _stat OldScr ;

	if (_stat ((LPCSTR) m_szScriptSource, &OldScr) == ENOENT)
		return ;

	m_timeOldScr = OldScr.st_mtime ;
	return ;
}


/*----------------------------------------------------------------------------
	Function : IsScrModified
	Input    : None
	Output   : Boolean
	Synopsis : Checks whether scr file has been modified 
				  after having read it from the PROM last time
----------------------------------------------------------------------------*/
BOOL CScriptDlg::IsScrModified()
{
	struct _stat ScrFileStat ;
	if (_stat ((LPCSTR) m_szScriptSource, &ScrFileStat) == ENOENT)
	{
		MessageBox ("Source not found", "ProxyServer - Script", MB_OK | MB_ICONINFORMATION) ;
		return FALSE ;
	}

	return (ScrFileStat.st_mtime > m_timeOldScr) ;
}


/*----------------------------------------------------------------------------
	Function : CombineScrAndCod
	Input    : None
	Output   : Boolean
	Synopsis : Compresses port?.scr into port?.sc_
				  Puts port?.sc_ into port?.fil
				  Appends port?.cod to port?.fil
----------------------------------------------------------------------------*/
BOOL CScriptDlg::CombineScrAndCod()
{
	FILE *fptrBoth, *fptrSrc, *fptrCode ;

	fptrBoth = fopen ((LPCSTR) m_szFullScript, "wb") ;
	if (fptrBoth == NULL)
	{
		MessageBox ("File creation error", "ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	fptrSrc = fopen ((LPCSTR) m_szScriptSource, "rb") ;
	if (fptrSrc == NULL)
	{
		MessageBox ("Source script cannot be opened", "ProxyServer - Script Write",
									MB_OK | MB_ICONHAND) ;
		fclose (fptrBoth) ;
		return FALSE ;
	}

	if (!CompressSrc (fptrSrc, fptrBoth))
	{
		fclose (fptrBoth) ;
		fclose (fptrSrc) ;
	 	return FALSE ;
	}
	fclose (fptrSrc) ;

	fptrCode = fopen ((LPCSTR) m_szScriptCode, "rb") ;
	if (fptrCode == NULL)
	{
		MessageBox ("Compiled script cannot be opened",
				"ProxyServer - Script Write", MB_OK | MB_ICONHAND) ;
		fclose (fptrBoth) ;
		return FALSE ;
	}

	BOOL bFileEnd = FALSE ;
	WORD nRead ;

	BYTE *ReadBuf = new BYTE[1024] ;
	if (ReadBuf == NULL)
	{
		MessageBox ("Out of Memory", "ProxyServer - Script Write",
														MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	ScriptInfo.Ports[m_nPortNum].CodLen = 0 ;
	while (!bFileEnd)
	{
		nRead = fread (ReadBuf, 1, 1024, fptrCode) ;

		if (nRead < 1024)
			bFileEnd = TRUE ;
		if (nRead == 0)
			break ;

		fwrite (ReadBuf, 1, nRead, fptrBoth) ;
		ScriptInfo.Ports[m_nPortNum].CodLen += nRead ;
	}
	delete [] ReadBuf ;

	fclose (fptrCode) ;
	fclose (fptrBoth) ;

	/* As of now total size that can be downloaded
		cannot exceed 2KB */
	if (FileSizeOverFlow())
		return FALSE ;

	return TRUE ;
}


/*----------------------------------------------------------------------------
	Function : FileSizeOverFlow
	Input    : None
	Output   : Boolean
	Synopsis : Examines whether full script (port?.fil) exceeds 2K
----------------------------------------------------------------------------*/
BOOL CScriptDlg::FileSizeOverFlow()
{
 	int FileHandleBoth = _open ((LPCSTR) m_szFullScript, _O_RDONLY) ;
	if (FileHandleBoth == -1)
	{
		MessageBox ("File Open error", "ProxyServer - Script Write", 
															MB_OK | MB_ICONHAND) ;
		return TRUE ;
	}

	/* Set the file handle to the beginning */
	_lseek (FileHandleBoth, 0L, SEEK_SET) ;
	/* Move to th end of file */
	if (_lseek (FileHandleBoth, 0L, SEEK_END) > SINGLE_SCR_SIZE)
	{
		MessageBox ("Script source and code cannot exceed 2K",
							"ProxyServer - Script Write", MB_OK | MB_ICONINFORMATION) ;
		return TRUE ;
	}
	_close (FileHandleBoth) ;
	return FALSE ;
}


/*----------------------------------------------------------------------------
	Function : CreateScrFile
	Input    : None
	Output   : Boolean stating success of creation of SCR file
	Synopsis : Creates an SCR file which is empty
----------------------------------------------------------------------------*/
BOOL CScriptDlg::CreateScrFile()
{
	FILE *fptr ;
	fptr = fopen ((LPCSTR) m_szScriptSource, "wb") ;
	if (fptr == NULL)
	{
		MessageBox ("File creation error", "ProxyServer - Script",
															MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}
	fclose (fptr) ;
	return TRUE ;
}



BOOL CScriptDlg::CalculateCRCForEtherNet()
{
	FILE *fptr ;
	fptr = fopen ((LPCSTR) m_szFullScript, "rb") ;

	if (fptr == NULL)
	{
		MessageBox ("File Open Error", "ProxyServer - Script Write",
															MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	BYTE *ReadBuf = new BYTE[512] ;
	if (ReadBuf == NULL)
	{
		fclose (fptr) ;
		remove ((LPCSTR) m_szFullScript) ;
		MessageBox ("Out of memory", "ProxyServer - Script Write", 
														MB_OK | MB_ICONHAND) ;
		return FALSE ;
	}

	BOOL bFileEnd = FALSE ;
	int nRead ;
	cScriptCom.CODE_CRC = 0xFFFF ;
	while (!bFileEnd)
	{
		nRead = fread (ReadBuf, 1, 512, fptr) ;
		if (nRead < 512)
			bFileEnd = TRUE ;
		if (!nRead)
			break ;
		for (int i = 0 ; i < nRead ; i ++)
			cScriptCom.updatecrc (ReadBuf[i]) ;
	}
	cScriptCom.updatecrc ((BYTE) 0) ;
	cScriptCom.updatecrc ((BYTE) 0) ;
	fclose (fptr) ;
	delete [] ReadBuf ;

	ScriptInfo.Ports[m_nPortNum].CRC 
							= cScriptCom.ltl_endian_word (cScriptCom.CODE_CRC) ;
	return TRUE ;
}


/*----------------------------------------------------------------------------
										END OF FILE
----------------------------------------------------------------------------*/
