// CTabDialog:
// written by Gerry High & David Hollifield
// 74750,2456 CIS
// for more info see the tabview.wri document
//
//      25-May-94 - Added 3D look to CTabDialog frame.
//                - Fixed memory leak in destructor (thanks Brett Tinling).
//                - Added context-sensitive help support for active tab.
//                - Added SwitchTo and SwitchFrom notification messages

#include "stdafx.h"
#include <ctype.h>
#include <afxpriv.h>
#include "tabdlg.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CTabDialog

IMPLEMENT_DYNAMIC(CTabDialog, CDialog)

BEGIN_MESSAGE_MAP(CTabDialog, CDialog)
	//{{AFX_MSG_MAP(CTabDialog)
	ON_WM_ERASEBKGND()
	ON_WM_LBUTTONDOWN()
	ON_WM_MOUSEACTIVATE()
	ON_WM_PAINT()
	ON_WM_SETFOCUS()
	ON_WM_SIZE()
   ON_WM_MENUCHAR()
	ON_MESSAGE(WM_COMMANDHELP, OnCommandHelp)
	ON_MESSAGE(WM_HELPHITTEST, OnHelpHitTest)
	//}}AFX_MSG_MAP
#ifdef  __CTL3D__
	ON_WM_CTLCOLOR()
	ON_WM_NCACTIVATE()
	ON_WM_NCPAINT()
#endif
END_MESSAGE_MAP()

static char BASED_CODE szSwitchFrom[]           = "tabdlg_SwitchFrom";
static char BASED_CODE szSwitchTo[]             = "tabdlg_SwitchTo";

const UINT NEAR msgSwitchFrom                           = ::RegisterWindowMessage(szSwitchFrom);
const UINT NEAR msgSwitchTo                                     = ::RegisterWindowMessage(szSwitchTo);

/////////////////////////////////////////////////////////////////////////////
// CTabDialog construction/destruction

CTabDialog::CTabDialog()
	: CDialog()
{
	Construct();
}

CTabDialog::CTabDialog(LPCSTR lpszTemplateName, CWnd* pParentWnd /*= NULL*/)
	: CDialog(lpszTemplateName, pParentWnd)
{
	Construct();
}

CTabDialog::CTabDialog(UINT nIDTemplate, CWnd* pParentWnd /*= NULL*/)
	: CDialog(nIDTemplate, pParentWnd)
{
	Construct();
}

void CTabDialog::Construct()
{
	m_curView = NULL;
	m_nTabs = 0;
	m_lookAndFeel = LAF_MSWORD;
}

CTabDialog::~CTabDialog()
{
	destroyFonts();
}

static BOOL NEAR bActive;
static int  NEAR dlgReturnCode = IDCANCEL;

int CTabDialog::DoModal()
{
	if (CDialog::Create(m_lpDialogTemplate, m_pParentWnd))
	{
		GetParent()->EnableWindow(FALSE);
		bActive = TRUE;

		ShowWindow(SW_SHOW);    // in case WS_VISIBLE not set (just like a
								//  real modal dialog)
		while (TRUE)
		{
			MSG msg;

			if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{
				if (!bActive)
				{
					break;
				}

				CTabInfo* pTab;

				if (m_nTabs == 0)
					goto DoDispatch;

				pTab = (CTabInfo*) m_tabArray[m_tabArray.m_curTab];

				if (!((CDialog*)pTab->m_pView)->IsDialogMessage(&msg))
				{
DoDispatch:
					TranslateMessage(&msg);
					DispatchMessage(&msg);
				}
			}
		}

		GetParent()->EnableWindow(TRUE);
	   DestroyWindow();
		bActive = TRUE;
		return dlgReturnCode;
	}
	return IDCANCEL;
}

/////////////////////////////////////////////////////////////////////////////
// CTabDialog operations

CDialog* CTabDialog::addTabView(
	CDialog* pDialog,
	char* tabLabel,
	BOOL border,
	BOOL show,
	int tabWidth
	)
{
	// Setup defaults

	if(!tabLabel)
		return NULL;

	CTabInfo* pTab = new CTabInfo;
	if(!pTab)
		return NULL;
		
	pTab->m_pView = createTabView(pDialog, this, border, show);
	if(!pTab->m_pView)
	{
		delete pTab;
		return NULL;
	}
	pTab->m_tabWidth = tabWidth;
	int sLen = strlen(tabLabel);
	pTab->m_tabLabel = (char *)new char[sLen + 1 ];
	strcpy(pTab->m_tabLabel,tabLabel);
	for(int i=0;i<sLen;i++)
	{
		if(tabLabel[i] == '&' && i != (sLen -1))
		{
			pTab->m_mnemonic = (char)(DWORD)AnsiUpper((LPSTR)MAKELP(0,tabLabel[i+1]));
			if(m_tabArray.m_position == TABSONLEFT ||
			   m_tabArray.m_position == TABSONLEFTBOT)
				memmove(&tabLabel[i],&tabLabel[i+1],sLen - i-1);
			break;
		}
	}
	
	m_tabArray.Add(pTab);
	 m_nTabs++;
    if (show)
	m_tabArray.m_curTab = m_nTabs - 1;

	return (CDialog*)pTab->m_pView;
}

BOOL CTabDialog::doSysCommand(UINT nID,LONG lParam)
{
	WPARAM wCmdType = nID & 0xFFF0;
	WORD dummy = HIWORD(lParam);
	char mnemonic = (char)(DWORD)AnsiUpper((LPSTR)MAKELP(0,(char)LOWORD(lParam)));
	switch(wCmdType)
	{
		case SC_KEYMENU:
			if(dummy != 0)
				break;
			for(int i = 0; i < m_nTabs; i++)
			{
				CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
				if(mnemonic == pTab->m_mnemonic && pTab->m_active)
				{
					switchTab(i);
					return 1;
				}
			}
			break;
	}
	return 0;
}

void CTabDialog::enableView(int viewIndex, BOOL bEnable)
{
	CTabInfo* pTab = (CTabInfo*)m_tabArray[viewIndex];
	if(viewIndex < 0 || viewIndex >= m_nTabs)
		return;

//can't disable current view
	if(m_curView != pTab->m_pView && bEnable != pTab->m_active)
	{
		pTab->m_active = bEnable;
		if(IsWindowVisible()) InvalidateRect(NULL);
	}
}

void CTabDialog::setFrameBorderOn(BOOL on)
{
	if(m_tabArray.m_frameBorderOn == on)
		return;
		
	m_tabArray.m_frameBorderOn = on;

	if(IsWindowVisible())
		InvalidateRect(NULL);
}

void  CTabDialog::setLAF(eLookAndFeel LAF)
{
	if(LAF == m_lookAndFeel)
		return;
		
	m_lookAndFeel = LAF;

	if(IsWindowVisible())
		InvalidateRect(NULL);
}

void CTabDialog::setMargin(int margin)
{
	if(margin == m_tabArray.m_margin)
		return;
		
	m_tabArray.m_margin = margin;

	repositionViews();
	if(IsWindowVisible())
		InvalidateRect(NULL);
}

void CTabDialog::setTabHeight(int height)
{
	if(height == m_tabArray.m_tabHeight)
		return;
		
	m_tabArray.m_tabHeight = height;

	repositionViews();
	if(IsWindowVisible())
		InvalidateRect(NULL);
}

void CTabDialog::setTabPosition (eTabPosition position)
{
	int i, j ;

	if (position == m_tabArray.m_position)
		return ;
			
	switch (position)
	{
		case TABSONLEFT :
		case TABSONLEFTBOT :
		case TABSONRIGHT :
		case TABSONRIGHTBOT :
			m_tabArray.m_position = position ;

			for (j = 0 ;j < m_nTabs ;j ++)
			{
				CTabInfo* pTab = (CTabInfo*) m_tabArray[j] ;
				int sLen = strlen (pTab->m_tabLabel) ;
				for (i = 0 ; i < sLen ; i ++)
				{
					if (pTab->m_tabLabel[i] == '&' && i != (sLen - 1))
					{
						memmove (&pTab->m_tabLabel[i],
							&pTab->m_tabLabel[i+1], sLen - i) ;
						break ;
					}
				}
			}
			break ;

		default :
			m_tabArray.m_position = TABSONTOP ;

			for (j = 0 ; j < m_nTabs ; j ++)
			{
				CTabInfo* pTab = (CTabInfo*) m_tabArray[j] ;
				int sLen = strlen (pTab->m_tabLabel) ;
				for (i = 0 ; i < sLen ; i ++)
				{
					if((char)(DWORD) AnsiUpper ((LPSTR) MAKELP(0,
							pTab->m_tabLabel[i])) == pTab->m_mnemonic
															&& i != (sLen - 1))
					{
						memmove (&pTab->m_tabLabel[i+1],
								&pTab->m_tabLabel[i], sLen - i) ;
						pTab->m_tabLabel[i] = '&' ;
						break ;
					}
				}
			}
			break ;
	}

	if(m_nTabs != 0)
	{
		createFonts() ;
		repositionViews() ;
		if (IsWindowVisible())
			InvalidateRect (NULL) ;
	}
}

void CTabDialog::switchTab (int viewIndex)
{
	CTabInfo* pTab = (CTabInfo*) m_tabArray[viewIndex] ;
	if (viewIndex < 0 || viewIndex >= m_nTabs)
		return ;

	if (m_curView != pTab->m_pView)
	{
		if (m_curView)
		{
			m_curView->SendMessage(msgSwitchFrom);
			m_curView->ShowWindow(SW_HIDE);
		}

		ASSERT(pTab->m_pView);
		m_curView = (CDialog*)pTab->m_pView;
		m_tabArray.m_curTab = viewIndex;
		m_curView->ShowWindow(SW_SHOW);
		m_curView->SendMessage(msgSwitchTo);
		m_curView->SetFocus();
		InvalidateRect(NULL);
	}
//	InvalidateRect(NULL);
//	UpdateWindow();

}

/////////////////////////////////////////////////////////////////////////////
// CTabDialog implementation

void CTabDialog::createFonts()
{
	destroyFonts();
		
	CClientDC dc(this);
	int nEscapement = 0;
	
	switch (m_tabArray.m_position)
	{
		case TABSONLEFT:
		case TABSONLEFTBOT:
		case TABSONRIGHT:
		case TABSONRIGHTBOT:
			if(m_tabArray.m_position == TABSONLEFT || m_tabArray.m_position == TABSONLEFTBOT)
				nEscapement = 900;
			else
				nEscapement = -900;
			break;

		default:
			nEscapement = 0;
			break;
	}

	LOGFONT lf;

	memset(&lf, '\0', sizeof(LOGFONT));

	lf.lfHeight = -MulDiv(8, dc.GetDeviceCaps(LOGPIXELSY), 72);
	lf.lfCharSet = ANSI_CHARSET;
	lf.lfQuality = DEFAULT_QUALITY;
	lf.lfClipPrecision = CLIP_LH_ANGLES | CLIP_STROKE_PRECIS;
	strcpy(lf.lfFaceName, "Arial");

	if (nEscapement == 0)
	{
		lf.lfPitchAndFamily = FF_SWISS ;   
	}
	else
	{
		lf.lfPitchAndFamily = FF_SWISS | 0x04;
		lf.lfEscapement = nEscapement;
		lf.lfOrientation = nEscapement;
	}

	// Create the normal font
	lf.lfWeight = FW_NORMAL;

	m_tabArray.m_normalFont = new CFont;
	m_tabArray.m_normalFont->CreateFontIndirect(&lf);

	// Create the bold font
	lf.lfWeight = FW_BOLD;

    m_tabArray.m_boldFont = new CFont;
	m_tabArray.m_boldFont->CreateFontIndirect(&lf);
}

CDialog* CTabDialog::createTabView(
	CDialog* pDialog,
	CWnd* parentWnd,
	BOOL border,
	BOOL show
	)
{
	CRect viewRect;

	switch (m_tabArray.m_position)
	{
		case TABSONLEFT:
		case TABSONLEFTBOT:
			viewRect.SetRect(m_tabArray.m_margin + 5 + m_tabArray.m_tabHeight,
							 m_tabArray.m_margin + 5,
							 m_width - (m_tabArray.m_margin + 5),
							 m_height - (m_tabArray.m_margin + 5));
			break;

		case TABSONRIGHT:
		case TABSONRIGHTBOT:
			viewRect.SetRect(m_tabArray.m_margin + 5,
							 m_tabArray.m_margin + 5,
							 m_width - (m_tabArray.m_margin + 5),
							 m_height- (m_tabArray.m_margin + 5));
			break;

		default:
			viewRect.SetRect(m_tabArray.m_margin + 5,
							 m_tabArray.m_margin + 5 + m_tabArray.m_tabHeight,
							 m_width - (m_tabArray.m_margin + 5),
							 m_height- (m_tabArray.m_margin + 5));
			break;
	}

	pDialog->SetParent(parentWnd);
	pDialog->SetWindowPos(&CWnd::wndBottom,
							viewRect.left,
							viewRect.top,
							viewRect.Width(),
							viewRect.Height(),
							SWP_NOREDRAW | SWP_NOACTIVATE);
    pDialog->ShowWindow (SW_HIDE);

	return pDialog;
}

void CTabDialog::destroyFonts()
{
	if(m_tabArray.m_boldFont)
		delete m_tabArray.m_boldFont;
	if(m_tabArray.m_normalFont)
		delete m_tabArray.m_normalFont;
}

void CTabDialog::repositionViews()
{
	for(int i=0;i<m_nTabs;i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
		switch (m_tabArray.m_position)
		{
			case TABSONLEFT:
			case TABSONLEFTBOT:
				pTab->m_pView->SetWindowPos(NULL,
											m_tabArray.m_margin + 5 + m_tabArray.m_tabHeight,m_tabArray.m_margin + 5,
											m_width - 2 * (m_tabArray.m_margin + 5) - m_tabArray.m_tabHeight,
											m_height - 2 * (m_tabArray.m_margin + 5),
											SWP_NOZORDER);
				break;

			case TABSONRIGHT:
			case TABSONRIGHTBOT:
				pTab->m_pView->SetWindowPos(NULL,
											m_tabArray.m_margin + 5,
											m_tabArray.m_margin + 5,
											m_width - 2 * (m_tabArray.m_margin + 5) - m_tabArray.m_tabHeight,
											m_height - 2 * (m_tabArray.m_margin + 5),
											SWP_NOZORDER);
				break;

			default:
				pTab->m_pView->SetWindowPos(NULL,
											m_tabArray.m_margin + 5,
											m_tabArray.m_margin + 5 + m_tabArray.m_tabHeight,
											m_width - 2 * (m_tabArray.m_margin + 5),
											m_height - 2 * (m_tabArray.m_margin + 5) - m_tabArray.m_tabHeight,
											SWP_NOZORDER);
				break;
		}
	}
}

BOOL CTabDialog::switchTopTab(CPoint point)
{
	CRect rect;
	int x = m_tabArray.m_margin, y = m_tabArray.m_margin;

	rect.top = y;
	rect.bottom = y + m_tabArray.m_tabHeight;

	for(int i = 0; i < m_nTabs; i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
		rect.left = x;
		rect.right = rect.left + pTab->m_tabWidth;
		if(rect.PtInRect(point) && pTab->m_active)
		{
			switchTab(i);
			return TRUE;
		}
		x += pTab->m_tabWidth;
	}
	return(FALSE);
}

BOOL CTabDialog::switchVerticalTab(CPoint point)
{
	CRect rect;
	int x = (m_tabArray.m_position == TABSONLEFT || m_tabArray.m_position == TABSONLEFTBOT) ? m_tabArray.m_margin : 
		(m_width - m_tabArray.m_margin - m_tabArray.m_tabHeight);
	int y = (m_tabArray.m_position == TABSONLEFT || m_tabArray.m_position == TABSONRIGHT) ? m_tabArray.m_margin:(m_height - m_tabArray.m_margin);

	rect.left = x;
	rect.right = rect.left + m_tabArray.m_tabHeight;
	
	for(int i = 0; i < m_nTabs; i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
		if(m_tabArray.m_position == TABSONLEFT || m_tabArray.m_position == TABSONRIGHT)
		{
			rect.top        = y;
			rect.bottom = y + pTab->m_tabWidth;
		}
		else
		{
			rect.bottom = y;
			rect.top = rect.bottom - pTab->m_tabWidth;
		}
		if(rect.PtInRect(point) && pTab->m_active)
		{
			switchTab(i);
			return TRUE;
		}
		if(m_tabArray.m_position == TABSONLEFT || m_tabArray.m_position == TABSONRIGHT)
			y += pTab->m_tabWidth;
		else
			y -= pTab->m_tabWidth;
	}
	return(FALSE);
}

/////////////////////////////////////////////////////////////////////////////
// CTabDialog message handlers

LRESULT CTabDialog::OnCommandHelp(WPARAM wParam, LPARAM lParam)
{
	if (m_curView == NULL)
		return FALSE;

	return m_curView->SendMessage(WM_COMMANDHELP, wParam, lParam);
}

void CTabDialog::OnCancel()
{
	bActive = FALSE;
    dlgReturnCode = IDCANCEL;
}

BOOL CTabDialog::OnEraseBkgnd(CDC* pDC)
{
	CBrush backBrush(GetSysColor(COLOR_BTNFACE));
	CBrush* pOldBrush = pDC->SelectObject(&backBrush);
	
	CRect rect;
	pDC->GetClipBox(&rect); //erase the area needed
	pDC->PatBlt(rect.left,rect.top,rect.Width(),rect.Height(),PATCOPY);
	pDC->SelectObject(pOldBrush);
	return TRUE;
}

#if 0
//
// To take care of CTRL + TAB keystroke.
// Srini - 21 Feb 1995
//

void CTabDialog::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	if (nChar == VK_TAB) {
		if (::GetAsyncKeyState(VK_CONTROL) & 0x8000) {
			
			for (int nTabIdx = 0; nTabIdx < m_nTabs; nTabIdx++) {

				CTabInfo* pTab = (CTabInfo*) m_tabArray[nTabIdx];
				if (pTab->m_active) {

					int             nNextTab = nTabIdx + 1;

					// If last tab, then shift to 1st tab.
					if (nNextTab >= m_nTabs)
						nNextTab = 0;

					switchTab(nNextTab);

					break;
				}
		}

			return;
		}
	}

	CDialog::OnKeyDown(nChar, nRepCnt, nFlags);
}
#endif

LRESULT CTabDialog::OnHelpHitTest(WPARAM wParam, LPARAM lParam)
{
	if (m_curView == NULL)
		return FALSE;

	return m_curView->SendMessage(WM_HELPHITTEST, wParam, lParam);
}

BOOL CTabDialog::OnInitDialog()
{
	CDialog::OnInitDialog();
#ifdef  __CTL3D__
	Ctl3dSubclassDlg(m_hWnd, CTL3D_ALL);
#endif
	createFonts();

	if (m_nTabs)
	{
		switchTab(m_tabArray.m_curTab);
	}
	
	return TRUE;  // return TRUE  unless you set the focus to a control
}

void CTabDialog::OnLButtonDown(UINT nFlags, CPoint point)
{
	CDialog::OnLButtonDown(nFlags, point);

	switch (m_tabArray.m_position)
	{
		case TABSONLEFT:
		case TABSONLEFTBOT:
		case TABSONRIGHT:
		case TABSONRIGHTBOT:
			if ( switchVerticalTab(point) )
				return;

		default:
			if ( switchTopTab(point) )
				return;
	}
}

LRESULT CTabDialog::OnMenuChar (UINT character, UINT menu_flags, CMenu* menu_handle)
{
	char mnemonic = (char)(DWORD)AnsiUpper((LPSTR)MAKELP(0,(char)character));

	for (int i = 0; i < m_nTabs; i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
		if( (mnemonic == pTab->m_mnemonic)  &&  pTab->m_active)
		{
			switchTab(i);
			return MAKELONG(0, 1);
		}
    }
    return (LRESULT) 0;
}

int CTabDialog::OnMouseActivate(CWnd* pDesktopWnd, UINT nHitTest, UINT message)
{
	return MA_ACTIVATE;
}

void CTabDialog::OnOK()
{
	for(int i=0; i < m_nTabs; i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];
		pTab->m_pView->UpdateData(TRUE);
	}

	bActive = FALSE;
    dlgReturnCode = IDOK;
}

void CTabDialog::OnPaint()
{
	CPaintDC dc(this); // device context for painting
	
	if(m_lookAndFeel == LAF_CHICAGO && m_tabArray.m_position == TABSONLEFT)
	{
		TRACE0("Warning: left tabs not supported for LAF = CHICAGO");
		m_lookAndFeel = LAF_MSWORD;
	}

	if(m_lookAndFeel == LAF_CHICAGO)
	{       
		m_tabArray.drawChicagoTabs(this, &dc);
	}
	else // MS WORD STYLE LOOK--compliments of David Hollifield
	{
		m_tabArray.drawMSWordTabs(this, &dc);
	}

	// Do not call CDialog::OnPaint() for painting messages
}

void CTabDialog::OnSetFocus(CWnd* pOldWnd)
{
	CDialog::OnSetFocus(pOldWnd);
	
	if (m_curView)
	{
		// Set the keyboard focus to this view
		m_curView->SetFocus();
		return;
	}
}

void CTabDialog::OnSize(UINT nType, int cx, int cy)
{
	CDialog::OnSize(nType, cx, cy);
	m_width = cx;
	m_height = cy;
	
	for(int i=0;i<m_nTabs;i++)
	{
		CTabInfo* pTab = (CTabInfo*)m_tabArray[i];

		switch (m_tabArray.m_position)
		{
			case TABSONLEFT:
			case TABSONLEFTBOT:
			case TABSONRIGHT:
			case TABSONRIGHTBOT:
				pTab->m_pView->SetWindowPos(NULL,
											0,
											0,
											cx - 2 * (m_tabArray.m_margin + 5) - m_tabArray.m_tabHeight,
											cy - 2 * (m_tabArray.m_margin + 5),
											SWP_NOMOVE|SWP_NOZORDER);
				break;

			default:
				pTab->m_pView->SetWindowPos(NULL,
											0,
											0,
											cx - 2 * (m_tabArray.m_margin + 5),
											cy - 2 * (m_tabArray.m_margin + 5) - m_tabArray.m_tabHeight,
											SWP_NOMOVE|SWP_NOZORDER);
				break;
		}
	}
}

#ifdef  __CTL3D__
HBRUSH CTabDialog::OnCtlColor(CDC* pDC, CWnd* pWnd, UINT nCtlColor)
{
	MSG* pMsg = (MSG*)GetCurrentMessage();

	CDialog::OnCtlColor(pDC, pWnd, nCtlColor);
	return (HBRUSH)Ctl3dCtlColorEx(pMsg->message, pMsg->wParam, pMsg->lParam);
}

BOOL CTabDialog::OnNcActivate(BOOL)
{
	MSG* pMsg = (MSG*)GetCurrentMessage();
	return (BOOL)Ctl3dDlgFramePaint(pMsg->hwnd, pMsg->message, pMsg->wParam, pMsg->lParam);
}

void CTabDialog::OnNcPaint()
{
	MSG* pMsg = (MSG*)GetCurrentMessage();
	Ctl3dDlgFramePaint(pMsg->hwnd, pMsg->message, pMsg->wParam, pMsg->lParam);
}
#endif

/////////////////////////////////////////////////////////////////////////////
// CTabDialog diagnostics

#ifdef _DEBUG
void CTabDialog::AssertValid() const
{
	CDialog::AssertValid();
}

void CTabDialog::Dump(CDumpContext& dc) const
{
	CDialog::Dump(dc);
}
#endif //_DEBUG
