/****************************************************************************
 * Filename   : status.c
 * Purpose    : This file contains the functions to handle the status bar
 *              window.
 ****************************************************************************/

#include 	<windows.h>

#include	"mew.h"
#include	"session.h"
#include	"main.h"
#include	"fnkey.h"
#include "status.h"
#include	"term.h"
#include	"emlnfuns.h"
#include	"wstdio.h"
#include	"config.h"
#include	"file.h"
#include	"mewlog.h"
#include	"scroll.h"
#include	"myprint.h"
#include	"looks.h"
#include	"dlgbox.h"

extern BOOL IsPortIP ;
char	*MainStatusStrs[MAX_NUM_STATES+1] = {
	"Console display mode",
	"Scroll mode",
	"WAN Device Configuration mode",
	"LANTalker Configuration mode",
	""
};
char Temp[40] ;

char	*StatusFldStrs[] = {
	"Log closed", "Log open", "Log paused",
	"Print off", "Print on",
};

StatusField	StatFields[MAX_STAT_FIELDS] = {
	{"Log closed"},{"Print off"}, 
};

LineParamsType	TmpConParams;	/* temporary line prameter values */


int	CurHlpIdx = 0;				/* current one line help index */
BOOL	HelpInStatus = FALSE;	/* flag set if one line help is being shown */
										/* in the bottom status bar. */

#define	TOP_BAR_HELP_START	60

/****************************************************************************
 * Function   : StatusWndProc (hWnd, message, wParam, lParam)
 * Purpose    : Window function for the function keys window. (function keys??)
 ****************************************************************************/
long FAR PASCAL
StatusWndProc (HWND hWnd, unsigned message, WORD wParam, LONG lParam)
{
	HDC 	hDc;
	int  	index;
	PAINTSTRUCT ps;

   switch (message){
		case WM_MOUSEMOVE:
			if (CheckOverButton(hWnd, lParam))
				SetCursor (hFinger);
			else
				SetCursor (hStandard);
			break;

		case WM_LBUTTONDOWN :
			if (HelpInStatus || ScrollLock )
				break;
			if ((index = CheckClickOnField (hWnd, lParam)) != -1) {
				hDc = GetDC (hWnd);
				switch (index) {
					case LOG_FLD_IDX:
						LogOptionsDlg(hWnd);
						break;
					case PRINT_FLD_IDX:
						TogglePrinter ();
						break;
				}
				DrawPushButtonUp(hDc, (LPRECT) &StatFields[index].Coords,
							(LPSTR)&StatFields[index].Title[0]);
				ReleaseDC (hWnd, hDc);
			}
			break;

		case WM_PAINT:
			hDc = BeginPaint (hWnd, (LPPAINTSTRUCT)&ps);
			DisplayStatusBar (hDc);
			ValidateRect (hWnd, NULL);
			EndPaint(hWnd, (LPPAINTSTRUCT)&ps);
	    	break;

		case WM_CLOSE:
			DestroyWindow(hWnd);
         	break;

		default:
	    	return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return(NULL);
}

/****************************************************************************
 * Function   : CreateStatusWnd (HWND)
 * Purpose    : creates the status bar widow if it needs to be shown.
 ****************************************************************************/
int
CreateStatusWnd (HWND hParentWnd)
{
	if (! ShowStatusWnd)
		return FALSE;

    hStatusWnd = CreateWindow("ROUSTATUS",
							(LPSTR)NULL,
							WS_CHILD | WS_VISIBLE | WS_BORDER,
							StatusWndRect.left,
							StatusWndRect.top,
							(StatusWndRect.right - StatusWndRect.left),
							(StatusWndRect.bottom - StatusWndRect.top),
							hParentWnd,
							IDC_STATUS_WND,	/* child control ID */
							hInst,
							(LPSTR)NULL);

	if (! hStatusWnd) {
		ShowStatusWnd = FALSE;
		return FALSE;
	}
	ComputeStatFieldCoords ();
	return TRUE;
}

/****************************************************************************
 * Function   : DisplaySessionStatus (void)
 * Purpose    : displays the current status values in status bar.
 ****************************************************************************/
void
DisplaySessionStatus (void)
{
	HDC	hDc;

	if (! ShowStatusWnd)
		return;
	if (! hStatusWnd)
		return;
	hDc = GetDC (hStatusWnd);
	DisplayStatusBar (hDc);
	ReleaseDC (hStatusWnd, hDc);
}

/****************************************************************************
 * Function   : DisplayStatusBar (HDC)
 * Purpose    : displays the current status values in status bar.
 ****************************************************************************/
int
DisplayStatusBar (HDC hDc)
{
	int	i;

	if (! ShowStatusWnd)
		return FALSE;

	if (HelpInStatus) {
		ShowCurOneLineHelp (hDc);
		return TRUE;
	}

	if (LogState == LOG_OPEN)
		strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[1]);	/* log open */
	else if (LogState == LOG_CLOSED)
		strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[0]);	/* log closed */
	else
		strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[2]);	/* log paused */

	if (PrintLogging)
		strcpy (StatFields[PRINT_FLD_IDX].Title, StatusFldStrs[4]);	/* print on */
	else
		strcpy (StatFields[PRINT_FLD_IDX].Title, StatusFldStrs[3]);	/* print off */

	for (i=0; i < MAX_STAT_FIELDS; i++) {
	  	DrawPushButtonUp(hDc, (LPRECT) &StatFields[i].Coords,
			(LPSTR) &StatFields[i].Title[0]);
	}
	return TRUE;
}

void	ReDisplayStatusBar(void)
{
	if (! ShowStatusWnd || HelpInStatus)
		return;
	InvalidateRect (hStatusWnd, NULL, TRUE);
	UpdateWindow(hStatusWnd);
}
/****************************************************************************
 * Function   : ComputeStatFieldCoords (void)
 * Purpose    : computes the status field dimensions.
 ****************************************************************************/
int
ComputeStatFieldCoords (void)
{
	int		i;
	HDC		hDc;
	RECT	rct;
	int		Space = 0;
	int		TextLen = 0;

	hDc = GetDC (hStatusWnd);

	for (i=0; i < MAX_STAT_FIELDS; i++) {
		TextLen += (int)GetTextExtent(hDc, (LPSTR)StatFields[i].Title, strlen(StatFields[i].Title));
	}
	if ((StatusWndRect.right - StatusWndRect.left - (KEY_MARGIN << 1))
						> TextLen)
		Space = ((StatusWndRect.right - StatusWndRect.left -
				(KEY_MARGIN << 1)) - TextLen) / MAX_STAT_FIELDS;

//	if (Space < 4)
		Space = KEY_MARGIN * 4;

	rct.left = KEY_MARGIN; rct.top = 0;
	rct.bottom = StatusWndRect.bottom - StatusWndRect.top;
	for (i=0; i < MAX_STAT_FIELDS; i++) {
		StatFields[i].Coords.left = rct.left;
		StatFields[i].Coords.top = rct.top;
		StatFields[i].Coords.right = rct.left + Space +
			+ (int)GetTextExtent(hDc, (LPSTR)StatFields[i].Title, strlen(StatFields[i].Title));

		StatFields[i].Coords.bottom = rct.bottom;
		rct.left = StatFields[i].Coords.right;
	}
//	StatFields[i-1].Coords.right = StatusWndRect.right - KEY_MARGIN;

	ReleaseDC (hStatusWnd, hDc);
	return TRUE;
}

/****************************************************************************
 * Function   : CheckClickOnField (HWND, LONG)
 * Purpose    : checks if the mouse has been clicked in any of the fields.
 ****************************************************************************/
int
CheckClickOnField (HWND hWnd, LONG XYPos)
{
	int		index;
	int		XPos, YPos;
	HDC		hDc;

	XPos = (int)XYPos;
	YPos = (int)(XYPos >> 16);
	for (index=0; index < MAX_STAT_FIELDS; index++) {
		if (((XPos < StatFields[index].Coords.right - KEY_HORZ_CLEARANCE)
			&& (XPos > StatFields[index].Coords.left + KEY_HORZ_CLEARANCE))
			&& ((YPos < StatFields[index].Coords.bottom - KEY_VERT_CLEARANCE)
			&& (YPos > StatFields[index].Coords.top + KEY_VERT_CLEARANCE))) {
			/* also depress the key */
				
				hDc = GetDC (hWnd);
				DrawPushButtonDn (hDc, (LPRECT) &StatFields[index].Coords,
						(LPSTR) &StatFields[index].Title[0]);
				ReleaseDC (hWnd, hDc);
				return (index);
		}
	}
	return (-1);
}
/****************************************************************************
 * Function   : CheckOverButton (HWND, LONG)
 * Purpose    : checks if the mouse has been clicked in any of the fields.
 ****************************************************************************/
int
CheckOverButton (HWND hWnd, LONG XYPos)
{
	int		index;
	int		XPos, YPos;

	XPos = (int)XYPos;
	YPos = (int)(XYPos >> 16);
	for (index=0; index < MAX_STAT_FIELDS; index++) {
		if (((XPos < StatFields[index].Coords.right - KEY_HORZ_CLEARANCE)
			&& (XPos > StatFields[index].Coords.left + KEY_HORZ_CLEARANCE))
			&& ((YPos < StatFields[index].Coords.bottom - KEY_VERT_CLEARANCE)
			&& (YPos > StatFields[index].Coords.top + KEY_VERT_CLEARANCE))) {
				return TRUE;
		}
	}
	return FALSE;
}

/****************************************************************************
 * FUNCTION: UpdateStatusBar (int)
 * PURPOSE	: updates the specified field in the status bar.
 ****************************************************************************/
void
UpdateStatusBar (int FldIdx)
{
	HDC hDc;

	if (! ShowStatusWnd)
		return;

	if (HelpInStatus)
		return;

	switch (FldIdx) {
		case LOG_FLD_IDX :
			if (LogState == LOG_OPEN)
				strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[1]);	/* log open */
			else if (LogState == LOG_CLOSED)
				strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[0]);	/* log closed */
			else
				strcpy (StatFields[LOG_FLD_IDX].Title, StatusFldStrs[2]);	/* log paused */
				
			break;
		case PRINT_FLD_IDX :
			if (PrintLogging)
				strcpy (StatFields[PRINT_FLD_IDX].Title, StatusFldStrs[4]);	/* print on */
			else
				strcpy (StatFields[PRINT_FLD_IDX].Title, StatusFldStrs[3]);	/* print off */
			break;
		default:
			return;
	}

	hDc = GetDC (hStatusWnd);
	DrawPushButtonUp(hDc, (LPRECT) &StatFields[FldIdx].Coords,
		(LPSTR) &StatFields[FldIdx].Title[0]);
	ReleaseDC (hStatusWnd, hDc);
	return;
}

char *
GetStatusStr (void)
{

	if (CurState & MSK_SCROLL_MODE)
		return (MainStatusStrs[ST_SCROLL_MODE]);
	else if (CommandLineArgument == 4)
	{
		if (!IsPortIP)
			return (MainStatusStrs[ST_WANCHAT_MODE]);
		else
		{
			strcpy (Temp, MainStatusStrs[ST_WANCHAT_MODE]) ;
			strcat (Temp, " (IP Port)") ;
			return (&Temp[0]) ;
		}
	}
	else if (CommandLineArgument == 5)
		return (MainStatusStrs[ST_MUXCONF_MODE]);
	else
		return (MainStatusStrs[ST_NORMAL_MODE]);

}

extern	FileListType	*TrmList;	/* The terminal names list */
/****************************************************************************
 * FUNCTION: ShowOneLineHelp (int)
 * PURPOSE	: displays one line help in the bottom status bar.
 ****************************************************************************/
void
ShowOneLineHelp (int index)
{
	HDC		hDc;
	RECT	rct;
	BYTE	HlpStr[128];

	if (!hStatusWnd || ! ShowStatusWnd)
		return;

	HelpInStatus = TRUE;
	CurHlpIdx = index;
	hDc = GetDC (hStatusWnd);
	InvalidateRect (hStatusWnd, NULL, TRUE);
	LoadString(hInst, index, (LPSTR) HlpStr, 128);
	rct.left = 10; rct.top = 0;
	rct.right = StatusWndRect.right - StatusWndRect.left - 10;
	rct.bottom = StatusWndRect.bottom - StatusWndRect.top;
	DisplayQHelp(hDc, (LPRECT)&rct, (LPSTR) HlpStr);
	ReleaseDC (hStatusWnd, hDc);
}

/****************************************************************************
 * FUNCTION: ShowCurOneLineHelp (HDC)
 * PURPOSE	: displays the current one line help, if paint is requested.
 ****************************************************************************/
void
ShowCurOneLineHelp (HDC hDc)
{
	RECT	rct;
	BYTE	HlpStr[128];

	rct.left = 10; rct.top = 0;
	rct.right = StatusWndRect.right - StatusWndRect.left - 10;
	rct.bottom = StatusWndRect.bottom - StatusWndRect.top;
	LoadString(hInst, CurHlpIdx, HlpStr, 128);
	DisplayQHelp(hDc, (LPRECT) &rct, (LPSTR) HlpStr);
}

/****************************************************************************
 * FUNCTION: CloseOneLineHelp (void)
 * PURPOSE	: closes the one line help and displays the usual status bar.
 ****************************************************************************/
void
CloseOneLineHelp (void)
{
	HelpInStatus = FALSE;
	CurHlpIdx = 0;
	if (hStatusWnd) {
		InvalidateRect (hStatusWnd, NULL, TRUE);
		UpdateWindow(hStatusWnd);
	}
}

/****************************************************************************
 * FUNCTION	: GetOneLineHelpIdx (WORD)
 * PURPOSE	: based on the menu item ID, gets the index of the help string.
 ****************************************************************************/
int
GetOneLineHelpIdx (WORD Param)
{
	int	CtxtId;

	CtxtId = 
		(Param == IDM_FILE_PAGESET) ?  	1 :
		(Param == IDM_FILE_PRINTERSET)?	2 :
		(Param == IDM_FILE_PRINT) ?  		3 :
		(Param == IDM_FILE_EXIT) ?  		4 :
													0;
	if (CtxtId)
		return (CtxtId);
	CtxtId =
		(Param == IDM_EDIT_BUFFTOCLIP) ? 5 :
		(Param == IDM_EDIT_BUFFTOFILE) ? 6 :
		(Param == IDM_EDIT_BUFFTOPRNT) ? 7 :
		(Param == IDM_EDIT_SCROLL)     ? 8 :
													0;
	if (CtxtId)
		return (CtxtId);
    CtxtId =
		(Param == IDM_SET_FONTS) ?			9 :
		(Param == IDM_SET_COLOR) ?      10 :
													0;
	if (CtxtId)
		return (CtxtId);
    CtxtId =
		(Param == IDM_PER_RESET) ?  		11 :
		(Param == IDM_PER_LOG) ?			12 :
													0;
	if (CtxtId)
		return (CtxtId);
    CtxtId =
		(Param == IDM_WIN_SETUP) ?  		13 :
		(Param == IDM_WIN_TOP) ?  			14 :
		(Param == IDM_WIN_BOT) ?  			15 :
		(Param == IDM_WIN_TBAR) ?  		16 :
													0;
	if (CtxtId)
		return (CtxtId);
    CtxtId =
		(Param == IDM_WANTALK_1) ?  		17 :
		(Param == IDM_WANTALK_2) ?  		18 :
		(Param == IDM_WANTALK_3) ?  		19 :
		(Param == IDM_SET_DTR)   ?  		20 :
		(Param == IDM_RESET_DTR) ?  		21 :
														0;

	return (CtxtId);
}

/****************************************************************************
 * FUNCTION	: GetTopBarOneLineHelpIdx (WORD)
 * PURPOSE	: based on the menu item handle, gets the index of the help string.
 ****************************************************************************/
int
GetTopBarOneLineHelpIdx (WORD MenuHandle)
{
	int		i;
	HMENU	hMain, hTmp;

	hMain = GetMenu(hMainWnd);
	for (i=0; i < 6; i++) {		//File, edit, set, perform, win and help menu
		hTmp = GetSubMenu (hMain, i);
		if (hTmp == MenuHandle)
			return (i+TOP_BAR_HELP_START);
	}
	return (0);
}

BYTE	GetKbdLeds(void);
void	SetKbdLeds(BYTE NewStatus);

/****************************************************************************
 * FUNCTION	: HandleKbdLeds (BYTE, BYTE)
 * PURPOSE	: Changes the keyboard LEDs status in the ROM BIOS area.
 ****************************************************************************/
void
HandleKbdLeds (BYTE KbdLed, BYTE Flag)
{
	BYTE	CurStatus;

	CurStatus = GetKbdLeds();
	switch (Flag) {
		case LIGHT_ON:
			CurStatus |= (1 << KbdLed);
			break;
		case LIGHT_OFF:
			CurStatus &= ~(1 << KbdLed);
			break;
		case LIGHT_TOGGLE:
			CurStatus ^= (1 << KbdLed);
			break;
	}
	SetKbdLeds(CurStatus);
}
