/****************************************************************************
 * Filename   : time.c
 * Purpose    : This file contains the functions to handle the time bar
 *              window.
 ****************************************************************************/

#include <windows.h>

#include	"mew.h"
#include	"main.h"
#include	"session.h"
#include	"file.h"
#include	"status.h"
#include "time.h"
#include "looks.h"

extern	char	*NoTitleStr;

#define	MEW_TIMER_ID	0x53

TIME	ConTime;					/* var maintaining the connection time */
RECT	TimeFields[MAX_TIME_FLDS];	/* fields in the time bar window */

char 	*MonthNames[NUM_MONTHS] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

/****************************************************************************
 * Function   : TimeWndProc (hWnd, message, wParam, lParam)
 * Purpose    : Window function for the time bar window.
 ****************************************************************************/
long FAR PASCAL
TimeWndProc (HWND hWnd, unsigned message, WORD wParam, LONG lParam)
{
    HDC 	hDc;
	BYTE	buf[40];
    PAINTSTRUCT ps;

    switch (message){
		case WM_MOUSEMOVE:
			SetCursor (hCurCursor);
			break;

		case WM_TIMER:
			if (! ShowTimeWnd)
				break;
			hDc = GetDC (hWnd);
			GetAsciiDate(buf);
			DrawStatusWinText(hDc, (LPRECT) &TimeFields[DATE_FLD_IDX],
						(LPSTR)buf, TRUE);
			GetAsciiTime(buf);
			DrawStatusWinText(hDc, (LPRECT) &TimeFields[TIME_FLD_IDX],
						(LPSTR)buf, TRUE);

			ReleaseDC (hWnd, hDc);
			break;

		case WM_PAINT:
	   	hDc = BeginPaint (hWnd, (LPPAINTSTRUCT)&ps);
			DisplayTimeBar (hDc);
			ValidateRect (hWnd, NULL);
	   	EndPaint(hWnd, (LPPAINTSTRUCT)&ps);
	    	break;

		case WM_CLOSE:
			DestroyWindow(hWnd);
         	break;

		default:
	    	return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return(NULL);
}

/****************************************************************************
 * Function   : CreateTimeWnd (HWND)
 * Purpose    : creates the time bar widow if it needs to be shown.
 ****************************************************************************/
int
CreateTimeWnd (HWND hParentWnd)
{
	if (! ShowTimeWnd)
		return FALSE;

    hTimeWnd = CreateWindow("ROUTIME",
							(LPSTR)NULL,
							WS_CHILD | WS_VISIBLE | WS_BORDER,
							TimeWndRect.left,
							TimeWndRect.top,
							(TimeWndRect.right - TimeWndRect.left),
							(TimeWndRect.bottom - TimeWndRect.top),
							hParentWnd,
							IDC_TIME_WND,	/* child control ID */
							hInst,
							(LPSTR)NULL);

	if (! hTimeWnd) {
		ShowTimeWnd = FALSE;
		return FALSE;
	}
	ComputeTimeFieldCoords ();
	ConTime.hour = 0; ConTime.minute = 0; ConTime.second = 0;
	SetTimer (hTimeWnd, MEW_TIMER_ID, (WORD)(((UINT)MINUTE)*1000), NULL);
	return TRUE;
}

/****************************************************************************
 * Function   : ComputeTimeFieldCoords (void)
 * Purpose    : computes the time fields dimensions.
 ****************************************************************************/
int
ComputeTimeFieldCoords (void)
{
	TimeFields[DATE_FLD_IDX].left = KEY_MARGIN;
	TimeFields[DATE_FLD_IDX].top = 0;
	TimeFields[DATE_FLD_IDX].right = 150;
	TimeFields[DATE_FLD_IDX].bottom = TimeWndRect.bottom - TimeWndRect.top;

	TimeFields[TIME_FLD_IDX].left = TimeFields[DATE_FLD_IDX].right;
	TimeFields[TIME_FLD_IDX].top = 0;
	TimeFields[TIME_FLD_IDX].right = TimeFields[DATE_FLD_IDX].right + 120;
	TimeFields[TIME_FLD_IDX].bottom = TimeWndRect.bottom - TimeWndRect.top;

	TimeFields[STATE_FLD_IDX].left = TimeFields[TIME_FLD_IDX].right;
	TimeFields[STATE_FLD_IDX].top = 0;
	TimeFields[STATE_FLD_IDX].right = TimeWndRect.right - 90;
	TimeFields[STATE_FLD_IDX].bottom = TimeWndRect.bottom - TimeWndRect.top;

	TimeFields[CONTIME_FLD_IDX].left = TimeFields[STATE_FLD_IDX].right;
	TimeFields[CONTIME_FLD_IDX].top = 0;
	TimeFields[CONTIME_FLD_IDX].right = TimeWndRect.right - KEY_MARGIN;
	TimeFields[CONTIME_FLD_IDX].bottom = TimeWndRect.bottom - TimeWndRect.top;

	return TRUE;
}

extern	BYTE ScriptLearnFile[];

/****************************************************************************
 * Function   : DisplayTimeBar (hDc)
 * Purpose    : displays the time bar window.
 ****************************************************************************/
int
DisplayTimeBar (HDC hDc)
{
//	char	*ptr;
	BYTE	buf[40];
	LPSTR	pStatusStr;

	if (! ShowTimeWnd)
		return FALSE;

	GetAsciiDate (buf);
	DrawStatusWin(hDc, (LPRECT) &TimeFields[DATE_FLD_IDX],(LPSTR)buf);
	GetAsciiTime (buf);
	DrawStatusWin(hDc, (LPRECT) &TimeFields[TIME_FLD_IDX],(LPSTR)buf);

	pStatusStr = GetStatusStr();
	lstrcpy(TmpStr, "");
	lstrcpy (TmpStr, pStatusStr);

	DrawStatusWin(hDc, (LPRECT) &TimeFields[STATE_FLD_IDX],(LPSTR)TmpStr);
	
	FormatAsciiTime (&ConTime, buf, TIME_NORMAL);
	DrawStatusWin(hDc, (LPRECT) &TimeFields[CONTIME_FLD_IDX],(LPSTR)buf);
	return TRUE;
}

/****************************************************************************
 * Function   : UpdateStatusInBar (void)
 * Purpose    : updates the status string in the time bar's display.
 ****************************************************************************/
void
UpdateStatusInBar (void)
{
	HDC		hDc;
	LPSTR	pStatusStr;

	if (! ShowTimeWnd)
		return;

	hDc = GetDC (hTimeWnd);
	pStatusStr = GetStatusStr();
	lstrcpy(TmpStr, "");
	lstrcpy (TmpStr, (char *)pStatusStr);
	DrawStatusWinText(hDc, (LPRECT) &TimeFields[STATE_FLD_IDX],
				(LPSTR)TmpStr, TRUE);
	ReleaseDC (hTimeWnd, hDc);
	return;
}

/****************************************************************************
 * Function   : GetAsciiTimeDate (LPSTR)
 * Purpose    : Builds the time and date string to display in the
 *				main window footer box.
 ****************************************************************************/
int
GetAsciiTimeDate (LPSTR pBuf)
{
	TIME	cTime;
	DATE	cDate;

	GetTime (&cTime);
	GetDate (&cDate);
	FormatAsciiDate (&cDate, pBuf, DATE_EXPAND);
	lstrcat (pBuf, " ");
	FormatAsciiTime (&cTime, &pBuf[strlen(pBuf)], TIME_AMPM);
	return TRUE;
}

/****************************************************************************
 * Function   : GetAsciiDate (LPSTR)
 * Purpose    : gets the current date in the normal mm/dd/yy format into the
 *				buffer passed.
 ****************************************************************************/
void
GetAsciiDate (LPSTR buf)
{
	static DATE	cDate;

	GetDate (&cDate);
	FormatAsciiDate (&cDate, buf, DATE_EXPAND);
	return;
}

/****************************************************************************
 * Function   : FormatAsciiDate (DATE *, LPSTR, WORD)
 * Purpose    : formats the date into either NORMAL or EXPAND format.
 ****************************************************************************/
void
FormatAsciiDate (DATE *pDate, LPSTR buf, WORD format)
{
	BYTE	tmp[8];

	if (format == DATE_EXPAND) {	/* format "Jan 3, 1991" */
		/* build the date string */
#if 0
		lstrcpy (buf, "  ");
		lstrcat (buf, MonthNames[pDate->month - 1]);
		lstrcat (buf, ". ");
#endif
		lstrcpy (buf, MonthNames[pDate->month - 1]);
//		itoa (pDate->day, tmp, 10);
		wsprintf(tmp, " %d",pDate->day);
		lstrcat (buf, tmp);
		lstrcat (buf, ", ");
#if 0
//		itoa (pDate->year, tmp, 10);
		wsprintf(tmp, "%d", pDate->day);
#endif
//		itoa ((pDate->year-1900), tmp, 10);
		wsprintf(tmp, "%4d ", pDate->year);
		lstrcat (buf, tmp);
	}
	else if (format == DATE_NORMAL) {/* normal date format mm/dd/yy - 01/03/1991 */
//		itoa (pDate->month, tmp, 10);
		wsprintf(tmp, "%d", pDate->month);
		if (! tmp[1]) {
			tmp[1] = tmp[0]; tmp[0] = '0'; tmp[2] = 0;
		}
		lstrcpy (buf, tmp);
		lstrcat (buf, "/");
//		itoa (pDate->day, tmp, 10);
		wsprintf(tmp, "%d", pDate->day);
		if (! tmp[1]) {
			tmp[1] = tmp[0]; tmp[0] = '0'; tmp[2] = 0;
		}
		lstrcat (buf, tmp);
		lstrcat (buf, "/");
//		itoa (pDate->year, tmp, 10);
		wsprintf(tmp, "%4d", pDate->year);
		lstrcat (buf, tmp);
	}
	return;
}

/****************************************************************************
 * Function   : GetAsciiTime (LPSTR)
 * Purpose    : gets the current system time in the normal form hh:mm:ss.
 ****************************************************************************/
void
GetAsciiTime (LPSTR buf)
{
	static TIME	cTime;

	GetTime (&cTime);
	FormatAsciiTime (&cTime, buf, TIME_AMPM);
	return;
}

#define	AM 			1
#define	PM 			2

/****************************************************************************
 * Function   : FormatAsciiTime (TIME *, LPSTR, WORD)
 * Purpose    : formats the time into the buffer passed according to the
 *				format specified - AMPM format or NORMAL
 ****************************************************************************/
void
FormatAsciiTime (TIME *pTime, LPSTR buf, WORD format)
{
	int		hr;
	BYTE	tmp[8];
	BYTE	AmOrPm;

	buf[0] = 0;
	if (format == TIME_AMPM) {	/* format is hh:mm (am/pm) */
		/* build the time string */
		if ((hr = pTime->hour) > 12) {
			AmOrPm = PM;		// it is post meridian part of day.
			hr -= 12;			// subtract 12 from Hrs if > than 12
		} else if (hr == 12) {
			AmOrPm = PM;		// don't subtract 12 in this case from Hrs.
		} else {
			AmOrPm = AM;
		}
//		itoa (hr, tmp, 10);
		wsprintf(tmp,"%2d", hr);
		lstrcpy (buf, tmp);
		lstrcat (buf, ":");
//		itoa (pTime->minute, tmp, 10);
		wsprintf(tmp, "%d",pTime->minute);
		if (! tmp[1]) {
			tmp[2] = 0; tmp[1] = tmp[0]; tmp[0] = '0';
		}
		lstrcat (buf, tmp);
		if (AmOrPm == AM)
			lstrcat (buf, "am");
		else
			lstrcat (buf, "pm");
	}
	else if (format == TIME_NORMAL) {	/* normal format - hh:mm:ss */
//		itoa (pTime->hour, tmp, 10);
		wsprintf(tmp, "%d", pTime->hour);
		if (! tmp[1]) {
			tmp[1] = tmp[0]; tmp[0] = '0'; tmp[2] = 0;
		}
		lstrcpy (buf, tmp);
		lstrcat (buf, ":");
//		itoa (pTime->minute, tmp, 10);
		wsprintf(tmp, "%d", pTime->minute);
		if (! tmp[1]) {
			tmp[1] = tmp[0]; tmp[0] = '0'; tmp[2] = 0;
		}
		lstrcat (buf, tmp);
		lstrcat (buf, ":");
//		itoa (pTime->second, tmp, 10);
		wsprintf(tmp, "%d", pTime->second);
		if (! tmp[1]) {
			tmp[1] = tmp[0]; tmp[0] = '0'; tmp[2] = 0;
		}
		lstrcat (buf, tmp);
	}
}
